/**
 * vue-axios 封装(报错/权限/跳转/拦截/提示)
 * 基础需求:
 *    统一捕获接口报错 => axiso 内置拦截器
 *    弹窗提示 => 自定义(如: `Element-ui Message` 组件)
 *    报错重定向 => 路由钩子(根据实际需要配置)
 *    基础权限 => 服务端过期时间戳和token, 以及借助路由钩子(根据实际需要决定是否配置)
 *    表单序列化 => npm 模块 qs
 */

import axios from 'axios'
import qs from 'qs'

export const Axios = axios.create({
    baseURL: '/', // 反向代理配置
    timeout: 10000,
    responseType: 'json',
    withCredentials: true, // 是否允许携带cookie等
    headers: {
        'Content-Type': 'application/x-www-form-urlencodedcharset=utf-8'
    }
})

// POST传参序列化(添加请求拦截器)
Axios.interceptors.request.use(
    config => {
        // 在发送请求之前做的事情
        if (
            config.method === 'post' ||
            config.method === 'put' ||
            config.method === 'delete'
        ) {
            // 序列化
            config.data = qs.stringify(config.data)
        }

        // 若是有权限token, 则给请求头加上token
        if (window.localStorage.token) {
            config.headers.Authorization = window.localStorage.token
        }
        return config
    },
    error => {
        // 错误提示信息, 可以自定义, 如 `Element-ui Message` 弹窗组件
        // `error.data.error.message` 根据实际返回错误数据信息调整
        console.error(error.data.error.message)
    }
)

// 返回状态判断(添加响应拦截器)
Axios.interceptors.response.use(
    res => {
        let data = res.data
        return data
    },
    error => {
        let errorInfo = error.data ? (error.data.error ? error.data.error.message : error.data) : {}
        return Promise.reject(errorInfo)
    }
)

// Vue-axios 中文文档 https://www.kancloud.cn/yunye/axios/234845
