/**
 * @name [utils.js 工具方法]
 */
import type from './type' // 数据类型校验

export default {
    ...type,
    /**
     * [zeroPadding 数值小于10自动补零]
     * @param {Number, String} num
     * @return {String} 补零后数字值
     */
    zeroPadding (num) {
        return num < 10 ? `0${num}` : `${num}`
    },
    /**
     * [formatDate 格式化时间]
     * @param {Date} date
     * @return {String} '年月日时'
     * @example
     *    '20180319'
     */
    formatDate (date = new Date(), separator = '') {
        let self = this
        let year = date.getFullYear()
        let month = self.zeroPadding(date.getMonth() + 1)
        let day = self.zeroPadding(date.getDate())
        let hour = self.zeroPadding(date.getHours())
        let minute = self.zeroPadding(date.getMinutes())

        let dateItemObj = {
            year,
            month,
            day,
            hour,
            minute
        }
        let finalParams
        // 有分隔符
        if (separator) {
            finalParams = Object.assign({}, dateItemObj, {
                timerStr: `${year}${separator}${month}${separator}${day}`
            })
        } else {
            finalParams = Object.assign({}, dateItemObj, {
                timerStr: `${year}${month}${day}${hour}`
            })
        }

        return finalParams
    },
    timerHandle (start, end) {
        let self = this
        let startTime = new Date(start)
        let endTime = new Date(end)
        let formatStart = ''
        let formatEnd = ''
        // 如果结束日期小于开始日期，交换两个时间
        if (Date.parse(endTime) < Date.parse(startTime)) {
            formatStart = self.formatDate(endTime, '-')
            formatEnd = self.formatDate(startTime, '-')
        } else {
            formatStart = self.formatDate(startTime, '-')
            formatEnd = self.formatDate(endTime, '-')
        }
        let startHour = `${formatStart.hour}:00`
        let endHour = `${formatEnd.hour}:00`
        let tempDate
        if (formatStart.timerStr === formatEnd.timerStr) { // 同一天
            tempDate = `${formatEnd.timerStr} ${startHour} - ${endHour}`
        } else {
            tempDate = `${formatStart.timerStr} ${startHour} - ${formatEnd.timerStr} ${endHour}`
        }

        return tempDate
    },
    /**
     * [splitArr 分割数组为指定数量一组]
     * @param {Array} arr 目标数组
     * @param {Int} groupLen 指定数量
     * @return {Array} 分割后的数组
     */
    splitArr (arr = [], groupLen = 2) {
        if (!arr.length) return

        let result = []
        for (let i = 0, len = arr.length; i < len; i += groupLen) {
            result.push(arr.slice(i, i + groupLen))
        }
        return result
    },
    /**
     * [isEmpty 判断对象对否为空]
     * @param {Object} target
     * @return {Boolean} true - 空对象, false - 非空对象
     */
    isEmpty (target) {
        if (!target) {
            return true
        } else if (typeof target === 'object' && !Object.keys(target).length) {
            return true
        } else {
            return false
        }
    },
    /**
     * [replacePunctuation 替换字符串中的所有标点符号]
     * @param {String} str 目标字符串
     * @return {String} 去除标点符号的字符串
     */
    replacePunctuation (str) {
        if (typeof str !== 'string') {
            str = str.toString()
        }
        return str.replace(/\+/g, ' ')
    // return str.replace(/[\ |\~|\`|\!|\@|\#|\$|\%|\^|\&|\*|\(|\)|\-|\_|\+|\=|\||\\|\[|\]|\{|\}|\;|\:|\"|\'|\,|\<|\.|\>|\/|\?]/g, " ");
    },
    /**
     * [currDates 当前日期]
     * @return {Object} y - 年, like => 2018
     *                  m - 月, like => 03
     *                  d - 日, like => 02
     *                  h - 时, like => 09
     *                  M - 分, like => 09
     *                  s - 秒, like => 07
     *                  ym - 年月, like => 201804
     *                  ymd - 年月日, like => 20180402
     */
    currDates () {
        let self = this
        let date = new Date()
        let y = date.getFullYear()
        let m = `${self.zeroPadding(date.getMonth() + 1)}`
        let d = `${self.zeroPadding(date.getDate())}`
        let h = `${self.zeroPadding(date.getHours())}`
        let M = `${self.zeroPadding(date.getMinutes())}`
        let s = `${self.zeroPadding(date.getSeconds())}`
        let ym = `${y}${m}`
        let ymd = `${ym}${d}`
        return {
            y,
            m,
            d,
            h,
            M,
            s,
            ym,
            ymd
        }
    },
    getFirstEndDay (assignDay = new Date(), prevMonth = false) {
        let self = this
        let nowdays = new Date(assignDay)
        if (nowdays === 'Invalid Date') {
            console.error('不合法的日期格式')
            return ''
        }
        let y = nowdays.getFullYear()
        let m = self.zeroPadding(prevMonth ? nowdays.getMonth() : nowdays.getMonth() + 1)
        if (m === 0 && prevMonth) {
            m = 12
            y = y - 1
        }

        let firstDay = `${y}-${m}-01`
        let myDate = new Date(y, m, 0)
        let lastDay = `${y}-${m}-${myDate.getDate()}`
        let nextMonthFirstDay = self.getDateStr(lastDay, 1)

        return {
            firstDay,
            lastDay,
            nextMonthFirstDay
        }
    },
    /**
     * [getDateStr 获取指定日期前后的日期]
     * @param {String, Date} assignDay 指定日期, default: 当前日期
     * @param {Int} addDayCount 指定天数, 0 - 今天, > 0 今天后的日期, < 0 今天前的日期
     * @return {String} 指定日期前后的日期字符串
     */
    getDateStr (assignDay, addDayCount) {
        let self = this
        let dd = new Date(assignDay)
        dd.setDate(dd.getDate() + addDayCount) // 获取 addDayCount 后的日期
        let y = dd.getFullYear()
        let m = self.zeroPadding(dd.getMonth() + 1)
        let d = self.zeroPadding(dd.getDate())

        return `${y}-${m}-${d}`
    },
    /**
     * [getScrollBarWidth 获取滚动条实际宽度]
     * @return {Int} 滚动条实际宽度
     */
    getScrollBarWidth () {
        let noScroll
        let scroll
        let oDiv = document.createElement('div')
        oDiv.style.cssText = 'position:absolute; top:-1000px; width:100px; height:100px; overflow:hidden;'
        noScroll = document.body.appendChild(oDiv).clientWidth
        oDiv.style.overflowY = 'scroll'
        scroll = oDiv.clientWidth
        document.body.removeChild(oDiv)

        return noScroll - scroll
    },
    /**
     * [modalOpenEvent Modal显示事件]
     * @description Modal显示时禁止浏览器滚动, 同时设置body 'padding-right' 为 '滚动条实际宽度', 防止抖动
     * @param {Boolean} bl 是否显示Modal
     *    true - 显示Modal, 同时禁止浏览器滚动
     *    false - 隐藏Modal, 同时允许浏览器滚动
     */
    modalOpenEvent (bl) {
        let self = this
        let oBody = document.body
        let scrollBarWidth = `${self.getScrollBarWidth()}px`

        oBody.className = bl ? 'modal-open' : ''
        oBody.style.cssText = bl ? `padding-right: ${scrollBarWidth}` : ''
    },
    /**
     * 获取每月的活动事件
     * 按月返回数据，由于有跨天的活动，所以需要处理活动日历数据,按日期展开ajax数据
     * @param {*} data :shlib.events.month返回数据 result.datas
     * @param {*} date : 日期对象{start: '2018-03-01',end:'2018-04-01'}
     */
    getEventsData (data, date) {
        if (data.length === 0) {
            return []
        }
        let self = this
        date.start = new Date(date.start + ' 00:00:00')
        date.end = new Date(date.end + ' 00:00:00')
        let tempList = []
        // let events = [];  // 日历活动提示
        // let dates = []; // 保存日期列表

        for (let j = 0, k = data.length; j < k; j++) {
            tempList = tempList.concat(spreadData(data[j]))
        }

        tempList.sort(function (a, b) {
            return a.date < b.date ? 1 : -1
        })

        return collectEventsData(tempList)

        // 判断活动是否跨天，如果是，则展开数据
        function spreadData (data) {
            // console.log(data);
            let startTime = new Date(data.v3)
            let endTime = new Date(data.v4)
            let formatStart = ''
            let formatEnd = ''
            let tempList = []
            if (Date.parse(endTime) < Date.parse(startTime)) {
                let tempTime = startTime
                startTime = endTime
                endTime = tempTime
                formatStart = self.formatDate(endTime, '-')
                formatEnd = self.formatDate(startTime, '-')
            } else {
                formatStart = self.formatDate(startTime, '-')
                formatEnd = self.formatDate(endTime, '-')
            }
            if (formatStart.timerStr !== formatEnd.timerStr) {
                // 活动跨天
                let s = 0
                let e = 0
                if (Date.parse(startTime) < Date.parse(date.start)) {
                    // 当前活动开始日期小于当前月份的开始日期
                    s = date.start
                } else {
                    s = startTime
                }
                if (Date.parse(endTime) > Date.parse(date.end)) {
                    // 当前活动结束日期大于当前月份的结束日期
                    e = new Date(date.end)
                    e.setDate(0)
                } else {
                    e = endTime
                }
                // 活动跨天，进行展开
                for (let j = 0, l = e.getDate() - s.getDate() + 1; j < l; j++) {
                    tempList.push({
                        date: Date.parse(s.getFullYear() + '-' + (s.getMonth() + 1) + '-' + (s.getDate() + j) + ' 00:' + j + ':00'),
                        header: s.getFullYear() + '-' + self.zeroPadding(s.getMonth() + 1) + '-' + self.zeroPadding(s.getDate() + j)
                        // id: data.id, // 活动详情页id
                        // cover: data.v7,
                        // title: data.v1,
                        // type: data.v5,
                        // organizers: data.v8 || '', // 实际暂无v8字段
                        // location: data.v6,
                        // time: self.timerHandle(data.v3, data.v4)
                    })
                }
                // console.log(tempList);
                return tempList
            } else {
                // 同一天
                return [{
                    date: Date.parse(data.v3),
                    header: startTime.getFullYear() + '-' + self.zeroPadding(startTime.getMonth() + 1) + '-' + self.zeroPadding(startTime.getDate())
                    // id: data.id, // 活动详情页id
                    // cover: data.v7,
                    // title: data.v1,
                    // type: data.v5,
                    // organizers: data.v8 || '', // 实际暂无v8字段
                    // location: data.v6,
                    // time: self.timerHandle(data.v3, data.v4)
                }]
            }
        }

        // 收集日历活动事件
        function collectEventsData (data) {
            let events = [] // 日历活动提示
            let dates = [] // 保存日期列表
            for (let j = 0, k = data.length; j < k; j++) {
                const element = data[j]
                let dname = element.header
                // let dname = element.time.split(' ')[0];
                if (!findDate(dname)) {
                    dates.push(dname)
                    findObj(dname)
                }
            }
            // console.log(events);
            return events // 活动日历事件列表

            function findDate (dateName) { // 查找日期列表是否已经存储了日期
                if (dates.indexOf(dateName) < 0) {
                    return false
                }
                return true
            }
            // 遍历列表找对象
            function findObj (dateName) {
                let eventitem = {
                    date: dateName,
                    eventCount: 0
                }
                for (let i = 0, l = data.length; i < l; i++) {
                    const item = data[i]
                    if (item.header === dateName) {
                        eventitem.eventCount++
                    }
                }
                events.unshift(eventitem)
            }
        }
    },

    /**
     * 数组转对象
     */
    array2obj (array, key) {
        var resObj = {}
        for (var i = 0; i < array.length; i++) {
            resObj[array[i][key]] = array[i]
        }
        return resObj
    },

    // 生成随机字符串
    randomStr (e) {
        e = e || 6
        var t = 'ABCDEFGHJKMNPQRSTWXYZabcdefhijkmnprstwxyz2345678'
        var a = t.length
        var n = ''
        for (let i = 0; i < e; i++) {
            n += t.charAt(Math.floor(Math.random() * a))
        }
        return n
    },

    // TODO: 数字每隔3位加逗号。 返回数组
    formatNumber (n) {
        let b = parseInt(n).toString()
        let list = []
        if (b.length >= 4) {
            let len = b.length
            if (len <= 3) { return b }
            let r = len % 3
            let value = r > 0 ? b.slice(0, r) + ',' + b.slice(r, len).match(/\d{3}/g).join(',') : b.slice(r, len).match(/\d{3}/g).join(',')
            list = value.match(/./g)
        } else {
            list = this.setSpecialNumber(n, n.length)
        }
        return list
    },

    // TODO: 数字切割
    // TODO: 接受参数1： 要切割的数字； 参数2： 要切割的数字长度；
    // TODO: 接受  返回一个数组。
    setSpecialNumber (value, length) {
        length = parseInt(length)
        value = parseInt(value)

        let arr = []
        for (let x = 0; x < length; x++) {
            arr[x] = 0
        }
        let i = 0
        while (true) {
            arr[i] = value % 10
            value = parseInt(value / 10)

            if (value === 0 || i === length - 1) break
            i++
        }
        let list = []
        let len = arr.length - 1
        for (let y = 0; len >= 0; len--, y++) {
            list[y] = arr[len]
        }

        return list
    },

    /**
     * @method getTextWith 获取文字属性
     * @param {String} text 文本
     * @param {String} fontStyle 文本样式
     * @returns
     */
    getTextWith (
        text = '',
        fontStyle = '14px/1.6 "Microsoft YaHei"' // 设置字体大小和字体
    ) {
        const canvas = document.createElement('canvas')
        const context = canvas.getContext('2d')
        context.font = fontStyle
        const elem = context.measureText(text)
        return elem
    }
}
