// 此文件主要是webpack开发环境和生成环境的通用配置
'use strict'
// 引入node path路径模块
const path = require('path')
const webpack = require('webpack')
// 引入webpack生产环境配置参数
const prodConfig = require('../config/config').build
const createVueLoaderOptions = require('./vue-loader.config')

// 拼接路径
function resolve(track) {
    return path.join(__dirname, '..', track)
}

// 资源路径
function assetsPath(_path) {
    return path.join(prodConfig.staticPath, _path)
}

const isDev = process.env.NODE_ENV === 'development'

const defaultPlugins = [
    // 主要作用是在此处可以根据isdev配置process.env,一是可以在js代码中可以获取到process.env
    // 二是webpack或则vue等根据process.env如果是development,会给一些特殊的错误提醒等,而这些特殊项在正式环境是不需要的
    new webpack.DefinePlugin({
        'process.env': {
            NODE_ENV: `"${process.env.NODE_ENV}"`
        }
    })
]

// webpack基本配置
module.exports = {
    // 项目入口文件 -> webpack从此处开始构建
    entry: path.resolve(__dirname, '../src/index.js'),
    // 配置模块如何被解析
    resolve: {
        // 自动解析文件扩展名(补全文件后缀)(左 -> 右)
        extensions: ['.js', '.vue', '.json'],
        // 配置别名映射
        alias: {
            // 键后加上$,表示精准匹配！
            vue$: 'vue/dist/vue.esm.js',
            '@': resolve('src'),
            utils: resolve('src/utils'),
            components: resolve('src/components'),
            public: resolve('public')
        }
    },
    module: {
        // 处理模块的规则(可在此处使用不同的loader来处理模块！)
        rules: [
            {
                test: /\.(vue|js|jsx)$/,
                loader: 'eslint-loader',
                exclude: /node_modules/,
                enforce: 'pre' // 先使用eslint预处理对应的文件, 校验通过再使用对应的loader处理
            },
            {
                test: /\.vue$/,
                loader: 'vue-loader', // 处理.vue文件
                options: createVueLoaderOptions(isDev)
            },
            {
                test: /\.jsx$/,
                loader: 'babel-loader' // 处理jsx文件
            },
            // 使用babel-loader来处理src下所有js文件, 详细babel配置在.babelrc, 用来转义ES6
            {
                test: /\.js$/,
                use: {
                    loader: 'babel-loader'
                },
                include: resolve('src')
            },
            // 使用url-loader(file-loader的一个再封装)对引入的图片进行编码,此处可将小于20480字节(20kb)的图片转为DataURL(base64),
            // 大于limit字节的会调用file-loader进行处理！
            // 图片一般发布后都是长缓存,故此处文件名加入hash做版本区分!
            {
                test: /\.(jpg|png|gif|svg|jpeg|txt|exl)$/,
                use: [
                    {
                        loader: 'url-loader',
                        options: {
                            limit: 1024,
                            name: '[name]_[hash:8].[ext]',
                            outputPath: 'images/', // 静态资源输出路径
                        }
                    }
                ]
            },
            {
                // 当前loader需要处理的文件类型(后缀)
                test: /\.(eot|ttf|svg)$/, // iconfont字体文件
                // 处理test中的文件类型需要用到的loader类型(名称)
                use: {
                    loader: 'file-loader', // 处理静态资源类型
                }
            }
        ]
    },
    plugins: defaultPlugins.concat([])
}
