/**
 * @file webpack.dev.conf 开发环境配置
 */
'use strict'
const path = require('path')
const webpack = require('webpack')
const devConfig = require('../config/config').dev // 开发环境配置
const baseConf = require('./webpack.base.conf')
const merge = require('webpack-merge')
const HtmlWebpackPlugin = require('html-webpack-plugin')
const FriendlyErrorsPlugin = require('friendly-errors-webpack-plugin')
const notifier = require('node-notifier')
const devConf = merge(baseConf, {
    output: {
        filename: '[name].[hash:8].js',
        publicPath: devConfig.publicPath
    },
    devtool: devConfig.devtoolType,
    devServer: {
        // HMR控制台log等级
        clientLogLevel: 'warning',
        // 热加载
        hot: true,
        // 自动刷新
        inline: true,
        // 自动打开浏览器
        open: false,
        // 它依赖于HTML5 history API, 如果设置为true, 所有的跳转将指向index.html
        historyApiFallback: true,
        // 主机名
        host: devConfig.host,
        // 端口号
        port: devConfig.port,
        // 配置反向代理解决跨域
        proxy: devConfig.proxyTable,
        // 代码进行压缩。加快开发流程和优化的作用
        compress: true,
        // 在浏览器上全屏显示编译的errors或warnings
        overlay: {
            errors: true,
            warnings: false
        },
        // 终端输出的只有初始启动信息, webpack 的警告和错误是不输出到终端的
        quiet: false
    },
    module: {
        // 处理模块的规则(可在此处使用不同的loader来处理模块)
        rules: [
            // 使用vue-style-loader!css-loader!postcss-loader处理以css结尾的文件
            {
                test: /\.css$/,
                use: [
                    'vue-style-loader',
                    {
                        loader: 'css-loader',
                        options: {
                            sourceMap: true
                        }
                    },
                    {
                        loader: 'postcss-loader',
                        options: {
                            sourceMap: true
                        }
                    }
                ]
            },
            // 使用vue-style-loader!css-loader!postcss-loader处理以less结尾的文件！
            {
                test: /\.less$/,
                use: [
                    'vue-style-loader',
                    {
                        loader: 'css-loader',
                        options: {
                            sourceMap: true
                        }
                    },
                    {
                        loader: 'less-loader',
                        options: {
                            sourceMap: true
                        }
                    },
                    {
                        loader: 'postcss-loader',
                        options: {
                            sourceMap: true
                        }
                    }
                ]
            },
            // 使用vue-style-loader!css-loader!postcss-loader处理以scss结尾的文件
            {
                test: /\.scss$/,
                use: [
                    'vue-style-loader',
                    {
                        loader: 'css-loader',
                        options: {
                            sourceMap: true
                        }
                    },
                    {
                        loader: 'sass-loader',
                        options: {
                            sourceMap: true
                        }
                    },
                    {
                        loader: 'postcss-loader',
                        options: {
                            sourceMap: true
                        }
                    }
                ]
            },
            {
                test: /\.(eot|svg|ttf|woff|woff2)(\?\S*)?$/,
                loader: 'file-loader'
            }
        ]
    },
    plugins: [
        // 开启HMR(热替换功能, 替换更新部分, 不重载页面)
        new webpack.HotModuleReplacementPlugin(),

        // 显示模块相对路径
        new webpack.NamedModulesPlugin(),

        // 编译出错时, 该插件可跳过输出, 确保输出资源不会包含错误
        // new webpack.NoEmitOnErrorsPlugin()

        new HtmlWebpackPlugin({
            title: 'vue-cli-init',
            filename: 'index.html',
            template: 'index.html',
            // js资源插入位置, true表示插入到body元素底部
            inject: true
        }),

        // 编译提示插件
        new FriendlyErrorsPlugin({
            // 编译成功提示
            compilationSuccessInfo: {
                message: [
                    `Your application is running here: http://${devConfig.host}:${devConfig.port}`
                ]
            },
            onErrors: function (severity, errors) {
                if (severity !== 'error') {
                    return
                }
                const error = errors[0]
                const filename = error.file && error.file.split('!').pop()
                // 编译出错时右下角弹出错误提示
                notifier.notify({
                    title: 'vue-cli-init',
                    message: severity + ': ' + error.name,
                    subtitle: filename || '',
                    icon: path.join(__dirname, 'xc-cli.png')
                })
            },
            clearConsole: true
        })
    ]
})

module.exports = devConf
