/**
 * 孵化器用户逻辑层 
 * 作者: 陈金晶
 * 主要包括有 孵化器账号的登录 
 * 预留好  重置密码  退出登录 接口 
 * 密码规则：6-18位 只允许有数字和字符，可以只有数字也可以只有字母 不允许有特殊字符 2023年02月21日确定需求
 */

import * as eccFormParamConfig from "../../../config/eccFormParamConfig";
import { ERRORENUM } from "../../../config/errorEnum";
import * as splitResultConfig from "../../../config/splitResultConfig";
import * as fuhuaqiData from "../../../data/fuHuaQi/fuhuaqi"
import { BizError } from "../../../util/bizError";
import * as sysTools from "../../../tools/system";
import { eccFormParam } from "../../../util/verificationParam";
import { extractData } from "../../../util/piecemeal";


/**
 * 小程序的孵化器登录
 * 小程序端
 * @param uscc 信用代码
 * @param pwd 密码
 * @returns fuhuaqiUserInfo:{uscc, name} 登录后的信息
 */
export async function login(uscc:string, pwd:string) {
    let fuhuaqiInfo = await fuhuaqiData.findFuHuaQiByUSCC(uscc);
    if(!fuhuaqiInfo) throw new BizError(ERRORENUM.账号不存在);

    let checkPwd = sysTools.getPwdMd5(fuhuaqiInfo.uscc, pwd);
    if (fuhuaqiInfo.pwd != checkPwd) throw new BizError(ERRORENUM.密码错误);

    const Token = sysTools.getToken(uscc);
    
    let fuhuaqiUserInfo = {
        uscc: fuhuaqiInfo.uscc,
        // name: fuhuaqiInfo.name,
        firstLogin : !fuhuaqiInfo.firstLoginIsChangePwd,
        token:Token
    };

    fuhuaqiInfo.token = Token;
    fuhuaqiInfo.tokenMs = new Date().valueOf();

    await fuhuaqiInfo.save();
    
    return fuhuaqiUserInfo;
}


/**
 * 首次登录修改密码
 * 小程序端
 * @param uscc 孵化器统一信用代码
 * @param pwd 新密码
 * @param confirmPwd 确认密码 
 * @returns 
 */
export async function firstLoginChangePwd(uscc:string, pwd:string, confirmPwd:string ) {
    if (pwd != confirmPwd) throw new BizError(ERRORENUM.密码不一致);
    if (pwd.search(/^[A-Za-z0-9]{6,18}$/) < 0) throw new BizError(ERRORENUM.密码只能由6至18位字符和数字组成);

    let dataBaseInfo = await fuhuaqiData.findFuHuaQiByUSCC(uscc);
    if (dataBaseInfo.firstLoginIsChangePwd) throw new BizError(ERRORENUM.不能重复修改密码, `重复调用了首次登录之后的修改密码接口${uscc}`);
    //todo 校验密码
    dataBaseInfo.pwd = sysTools.getPwdMd5(uscc, sysTools.md5PwdStr(pwd));
    dataBaseInfo.firstLoginIsChangePwd = true;
    await dataBaseInfo.save();

    return {isSuccess:true};
}


/**
 * 重置密码
 * 小程序端 2.0
 * @param uscc 信用代码
 * @param pwd 原密码  md5之后的
 * @param newPwd 新密码  未md5
 * @param confirmPwd 确认新密码  未md5
 * @returns {isSuccess:true/false}
 */
export async function changePassword(uscc:string, pwd:string, newPwd:string, confirmPwd:string ) {
    if (newPwd != confirmPwd) throw new BizError(ERRORENUM.密码不一致);
    if (newPwd.search(/^[A-Za-z0-9]{6,18}$/) < 0) throw new BizError(ERRORENUM.密码只能由6至18位字符和数字组成);

    let fuhuaqiInfo = await fuhuaqiData.findFuHuaQiByUSCC(uscc);
    if (!fuhuaqiInfo) throw new BizError(ERRORENUM.未找到数据);

    /**由于pwd是md5之后的 所以这里md5一次即可 */
    let checkPwd = sysTools.getPwdMd5(fuhuaqiInfo.uscc, pwd);
    if (fuhuaqiInfo.pwd != checkPwd) throw new BizError(ERRORENUM.密码错误);

    /**考虑到如果前端把新密码也md5 就不可以验证是不是符合规则 所以前端传的是明文
     * 我们初始化密码的时候 有两次加密 第一次是密码md5 然后再和uscc 进行md5
     * 于是这里也要如此操作
     */
    fuhuaqiInfo.pwd = sysTools.getPwdMd5(uscc, sysTools.md5PwdStr(newPwd));
    await fuhuaqiInfo.save();

    return {isSuccess:true};
}


/**
 * 退出登录
 * 小程序端
 * @param uscc 信用代码
 * @returns {isSuccess:true/false}
 */
export async function logout(uscc:string){
    if (typeof uscc != "string") throw new BizError(ERRORENUM.参数错误, uscc);
    let fuhuaqiInfo = await fuhuaqiData.findFuHuaQiByUSCC(uscc);

    fuhuaqiInfo.token = "";
    fuhuaqiInfo.tokenMs = 0;

    await fuhuaqiInfo.save();
    return {isSuccess:true};
}



export async function bindFuHuaQi(uscc:string, bindUscc:string, bindPwd:string) {
    let fuhuaqiInfo = await fuhuaqiData.findFuHuaQiByUSCC(uscc);

    // if ( !fuhuaqiInfo ) throw new BizError(ERRORENUM.); 
}

/**
 * 切换账号列表
 * 小程序端 2.0
 * @param uscc 
 */
export async function changeUserList(uscc:string) {
    

}