/**
 * 企业相关逻辑
 * 作者: 李欣鸣
 * 包括新注册企业，迁入企业
 * 
 */
import * as eccFormParamConfig from "../../../config/eccFormParamConfig";
import { ERRORENUM } from "../../../config/errorEnum";
import { BizError } from "../../../util/bizError";
import * as sysTools from "../../../tools/system";
import * as enterpriseData from "../../../data/enterprise/enterprise";
import * as splitResultConfig from "../../../config/splitResultConfig";
import * as configEnum from "../../../config/enum";
import { eccFormParam } from "../../../util/verificationParam";
import * as verificationEnumTools from "../../../util/verificationEnum";
import { checkChange, extractData } from "../../../util/piecemeal";


/**
 * 添加新注册企业
 * 小程序端
 * @param uscc 孵化器统一信用代码
 * @param param 表单参数
 * @returns {isSuccess:true/false}
 */
export async function registerEnterprise(uscc:string, param) {
    /**校验表单参数 */
    eccFormParam("新注册企业", eccFormParamConfig.RegisterEnterpriseParamConfig, param);
    verificationEnumTools.eccEnumValue('添加新注册企业', 'industry', configEnum.INDUSTRY, param.industry);
    if ( !sysTools.eccUscc(param.uscc) ) throw new BizError(ERRORENUM.统一社会信用代码不合法);

    /**不能出现重复的统一社会信用代码 草稿的也不行 */
    let dataBaseInfo = await enterpriseData.findEnterpriseByUscc(param.uscc);
    if ( dataBaseInfo&& dataBaseInfo.uscc)  throw new BizError(ERRORENUM.该企业已存在, `${param.uscc}已经在库中存在`);

    /**添加新注册企业标识 */
    param.draftId = sysTools.getDraftId(uscc);//草稿箱id
    param.isCreate = true;//标识此次操作是否是新注册企业并不是迁入企业
    const TaskId = sysTools.getTaskId(uscc);
    await enterpriseData.createEnterprise(uscc, TaskId, param);

    return {isSuccess:true};
}


/**
 * 修改新注册企业的表单
 * 小程序端
 * @param uscc 孵化器统一信用代码
 * @param param 表单参数
 * @returns {isSuccess:true/false}
 */
export async function updateRegisterEnterprise(uscc:string, param) {
    /**校验表单参数 */
    eccFormParam("新注册企业表单修改", eccFormParamConfig.EnterpriseParamFormConfig, param );
    verificationEnumTools.eccEnumValue('修改新注册企业', 'industry', configEnum.INDUSTRY, param.industry);
    if ( !sysTools.eccUscc(param.uscc) ) throw new BizError(ERRORENUM.统一社会信用代码不合法);

    let dataBaseInfo = await enterpriseData.findEnterpriseByDraftId(param.draftId);
    if (!dataBaseInfo || !dataBaseInfo.uscc) throw new BizError(ERRORENUM.未找到数据, `库中不存在${param.uscc}这个企业`);

    /**修改时校验uscc是否重复 */
    let selectInfo = await enterpriseData.selectOneEnterpriseByParam({uscc:param.uscc, draftId:{"$ne":param.draftId} });
    if (selectInfo && selectInfo.uscc) throw new BizError(ERRORENUM.统一社会信用代码重复, `${param.uscc}已经在库中存在 不可以修改为这个统一信用代码`);

    /**修改字段 */
    let changeList = checkChange(param, dataBaseInfo);
    if ( !changeList.length ) throw new BizError(ERRORENUM.数据无更新, `${param.uscc}数据无更新`);
    changeList.forEach(key => {
        dataBaseInfo[key] = param[key];
    });

    await dataBaseInfo.save();

    return {isSuccess:true};
}


/**
 * 添加迁入企业登记
 * 小程序端
 * @param uscc 孵化器统一信用代码
 * @param param 表单参数
 * @returns {isSuccess:true/false}
 */
export async function moveInEnterprise(uscc:string, param) {
    /**校验表单参数 */
    eccFormParam("迁入企业登记", eccFormParamConfig.MoveInEnterpriseParamConfig, param );
    verificationEnumTools.eccEnumValue('添加迁入企业', 'industry', configEnum.INDUSTRY, param.industry);
    if ( !sysTools.eccUscc(param.uscc) ) throw new BizError(ERRORENUM.统一社会信用代码不合法);

    /**不能出现重复的统一社会信用代码 草稿的也不行 */
    let dataBaseInfo = await enterpriseData.findEnterpriseByUscc(param.uscc);
    if ( dataBaseInfo&& dataBaseInfo.uscc)  throw new BizError(ERRORENUM.该企业已存在, `${param.uscc}已经在库中存在`);

    /**添加新注册企业标识 */
    param.draftId = sysTools.getDraftId(uscc);//草稿箱id
    param.isCreate = false;//标识此次操作是否是迁入企业并不是新注册企业
    const TaskId =  sysTools.getTaskId(uscc);
    await enterpriseData.createEnterprise(uscc, TaskId, param);

    return {isSuccess:true};
}


/**
 * 修改迁入企业登记信息
 * 小程序端
 * @param uscc 孵化器统一信用代码
 * @param param 表单参数
 * @returns {isSuccess:true/false}
 */
export async function updateMoveInEnterprise(uscc:string, param) {
    /**校验表单参数 */
    eccFormParam("修改迁入企业登记信息", eccFormParamConfig.EnterpriseInPutParamFormConfig, param);
    verificationEnumTools.eccEnumValue('修改迁入企业', 'industry', configEnum.INDUSTRY, param.industry);
    if ( !sysTools.eccUscc(param.uscc) ) throw new BizError(ERRORENUM.统一社会信用代码不合法);

    /**企业库中是否存在记录 */
    let dataBaseInfo = await enterpriseData.findEnterpriseByDraftId(param.draftId);
    if (!dataBaseInfo || !dataBaseInfo.uscc) throw new BizError(ERRORENUM.未找到数据, `库中不存在${param.uscc}这个企业`);

    /**修改时校验uscc是否重复 */
    let selectInfo = await enterpriseData.selectOneEnterpriseByParam({uscc:param.uscc, draftId:{"$ne":param.draftId} });
    if (selectInfo && selectInfo.uscc) throw new BizError(ERRORENUM.统一社会信用代码重复, `${param.uscc}已经在库中存在 不可以修改为这个统一信用代码`);

    /**修改数据 */
    let changeList = checkChange(param, dataBaseInfo);
    if ( !changeList.length ) throw new BizError(ERRORENUM.数据无更新, `${param.uscc}数据无更新`);
    changeList.forEach(key => {
        dataBaseInfo[key] = param[key];
    });
    await dataBaseInfo.save();

    return {isSuccess:true};
}


/**
 * 根据企业id查询企业信息
 * 包括新迁入企业和新注册企业的回显
 * 小程序端  回显
 * @param uscc 企业统一信用代码
 * @param draftId 草稿id
 * @returns {data} data 返回表单
 */
export async function getEnterpriseByDraftId(draftId:string) {
    let dataBaseInfo = await enterpriseData.findEnterpriseByDraftId(draftId);
    if (!dataBaseInfo || !dataBaseInfo.uscc) throw new BizError(ERRORENUM.未找到数据, `库中不存在draftId=${draftId}这个企业`);

    /**新注册企业和迁入企业回显信息不一样 */
    let data:any = {};
    if (dataBaseInfo.isCreate) data = extractData(splitResultConfig.EnterpriseInfoConfig, dataBaseInfo, false);
    else data = extractData(splitResultConfig.EnterpriseMoveInInfoConfig, dataBaseInfo, false);

    return {data};
}


/**
 * 根据企业id删除草稿企业
 * 小程序端
 * @param uscc 企业统一信用代码
 * @param draftId 草稿id
 * @returns {isSuccess:true/false}
 */
export async function deleteEnterpriseByDraftId(draftId:string) {
    let dataBaseInfo = await enterpriseData.findEnterpriseByDraftId(draftId);
    /**删除限制 */
    if (!dataBaseInfo || !dataBaseInfo.uscc) throw new BizError(ERRORENUM.未找到数据, `库中不存在draftId=${draftId}这个企业`);
    if (dataBaseInfo.draftLock) throw new BizError(ERRORENUM.已入库的数据不能删除, draftId);

    await enterpriseData.deleteEnterprise(draftId);

    return {isSuccess:true};
}


/**
 * 我的企业列表
 * 小程序端 2.0
 * @param uscc 孵化器统一信用代码
 * @param state 孵化器状态 遵循 枚举FUHUASTATE
 * @param name 孵化器名称
 * @param page 当前页面
 */
export async function myEnterprise(uscc:string, state:number, name:string, page:number) {
    
    let result:any = {};

    let selectParam:any = { fuHuaQiUscc:uscc};
    if (name) selectParam.name = name;
    if (state == configEnum.INSIDESTATE.全部) {
        selectParam.state = {"$lt":4};
    } else selectParam.state = state;
    
    let dataList = await enterpriseData.findEnterpriseListToPage(selectParam, (page-1)*10);

    /**当前条件数据量 */
    let dataCount = await enterpriseData.findEnterpriseCount(selectParam);
    /**当前条件页面数量 */
    result.dataCount = dataCount;
    result.pageCount = Math.ceil(dataCount/10);

    if (state != configEnum.FUHUASTATE.迁出) {
        /**查询所有的数据量 */
        let allCount = await enterpriseData.findEnterpriseCount({ fuHuaQiUscc:uscc });
        let virtualCount = 0;
        if (state == configEnum.FUHUASTATE.虚拟孵化) virtualCount = dataCount;
        else virtualCount = await enterpriseData.findEnterpriseCount({ fuHuaQiUscc:uscc, state:configEnum.FUHUASTATE.虚拟孵化 });

        result.dataNumberList = [
            {key:configEnum.INSIDESTATE.全部, value:`全部(${allCount})`},
            {key:configEnum.INSIDESTATE.实体孵化, value:`实体孵化(${allCount-virtualCount})`},
            {key:configEnum.INSIDESTATE.虚拟孵化, value:`虚拟孵化(${virtualCount})`},
        ];

    }

    let enterpriseList = [];

    dataList.forEach(info => {
        let changeData:any = extractData(splitResultConfig.myEnterpriseDataConfig, info, false);
        changeData.state = verificationEnumTools.changeEnumValue(configEnum.FUHUASTATE, changeData.state);
        enterpriseList.push(changeData);
    });

    result.enterpriseList = enterpriseList;
    
    return result;
}


/**
 * 修改企业孵化状态 修改为实体
 * 小程序端 2.0
 * @param fuHuaQiUscc 孵化器统一信用代码
 * @param uscc 企业统一信用代码
 * @param leasedArea 租赁面积
 * @returns {isSuccess:true/false}
 */
export async function updatePhysicalInfo(fuHuaQiUscc:string, uscc:string, leasedArea:number) {
    let enterpriseInfo = await enterpriseData.findEnterpriseByUscc(uscc);
    if (!enterpriseInfo) throw new BizError(ERRORENUM.该企业不存在, "修改企业孵化状态 修改为实体",`孵化器 ${fuHuaQiUscc} -> 企业${uscc}`);

    if (enterpriseInfo.fuHuaQiUscc != fuHuaQiUscc) throw new BizError(ERRORENUM.无法操作其他孵化器的企业, "修改企业孵化状态 修改为实体",`孵化器 ${fuHuaQiUscc} -> 企业${uscc}`);

    enterpriseInfo.leasedArea = leasedArea;

    await enterpriseInfo.save();

    return {isUsccess:true};
}


/**
 * 修改企业孵化状态 修改为虚拟
 * 小程序端 2.0
 * @param fuHuaQiUscc 孵化器统一信用代码
 * @param uscc 企业统一信用代码
 * @param virtualCauseDes 虚拟孵化其他原因
 * @param virtualCause 虚拟孵化模式
 * @returns {isSuccess:true/false}
 */
export async function updateVirtualInfo(fuHuaQiUscc:string, uscc:string, virtualCauseDes:string, virtualCause:number) {
    verificationEnumTools.eccEnumValue('修改企业孵化状态', '修改为虚拟', configEnum.VIRTUALCAUSE, virtualCause);

    if (virtualCause != configEnum.VIRTUALCAUSE.其他 && !virtualCauseDes) {
        throw new BizError(ERRORENUM.请填入其他原因, "修改企业孵化状态 修改为虚拟");
    }

    let enterpriseInfo = await enterpriseData.findEnterpriseByUscc(uscc);
    if (!enterpriseInfo) throw new BizError(ERRORENUM.该企业不存在, "修改企业孵化状态 修改为实体",`孵化器 ${fuHuaQiUscc} -> 企业${uscc}`);

    if (enterpriseInfo.fuHuaQiUscc != fuHuaQiUscc) throw new BizError(ERRORENUM.无法操作其他孵化器的企业, "修改企业孵化状态 修改为实体",`孵化器 ${fuHuaQiUscc} -> 企业${uscc}`);

    enterpriseInfo.virtualCause = virtualCause;
    if (virtualCauseDes)  enterpriseInfo.virtualCauseDes = virtualCauseDes;

    await enterpriseInfo.save();

    return {isUsccess:true};
}


/**
 * 修改企业孵化状态 修改为迁出
 * 小程序端 2.0
 * @param fuHuaQiUscc 孵化器统一信用代码
 * @param uscc 企业统一信用代码
 * @param moveOutType 迁出类型
 * @param moveOutCause 迁出原因 多选
 * @returns {isSuccess:true/false}
 */
export async function updateMoveOutInfo(fuHuaQiUscc:string, uscc:string,  moveOutType:number, moveOutCause ) {
    verificationEnumTools.eccEnumValue('修改企业孵化状态', '修改为迁出', configEnum.MOVEOUTTYPE, moveOutType);
    verificationEnumTools.eccEnumValue('修改企业孵化状态', '修改为迁出', configEnum.MOVEOUTCAUSE, moveOutCause);

    let enterpriseInfo = await enterpriseData.findEnterpriseByUscc(uscc);
    if (!enterpriseInfo) throw new BizError(ERRORENUM.该企业不存在, "修改企业孵化状态 修改为实体",`孵化器 ${fuHuaQiUscc} -> 企业${uscc}`);

    if (enterpriseInfo.fuHuaQiUscc != fuHuaQiUscc) throw new BizError(ERRORENUM.无法操作其他孵化器的企业, "修改企业孵化状态 修改为实体",`孵化器 ${fuHuaQiUscc} -> 企业${uscc}`);

    enterpriseInfo.moveOutType = moveOutType;
    enterpriseInfo.moveOutCause = moveOutCause;

    await enterpriseInfo.save();

    return {isUsccess:true};
}