/**
 * 孵化器  大学/科研院所合作机构 逻辑
 * 作者：lxm
 * 
 */

import * as fuhuaqiData from "../../../../data/fuHuaQi/fuhuaqi";
import { getRandomId } from "../../../../tools/system";
import * as icrData  from "../../../../data/fuHuaQi/instituQualify/icr";
import { changeEnumValue } from "../../../../util/verificationEnum";
import { COOPERATIONMODE } from "../../../../config/enum";
import moment = require("moment");
import { addPoint } from "../../../point";
import { FUHUAQINODEENUM, OPERATETYPEENUM, POINTTYPEENUM } from "../../../../config/pointConfig";
import { BizError } from "../../../../util/bizError";
import { ERRORENUM } from "../../../../config/errorEnum";


/**
 * 创建 大学/科研院所合作机构 记录
 * @param uscc 孵化器统一信用代码
 * @param po 合作机构
 * @param ce 合作企业
 * @param cMode 合作方式
 * @param cAchievements 合作成果 3.3去除
 * @param startTimeMs 开始时间
 * @returns isSuccess 是否操作成功
 */
export async function createIcr(uscc:string, po:string, ce:string, cMode, startTimeMs:number) {
    let fuHuaQiInfo = await fuhuaqiData.findFuHuaQiByUSCC(uscc);

    let id = getRandomId(uscc);
    await icrData.addData(id, uscc, fuHuaQiInfo.name, po, ce, cMode, startTimeMs);

    addPoint(POINTTYPEENUM.孵化器, uscc, FUHUAQINODEENUM.机构资质_大学科研院所合作机构, {po, ce, cMode, startTimeMs}, null, OPERATETYPEENUM.添加 );
    
    return {isSuccess:true};
}


/**
 * 修改孵化器 大学/科研院所合作机构 记录
 * @param uscc 孵化器统一信用代码
 * @param id 标识
 * @param po 合作机构
 * @param ce 合作企业
 * @param cMode 合作方式
 * @param cAchievements 合作成果 3.3去除
 * @param startTimeMs 开始时间
 * @returns isSuccess 是否操作成功
 */
export async function updateIcr(uscc:string, id:string, po:string, ce:string, cMode, startTimeMs:number) {
    let dataInfo = await icrData.findOneData(id);
    if (uscc != dataInfo.uscc) throw new BizError(ERRORENUM.参数错误, "修改大学/科研院所合作机构信息时", `${uscc}尝试修改${dataInfo.uscc}的信息`);
    let oldData = {
        po : dataInfo.po,
        ce : dataInfo.ce,
        cMode : dataInfo.cMode,
        startTimeMs : dataInfo.startTimeMs,
    };
    let newData = { po, ce, cMode, startTimeMs };
    addPoint(POINTTYPEENUM.孵化器, uscc, FUHUAQINODEENUM.机构资质_大学科研院所合作机构, oldData, newData, OPERATETYPEENUM.修改 );

    dataInfo.po = po;
    dataInfo.ce = ce;
    dataInfo.cMode = cMode;
    dataInfo.startTimeMs = startTimeMs;

    await dataInfo.save();

    return {isSuccess:true};
}


/**
 * 回显 大学/科研院所合作机构 记录
 * @param id 记录标识
 * @returns 
 */
export async function selectIcr(id:string) {
    let dataInfo = await icrData.findOneData(id);

    let startUpData = {
        po:dataInfo.po,
        ce:dataInfo.ce,
        cMode:dataInfo.cMode,
        startTimeMs:dataInfo.startTimeMs,
    };

    return {startUpData} 
}


/**
 * 删除 大学/科研院所合作机构 记录
 * @param id 记录标识
 * @returns 
 */
export async function deleteIcr(uscc:string, id:string) {

    let dataInfo = await icrData.findOneData(id);
    if (uscc != dataInfo.uscc) throw new BizError(ERRORENUM.参数错误, "删除大学/科研院所合作机构信息时", `${uscc}尝试修改${dataInfo.uscc}的信息`);

    let oldData = {
        po : dataInfo.po,
        ce : dataInfo.ce,
        cMode : dataInfo.cMode,
        startTimeMs : dataInfo.startTimeMs,
    };

    await icrData.deleteDataById(id);


    addPoint(POINTTYPEENUM.孵化器, uscc, FUHUAQINODEENUM.机构资质_大学科研院所合作机构, oldData, null, OPERATETYPEENUM.删除 );
    return {isSuccess:true} 
}


/**
 * 获取 大学/科研院所合作机构 记录列表
 * @param uscc 孵化器统一信用代码
 * @returns 
 */
export async function getIcrListData(uscc) {
    let registerList = await icrData.findListDataByParam({uscc});

    let dataList = [];
    registerList.forEach(info => {
        dataList.push({
            id:info.id,
            po:info.po,
            ce:info.ce,
            cMode:changeEnumValue(COOPERATIONMODE, info.cMode),
            startTimeMs:moment(info.startTimeMs).format("YYYY/MM"),
        });
    });

    return {dataList, title:"大学/科研院所合作机构"}
}
