import moment = require("moment");
import { COOPERATIONMODE, OPERATIONMODEL, OUTPUTTYPE, SPECIALITY, STATEENUM, TEACHERLV } from "../../../config/enum";
import { findAllFuHuaQiOperationNameMap, findFuHuaQiList, findFuHuaQiOperationNameMapByParam, findFuHuaQiUserCount, findFuHuaQiUserList } from "../../../data/fuHuaQi/fuhuaqi";
import { distinctFuHuaQiRegister, findI18nRegisterListByParam, findI18nRegisterListByParamToPage, getI18nRegisterCountByParam } from "../../../data/fuHuaQi/instituQualify/i18nRegister";
import * as icrData from "../../../data/fuHuaQi/instituQualify/icr";
import * as startupTeacherData from "../../../data/fuHuaQi/instituQualify/startupTeacher";
import * as zjBeyondLayoutData from "../../../data/fuHuaQi/instituQualify/zjBeyondLayout";
import { changeEnumValue } from "../../../util/verificationEnum";



/**
 * 国际化已登记列表
 * @param name 孵化器运营机构名称  模糊查询
 * @param page 页面 
 * @returns 
 */
export async function getI18nRegisterDataList(name:string, page:number) {
    let selectParam:any = {};
    if (name) selectParam.name = {"$regex":`${name}`};

    let dataBaseList = await findI18nRegisterListByParamToPage(selectParam, (page-1)*10);
    let count = await getI18nRegisterCountByParam(selectParam);
    let dataList = [];
    dataBaseList.forEach(info => {
        dataList.push({
            name:info.name,
            country:info.country || '',//国家
            partners:info.partners || '',//合作机构
            cMode:info.cMode || '',//合作方式
            cAchievements:info.cAchievements || '',//合作成果
        });
    });

    return {count, dataList, pageCount:Math.ceil(count/10) };
}


/**
 * 国际化未登记列表
 * @param name 孵化器运营机构名称 模糊
 * @param page 页面
 * @returns 
 */
export async function getI18nNotRegisterDataList(name:string, page:number) {
    let selectParam:any = {};
    if (name) selectParam.operationName = {"$regex":`${name}`};

    //填报孵化器列表
    let registerMap = await distinctFuHuaQiRegister();

    //全部孵化器map
    let fuHuaQiMap = await findFuHuaQiOperationNameMapByParam(selectParam);

    let notRegisterList = [];
    for (let uscc in fuHuaQiMap) {
        if (!registerMap[uscc]) notRegisterList.push({
            uscc, 
            name:fuHuaQiMap[uscc], 
            country: '-', 
            partners:'-', 
            cMode:'-',
            cAchievements: '-'
        });
    }

    let dataList = notRegisterList.slice((page-1)*10, ((page-1)*10)+10);
    let count = notRegisterList.length;
    
    return {count, dataList, pageCount:Math.ceil(count/10) };
}


/**
 * 导出 国际化登记列表  （已填报）
 * @param name 孵化器运营机构名称  
 * @param type 导出类型
 */
export async function outPutI18nRegisterData(fuHuaQiName:string, type:number) {

    let name = "全部";
    let selectParam:any = {};

    if (type == OUTPUTTYPE.当前数据 ) {
        name = "当前";
        if (fuHuaQiName) selectParam.name = {"$regex":`${fuHuaQiName}`};
    }

    let dataBaseList = await findI18nRegisterListByParam(selectParam);

    let dataList = [["运营机构名称", "国家", "合作机构","合作方式","合作成果"]];
    dataBaseList.forEach(info => {
        let subList = [info.name, info.country, info.partners, info.cMode, info.cAchievements ];
        dataList.push(subList);
    });

    return {dataList, name};

}


/**
 * 国际化未登记列表
 * @param name 孵化器运营机构名称 模糊
 * @param page 页面
 * @returns 
 */
export async function outPutI18nNotRegisterData(fuHuaQiName:string, type:number) {
    let name = "全部";
    let selectParam:any = {};

    if (type == OUTPUTTYPE.当前数据 ) {
        name = "当前";
        if (fuHuaQiName) selectParam.name = {"$regex":`${fuHuaQiName}`};
    }

    //填报孵化器列表
    let registerMap = await distinctFuHuaQiRegister();

    //全部孵化器map
    let fuHuaQiMap = await findFuHuaQiOperationNameMapByParam(selectParam);

    let notRegisterList = [];
    for (let uscc in fuHuaQiMap) {
        if (!registerMap[uscc]) notRegisterList.push({
            uscc, 
            name:fuHuaQiMap[uscc], 
            country: '-', 
            partners:'-', 
            cMode:'-',
            cAchievements: '-'
        });
    }

    let dataList = [["运营机构名称", "国家", "合作机构","合作方式","合作成果"]];
    notRegisterList.forEach(info => {
        let subList = [info.name, info.country, info.partners, info.cMode, info.cAchievements ];
        dataList.push(subList);
    });

    return {dataList, name};
}

export async function getIcrDataList(name:string, cMode:number, page:number) {
    let selectParam:any = {};
    if (name) selectParam.name = {"$regex":`${name}`};
    if (cMode) selectParam.cMode = {"$in":[cMode]};


    let dataBaseList = await icrData.findIcrListByParamToPage(selectParam, (page-1)*10);
    let count = await icrData.getIcrCountByParam(selectParam);
    let dataList = [];
    dataBaseList.forEach(info => {
        let startTimeMs = '';
        if (info.startTimeMs) {
            startTimeMs = moment(info.startTimeMs).format('YYYY-MM-DD')//开始时间
        }
        dataList.push({
            name:info.name,//运营机构名称
            po:info.po,//合作机构
            ce:info.ce,//合作企业
            cMode:changeEnumValue(COOPERATIONMODE, info.cMode),
            cAchievements:info.cAchievements,//合作成果
            startTimeMs
        });
    });

    return {count, dataList, pageCount:Math.ceil(count/10) };
}

export async function outputIcrDataList(fuHuaQiName:string, cMode:number, type:number) {
    let name = "全部";
    let selectParam:any = {};

    if (type == OUTPUTTYPE.当前数据 ) {
        name = "当前";
        if (fuHuaQiName) selectParam.name = {"$regex":`${fuHuaQiName}`};
        if (cMode) selectParam.cMode = {"$in":[cMode]};
    }


    let dataBaseList = await icrData.findListDataByParam(selectParam );

    let dataList = [["运营机构名称", "合作机构", "合作企业","合作方式","合作成果", "开始时间"]];
    dataBaseList.forEach(info => {
        let subList = [info.name, info.po, info.ce, info.cMode, info.cAchievements, info.startTimeMs];
        dataList.push(subList);
    });

    return {dataList, name};
}


export async function getStartupTeacherDataList(name:string, speciality:number, teacherLv:number, page:number) {
    let selectParam:any = {};
    if (name) selectParam.name = {"$regex":`${name}`};
    if (speciality) selectParam.speciality = {"$in":[speciality]};
    if (teacherLv) selectParam.teacherLv = {"$in":[teacherLv]};


    let dataBaseList = await startupTeacherData.findStartupTeacherListByParamToPage(selectParam, (page-1)*10);
    let count = await startupTeacherData.getStartupTeacherCountByParam(selectParam);
    let dataList = [];
    dataBaseList.forEach(info => {
        dataList.push({
            name:info.name,//运营机构名称
            teacherName:info.teacherName,//导师名称
            unit:info.unit,//单位
            duties:info.duties,//职务
            speciality:changeEnumValue(SPECIALITY, info.speciality),//专业特长
            teacherLv:changeEnumValue(TEACHERLV, info.teacherLv),//导师级别
            serviceEnterprises:info.serviceEnterprises,//服务企业名单
        });
    });

    return {count, dataList, pageCount:Math.ceil(count/10) };
}

export async function outputStartupTeacherDataList(fuHuaQiName:string, speciality:number, teacherLv:number, type:number) {
    let name = "全部";
    let selectParam:any = {};

    if (type == OUTPUTTYPE.当前数据 ) {
        name = "当前";
        if (fuHuaQiName) selectParam.name = {"$regex":`${fuHuaQiName}`};
        if (speciality) selectParam.speciality = {"$in":[speciality]};
        if (teacherLv) selectParam.teacherLv = {"$in":[teacherLv]};
    }

    let dataBaseList = await startupTeacherData.findListDataByParam(selectParam);

    let dataList = [["运营机构名称", "导师名称", "单位","职务","专业特长", "导师级别", "服务企业名单"]];
    dataBaseList.forEach(info => {
        let subList= [ info.name, info.teacherName, info.unit, info.duties, 
            changeEnumValue(SPECIALITY, info.speciality), changeEnumValue(TEACHERLV, info.teacherLv), info.serviceEnterprises,];
        dataList.push(subList);
    });

    return {dataList, name};
}


export async function getZJBeyondLayoutDataList(name:string, operationMode:number, page:number) {
    let selectParam:any = {};
    if (name) selectParam.name = {"$regex":`${name}`};
    if (operationMode) selectParam.operationMode = operationMode;

    let dataBaseList = await zjBeyondLayoutData.findZjBeyondLayoutListByParamToPage(selectParam, (page-1)*10);
    let count = await zjBeyondLayoutData.getZjBeyondLayoutCountByParam(selectParam);
    let dataList = [];
    dataBaseList.forEach(info => {
        dataList.push({
            name:info.name,//运营机构名称
            city:info.city,//布局城市
            area:info.area,//空间面积
            operationMode:changeEnumValue(OPERATIONMODEL, info.operationMode),//运营模式
        });
    });

    return {count, dataList, pageCount:Math.ceil(count/10) };
}


export async function outputZJBeyondLayoutDataList(fuHuaQiName:string, operationMode:number, type:number) {
    let name = "全部";
    let selectParam:any = {};

    if (type == OUTPUTTYPE.当前数据 ) {
        name = "当前";
        if (fuHuaQiName) selectParam.name = {"$regex":`${fuHuaQiName}`};
        if (operationMode) selectParam.operationMode = operationMode;
    }
    
    let dataBaseList = await zjBeyondLayoutData.findListDataByParam(selectParam);

    let dataList = [["运营机构名称", "布局城市", "空间面积","运营模式"]];
    dataBaseList.forEach(info => {
        let subList = [info.name, info.city, info.area,changeEnumValue(OPERATIONMODEL, info.operationMode) ];
        dataList.push(subList);
    });

    return {dataList, name};
}


export async function getPTPDataList(name:string, isCreatePTP:number, isParticipateInPTP:number, page:number) {
    let selectParam:any = {};
    if (name) selectParam.operationName = {"$regex":`${name}`};
    if (isCreatePTP) selectParam.isCreatePTP = isCreatePTP;
    if (isParticipateInPTP != 1 ) {
        if (isParticipateInPTP == 2) selectParam.isParticipateInPTP = true;
        else selectParam.isParticipateInPTP = false;
    }

    let dataBaseList = await findFuHuaQiUserList(selectParam, (page-1)*10);
    let count = await findFuHuaQiUserCount(selectParam);
    let dataList = [];
    dataBaseList.forEach(info => {
        dataList.push({
            name:info.operationName,//运营机构名称
            isCreatePTP:changeEnumValue(STATEENUM, info.isCreatePTP),//基金规模(万元)
            scalePTP:info.scalePTP,//基金规模
            isParticipateInPTP:info.isParticipateInPTP ? "是" : "否",//是否参与创投基金
            participatingFundCompany:info.participatingFundCompany//参股基金公司名
        });
    });

    return {count, dataList, pageCount:Math.ceil(count/10) };
}



export async function outputPTPDataList(fuHuaQiName:string, isCreatePTP:number, isParticipateInPTP:number, type:number) {
    let name = "全部";
    let selectParam:any = {};

    if (type == OUTPUTTYPE.当前数据 ) {
        name = "当前";
        if (fuHuaQiName) selectParam.operationName = {"$regex":`${fuHuaQiName}`};
        if (isCreatePTP) selectParam.isCreatePTP = isCreatePTP;
        if (isParticipateInPTP) {
            if (isParticipateInPTP == 1) selectParam.isParticipateInPTP = true;
            else selectParam.isParticipateInPTP = false;
        }
    }

    let dataBaseList = await findFuHuaQiList(selectParam);
    let dataList = [["运营机构名称", "是否创立创投基金", "基金规模(万元)", "是否参与创投基金", "参股基金公司名"]];
    dataBaseList.forEach(info => {
        let isParticipateInPTP = info.isParticipateInPTP ? "是" : "否";//是否参与创投基金
        let subList = [info.operationName, changeEnumValue(STATEENUM, info.isCreatePTP), info.scalePTP, isParticipateInPTP, info.participatingFundCompany];
        dataList.push(subList);
    });

    return {dataList, name};
}


export async function getProfessionalTechnologyDataList(name:string, isProfessionalTechnology:number, year:number, page:number) {
    let selectParam:any = {};
    if (name) selectParam.operationName = {"$regex":`${name}`};
    if (isProfessionalTechnology) selectParam.isProfessionalTechnology = isProfessionalTechnology;
    if (year) {
        let st = new Date(year+'-01-01 00:00:00').valueOf();
        let et = new Date(year+1+'-01-01 00:00:00').valueOf();
        selectParam["$and"] = [{"professionalTechnologyCreateTime":{"$gt":st}}, {"professionalTechnologyCreateTime":{"$lt":et}}];
    }

    let dataBaseList = await findFuHuaQiUserList(selectParam, (page-1)*10);
    let count = await findFuHuaQiUserCount(selectParam);
    let dataList = [];
    dataBaseList.forEach(info => {
        let professionalTechnologyCreateTime = '';
        if (info.professionalTechnologyCreateTime) {
            professionalTechnologyCreateTime = moment(info.professionalTechnologyCreateTime).format("YYYY年MM月DD日");
        }
        dataList.push({
            name:info.operationName,//运营机构名称
            isProfessionalTechnology:changeEnumValue(STATEENUM, info.isProfessionalTechnology),//是否专业技术平台
            professionalTechnologyName:info.professionalTechnologyName,//专业技术平台名称
            professionalTechnologyCreateTime,//时间  年份 xxxx年01月01日 的时间戳
            professionalTechnologyAmount:info.professionalTechnologyAmount,//投资金额 万元
        });
    });

    return {count, dataList, pageCount:Math.ceil(count/10) };
}

export async function outputProfessionalTechnologyDataList(fuHuaQiName:string, isProfessionalTechnology:number, year:number, type:number) {
    let name = "全部";
    let selectParam:any = {};

    if (type == OUTPUTTYPE.当前数据 ) {
        name = "当前";
        if (fuHuaQiName) selectParam.operationName = {"$regex":`${fuHuaQiName}`};
        if (isProfessionalTechnology) selectParam.isProfessionalTechnology = isProfessionalTechnology;
        if (year) {
            let st = new Date(year+'-01-01 00:00:00').valueOf();
            let et = new Date(year+1+'-01-01 00:00:00').valueOf();
            selectParam["$and"] = [{"professionalTechnologyCreateTime":{"$gt":st}}, {"professionalTechnologyCreateTime":{"$lt":et}}];
        }
    }

    let dataBaseList = await findFuHuaQiList(selectParam);
    let dataList = [["运营机构名称", "是否专业技术平台", "专业技术平台名称", "建立时间", "投资金额"]];
    dataBaseList.forEach(info => {
        let timeFormat = moment(info.professionalTechnologyCreateTime).format("YYYY/MM-DD");
        let subList = [info.operationName, changeEnumValue(STATEENUM, info.isProfessionalTechnology), info.professionalTechnologyName, timeFormat, info.professionalTechnologyAmount ];
        dataList.push(subList);
    });

    return {dataList, name};
}
