/**
 * 企业投资信息表
 * draftLock 草稿锁 当任务提交的时候，将此字段改为true
 * 一个企业在一个月只能录入一个融资数据 2023-02-07 确定此需求
 */

import {Schema} from 'mongoose';
import { baseDB } from '../../db/mongo/dbInit';

const financingSchema = new Schema({
    uscc:{type:String, index:true},//融资企业统一信用代码 冗余字段
    fuHuaQiUscc:{type:String, index:true},//孵化器统一信用代码 冗余字段
    name:String,//企业名称
    taskId:{type:String, index:true},//任务id
    logonAdd:String,//注册地址
    operatingAdd:String,//经营地址
    financingAmount:Number,//融资金额(万元)
    investmentInstitutionsName:String,//投资机构名称
    timeToObtainInvestment:Number,//获得投资时间
    fuHuaQiInvestment:{type:Boolean, default:false},//孵化器是否投资
    fuHuaQiInvestmentAmount:Number,//孵化器投资金额(万元)
    fuHuaQiInvestmentStyle:Number,//孵化器投资方式
    draftLock:{type:Boolean, default:false},//草稿锁，true为提交之后，false为草稿
    createTime:Number,//录入时间
    industry:[Number],//领域
    logonTime:Number,//注册时间
    financingRounds:Number,//融资轮次
});

var financingModel;
export function initModel(){
    financingModel = baseDB.model('financing', financingSchema);
    financingModel.selectOnceData = async function (paramater:object) {
        let selectInfo = await financingModel.findOne(paramater).exec();
        if (selectInfo) {
            if (!selectInfo.runSave) {
                selectInfo.runSave = selectInfo.save;
                selectInfo.save = save.bind(selectInfo)
            }
        }
        return selectInfo;
    }
}
export async function save(throwError=false) {
    if (!this.isModified()) return;
    await this.runSave({validateBeforeSave:false}).catch(err=>{
        console.log(err);
    });
}


/**
 * 通过taskId和企业uscc获取融资信息
 * 支持.save方法保存对象修改
 * @param taskId 任务id 格式遵循tool中getTaskId
 * @param uscc 企业统一信用代码
 * @returns {}
 */
export async function findFinancingInfoByTaskIdAndSucc(taskId:string, uscc:string) {
    return await financingModel.selectOnceData({taskId, uscc});
}


/**
 * 通过taskId 获取此次任务添加的融资列表
 * @param taskId 任务id 格式遵循tool中getTaskId
 * @returns [{}]
 */
export async function findFinancingListByTaskId(taskId:string) {
    return await financingModel.find({taskId, draftLock:false});
}


/**
 * 将所有taskId匹配数据draftLock字段修改为true
 * @param taskId 任务id 格式遵循tool中getTaskId
 */
export async function updateFinancingDraftLock(taskId:string) {
    return await financingModel.updateMany({taskId}, {$set:{draftLock:true}});
}


/**
 * 添加新的融资数据
 * @param fuHuaQiUscc 孵化器统一信用代码
 * @param taskId 任务id 格式遵循tool中getTaskId
 * @param param 所添加表单 需要提前验证
 */
export async function createFinancing(fuHuaQiUscc:string, taskId:string, uscc:string, logonTime:number,industry, param) {
    let addInfo = Object.assign({fuHuaQiUscc, taskId, uscc, logonTime, createTime:new Date().valueOf()}, param);
    if (logonTime) addInfo.logonTime = logonTime;
    await financingModel.create(addInfo);
}


/**
 * 删除创建企业的草稿报表
 * @param uscc 企业统一信用代码
 * @param taskId 任务id
 */
export async function deleteEnterpriseFinancing(uscc:string, taskId:string) {
    return await financingModel.deleteOne({uscc:uscc, taskId}).exec();
}


/**
 * 分页获取所有融资企业
 * @param selectParam 查询参数
 * @param skipCount 跳过数量
 * @returns [] 孵化器列表
 */
 export async function findFinancingListToPage(selectParam:object, skipCount:number) {
    return await financingModel.find(selectParam).skip(skipCount).limit(10);
}

/**
 * 获取所有融资企业
 * @param selectParam 查询参数
 * @returns number 数据数量
 */
 export async function findFinancingCount(selectParam:object) {
    return await financingModel.find(selectParam).count();
}

/**
 * 获取所有融资企业
 * @param selectParam 查询参数
 * @param skipCount 跳过数量
 * @returns [] 孵化器列表
 */
export async function findFinancingList(selectParam:object) {
    return await financingModel.find(selectParam);
}