/**
 * 企业经营数据相关 
 * 作者:lxm
 * 说明： 首页经营数据   经营数据补充
 * 备注:  填报的经营数据在  填报逻辑 里
 */

import moment = require("moment");
import { findBusinessDataByUsccAndYear } from "../../data/enterprise/businessdata";
import { eccEnumValue } from "../../util/verificationEnum";
import { BUSINESSDATATYPE } from "../../config/enum";
import { findRepleishDataByTypeAndYear, replenishData, selectRepleishData } from "../../data/enterprise/replenish";


/**
 * 获取经营数据
 * @param uscc 企业统一信用代码
 */
export async function getBusinessData(uscc:string) {
    let lastYear = moment().subtract(1, 'years').year();
    let thisYear = new Date().getFullYear();

    let thisYearData = await findBusinessDataByUsccAndYear(uscc, thisYear);
    let thisBusinessData = {RD:0, BI:0, TXP:0};
    thisYearData.forEach(info => {
        for (let key in thisBusinessData) {
            thisBusinessData[key] += info[key] || 0;
        }
    });
    
    let result = {};
    for (let key in thisBusinessData) {
        let {unit, count} = countUnitData(thisBusinessData[key]);
        result[`this${key}`] = {unit, count }
    }

    /**找缺失  要求是：缺一个季度就要爆红 只要提交了，就算是0也不算是缺失的数据 */
    let deletionMap = {RD:0, BI:0, TXP:0};

    let lastYearData = await findBusinessDataByUsccAndYear(uscc, lastYear);
    let lastBusinessData = {RD:0, BI:0, TXP:0};
    lastYearData.forEach(info => {
        for (let key in lastBusinessData) {
            lastBusinessData[key] += info[key] || 0;
            deletionMap[key] += 1;
        }
    });
    let lastYearRepleishData = await selectRepleishData(uscc, lastYear);
    lastYearRepleishData.forEach(info => {
        if (info.type == BUSINESSDATATYPE.研发投入) {
            lastBusinessData.RD += info.value;
            deletionMap["RD"] += 1;
        }
        if (info.type == BUSINESSDATATYPE.纳税) {
            lastBusinessData.TXP += info.value;
            deletionMap["TXP"] += 1;
        }
        if (info.type == BUSINESSDATATYPE.营业收入) {
            lastBusinessData.BI += info.value;
            deletionMap["BI"] += 1;
        }
    });

    for (let key in lastBusinessData) {
        let {unit, count} = countUnitData(lastBusinessData[key]);

        let deletion = deletionMap[key] <4;

        result[`last${key}`] = {unit, count, deletion}
    }

    return result;
}


/**
 * 获取格式  将金额数据转换为金额加单位
 * 单位为：元  万元  根据长度自动转换  
 * 超过千万 单位为万 否则单位为元
 * @param data 
 * @returns 
 */
function countUnitData(data:number) {
    if (!data) return {unit:"元", count:0};
    if (data/10000000 < 10) {
        return {
            unit:"元",
            count:Math.floor(data)
        }
    }
    return {
        unit:"万元",
        count:Math.floor(data/10000)
    }
}


/**
 * 补录经营数据
 * @param uscc 统一信用代码
 * @param quarter 补录季度
 * @param type 补录类型
 * @param value 值
 */
export async function replenishBusinessData(uscc:string, type:number, data) {
    eccEnumValue("补录经营数据", "type", BUSINESSDATATYPE, type);

    let lastYear = parseInt( moment().subtract(1, 'year').format("YYYY") );

    let resultData = [0,0,0,0];
    let lastYearData = await findBusinessDataByUsccAndYear(uscc, lastYear);
    lastYearData.forEach(info => {
        let addIndex = info.quarter - 1;
        if (type == BUSINESSDATATYPE.研发投入) resultData[addIndex] += info.RD;
        if (type == BUSINESSDATATYPE.纳税) resultData[addIndex] += info.TXP;
        if (type == BUSINESSDATATYPE.营业收入) resultData[addIndex] += info.BI;
    });
     
    /**获取上一年度补录 */
    let lastYearReplenishData = await findRepleishDataByTypeAndYear(uscc, type, lastYear);
    lastYearReplenishData.forEach(info => {
        let addIndex = info.quarter - 1;
        resultData[addIndex] += info.value;
    });

    let checkMap = {};
    data.forEach((val, index) => {
        if (index > 4) return;
        if (resultData[index] == 0 && val>0) {
            checkMap[index+1] = val;
        }
    });

    for (let key in checkMap) {
        let quarter = parseInt(key);
        let value = checkMap[key];
        await replenishData(uscc, lastYear, quarter, type, value);
    }

    
    return {isSuccess:true};
}


/**
 * 根据类型上一年度经营数据
 * @param uscc 统一信用代码
 */
export async function lastYearBusinessData(uscc:string, type:number) {
    eccEnumValue("补录经营数据", "type", BUSINESSDATATYPE, type);
    let lastYear = parseInt( moment().subtract(1, 'year').format("YYYY") );
    
    let lastYearData = await findBusinessDataByUsccAndYear(uscc, lastYear);
    let resultData = [0,0,0,0];
    lastYearData.forEach(info => {
        let addIndex = info.quarter - 1;
        if (type == BUSINESSDATATYPE.研发投入) resultData[addIndex] += info.RD;
        if (type == BUSINESSDATATYPE.纳税) resultData[addIndex] += info.TXP;
        if (type == BUSINESSDATATYPE.营业收入) resultData[addIndex] += info.BI;
    });
     
    /**获取上一年度补录 */
    let lastYearReplenishData = await findRepleishDataByTypeAndYear(uscc, type, lastYear);
    lastYearReplenishData.forEach(info => {
        let addIndex = info.quarter - 1;
        resultData[addIndex] += info.value;
    });

    return {dataList:resultData};
}