import { systemConfig } from "../config/serverConfig";
import { getRandomId } from "../tools/system";
import moment = require("moment");
import { findEnterpriseByUscc } from "../data/enterprise/enterprise";
import * as labelData from "../data/label";
import { LABELGOAL, LABELTYPE } from "../config/enum/labelEnum";
import * as dynamicsData from "../data/enterprise/dynamics";
import * as enterpriseRiskData from "../data/enterprise/enterpriseRisk";
import * as dishonestPersonData from "../data/enterprise/dishonestPerson";
import * as illegalityData from "../data/enterprise/illegality";
import * as qualificationsData from "../data/enterprise/qualifications";
import { ENTERPRISECHANGETYPE, ENTERPRISESIZE } from "../config/enum";
import { BizError } from "../util/bizError";
import * as request from 'request';
const md5 = require("md5");
const xlsx = require('node-xlsx');
const path = require('path');
const fs = require('fs');

function randomId(tableName:string) {
    let randomStr = `${new Date().valueOf()}_${Math.ceil(Math.random()*100000)}`;
    return `${tableName}_${md5(randomStr)}`;
}


function getQcc(url:string, query?, headers?) {
    if (!url || (url.search(/http:/) && url.search(/https:/)) ) throw new BizError(!url ? "请求地址为空" : "请求地址错误");
    return new Promise((resolve, reject)=>{
        let paramater:any = { url, json:true };
        if (query) paramater.qs = query;
        if (headers) paramater.headers = headers;
        request.get(paramater, function (err, r, body) {
            if (err) {
                return reject(err);
            }
            if (body) {
                if (body.Status == 200) return resolve(body.Result);
                else if (body.Status == 201) return resolve({});
            }
            console.log(query.searchKey, body.Message || body);
            return resolve({});
        });
    })
}


function getExcel(filePath) {
    const workSheetsFromFile = xlsx.parse(filePath);
    let sheetMap = {};
    let sheetList = [];

    for (let i = 0; i < workSheetsFromFile.length; i++) {
        let sheetInfo = workSheetsFromFile[i];
        sheetMap[sheetInfo.name] = sheetInfo.data;
        sheetList.push(sheetInfo);
    }

    return {sheetMap, sheetList}  
}

/**
 * 同步企查查数据
 * 逻辑：获取数据库所有企业uscc，进行企查查数据同步
 */
export async function updateQCCDataTask() {
    //读取excel表格
    let fuhuaqiExcelData = getExcel(path.join(__dirname.substring(0,__dirname.indexOf("out")), "res", '张江在孵企业总表0819.xlsx' ));
    
    let dataList = fuhuaqiExcelData.sheetList[0].data;
    for (let i = 0; i < dataList.length; i++) {
        let subList = dataList[i];
        
        let uscc = subList[6];
        
        let enterInfo = await findEnterpriseByUscc(uscc);
        if (!enterInfo || !enterInfo.uscc) continue

        await updateItemQCCData(uscc);

        console.log("同步", subList[0], "剩余", dataList.length-i);
    }
    console.log(`${fuhuaqiExcelData}家企业同步成功`);
}



/**
 * 更新单个企业
 * @param uscc 
 * @param eId 
 */
export async function updateItemQCCData(uscc) {
    await 工商信息(uscc);
    await 准入尽职调查(uscc);
    await 失信核查(uscc);
    await 严重违法核查(uscc);
    await 科创分(uscc);
    await 资质证书查询(uscc);
}

/**
 * 封装了一个请求方法
 * @param uscc 
 * @returns 
 */
function getReqParam(uscc, isKeyNo?) {
    const TimeSpan = Math.round(new Date().valueOf()/ 1000);
    let header = {
        Token:md5(`${systemConfig.qccKey}${TimeSpan}${systemConfig.qccSecretKey}`).toUpperCase(),
        TimeSpan
    };
    let query:any = {};
    if (isKeyNo == "isKeyNo") {
        query = {
            key:systemConfig.qccKey,
            keyNo:uscc,
        };
    } else if (isKeyNo == "Id") {
        query = {
            key:systemConfig.qccKey,
            id:uscc,
        };
    } else if (isKeyNo == "certId") {
        query = {
            key:systemConfig.qccKey,
            certId:uscc,
        };
    } else if (isKeyNo == "企业名称") {
        query = {
            key:systemConfig.qccKey,
            keyword:uscc,
        }
    } else {
        query = {
            key:systemConfig.qccKey,
            searchKey:uscc,
            // keyNo:"云合智网（上海）技术有限公司"
        };
    }
    
    return {header, query};
}


function formatAddress(original) {
    // 匹配省/直辖市（支持"北京市"/"上海"等格式）
    const cityMatch = original.match(/^(.*?(?:省|市|自治区|特别行政区))/);
    const city = cityMatch ? cityMatch[1] : original.split(/[市区县]/)[0] + '市';
    
    // 匹配区级（支持"浦东新区"/"朝阳区"等格式）
    const districtMatch = original.match(/(?:省|市)(.*?(?:区|县|市|旗))/);
    const district = districtMatch ? districtMatch[1] : '';
    
    // 处理详细地址（自动保留原格式，仅修正明显错误）
    let detail = original.replace(city, '').replace(district, '')
        .replace(/(\d+)[幢栋](\d+)层/g, (_, num1, num2) => {
            const chineseNums = ['零', '一', '二', '三', '四', '五', '六', '七', '八', '九'];
            return `${chineseNums[parseInt(num1)] || num1}幢${num2}${num2.length === 1 ? '室' : ''}`;
        })
        .trim();
    
    return [city, city, district || city, detail || '地址不详'];
}



/**
 * 同步工商信息数据
 * @param uscc 
 * @param eId 
 * @returns 
 */
async function 工商信息(uscc) {
    let {header, query} = getReqParam(uscc);
    
    let 工商信息:any = await getQcc('https://api.qichacha.com/ECIInfoVerify/GetInfo', query, header);//获取工商信息
    if (Object.keys(工商信息).length) {
        let enterInfo = await findEnterpriseByUscc(uscc);

        let khsfsb = getReqParam(uscc);
        let 客户身份识别:any = await getQcc('https://api.qichacha.com/CustomerDueDiligence/KYC', khsfsb.query, khsfsb.header);//获取工商信息
        let TermEnd = "无固定期限";
        if (工商信息.TermEnd) TermEnd = moment(工商信息.TermEnd).format("YYYY-MM-DD");
    
        if (工商信息.OperName) {
           enterInfo.legalPerson = 工商信息.OperName;
        }
        
        //企业基础信息
        let guanWang = "";
        if (工商信息.ContactInfo &&工商信息.ContactInfo.WebSite && 工商信息.ContactInfo.WebSite[0]) {
            enterInfo.guanWang = 工商信息.ContactInfo.WebSite[0].Url || "";
        }

        if (客户身份识别.Data && 客户身份识别.Data.Scale) {
            switch (客户身份识别.Data.Scale) {
                case "L":  enterInfo.qiYeGuiMo = ENTERPRISESIZE.大型; break;
                case "M":  enterInfo.qiYeGuiMo = ENTERPRISESIZE.中型; break;
                case "S":  enterInfo.qiYeGuiMo = ENTERPRISESIZE.小型; break;
                case "XS":  enterInfo.qiYeGuiMo = ENTERPRISESIZE.微型; break;
            }
        }
    
        if (工商信息.Address) {
            let logonAddressList = formatAddress(工商信息.Address);
            enterInfo.logonAddress = logonAddressList;
        }

        enterInfo.zhuCeHao = 工商信息.No;
        enterInfo.uZhiJiGouDaiMa = 工商信息.OrgNo;
        enterInfo.logonTime = new Date(工商信息.StartDate).valueOf();

        if (!enterInfo.jingYingFanWei) enterInfo.jingYingFanWei = 工商信息.Scope; //经营范围
            
        /**
         * 股权结构
         */
        if (工商信息.Partners) {
            let ownershipList = [];
            工商信息.Partners.forEach(info => {
                let addInfo:any = {
                    name:info.StockName,
                    finalBenefitPercent:info.FinalBenefitPercent,
                    stockPercent:info.StockPercent,
                    stockType:info.StockType,
                    realCapi:info.RealCapi,
                    paidUpCapitalUnit:info.PaidUpCapitalUnit,
                    subscribedCapital:info.SubscribedCapital,
                    subscribedCapitalUnit:info.SubscribedCapitalUnit,
                };
                // if (info.CapiDate) addInfo.capiDate = getMySqlMs(info.CapiDate);
                // if (info.ShoudDate) addInfo.shoudDate = getMySqlMs(info.ShoudDate);

                ownershipList.push(addInfo);
            });
            enterInfo.guQuanJieGou = ownershipList;
        }
    

        /**
         * 企业标签表
         */
        if (工商信息.TagList) {
            for (let i = 0; i < 工商信息.TagList.length; i++) {
                let info = 工商信息.TagList[i];
                let {Type, Name} = info;
                
                let labelInfo = await labelData.selectLabelOne({labelName:Name});

                if (!labelInfo || !labelInfo.id) {
                    //添加
                    labelInfo = {
                        id:randomId("label"),
                        labelName:Name,
                        ctMs:new Date().valueOf(),
                        goal:LABELGOAL.企业,
                        labelType:LABELTYPE.自定义标签,
                        state:false
                    };

                    await labelData.createOneLabel(labelInfo);
                }

                let thisEnterpriseHaveLabel = false;

                enterInfo.labels.forEach(itemLabelInfo => {
                    if (itemLabelInfo.labelId == labelInfo.id) thisEnterpriseHaveLabel = true;
                })
                if (!thisEnterpriseHaveLabel)  enterInfo.labels.push({labelId:labelInfo.id, state:true});
            }
            
        }

    
    
        /**
         * 变更记录目前小程序没有涉及到这部分数据，所以直接与企查查全量更新
         * 逻辑：先删除现有的，再重新添加
         */
        if (工商信息.ChangeRecords && 工商信息.ChangeRecords.length) {
            
            for (let i = 0; i < 工商信息.ChangeRecords.length; i++) {
                let info = 工商信息.ChangeRecords[i];
                let changeParam = {uscc, oldContent:info.AfterContent, changeDate:new Date(info.ChangeDate).valueOf()}
                let changeInfo = await dynamicsData.findOne(changeParam);
                if (!changeInfo || !changeInfo.id) {
                    let addChangeInfo = {
                        id:getRandomId("change"),
                        uscc,
                        enterpriseName:enterInfo.name,
                        fuHuaQiUscc:enterInfo.fuHuaQiUscc,
                        changeType:ENTERPRISECHANGETYPE.其他,
                        content:info.BeforeContent, //变更后
                        oldContent:info.AfterContent, //变更前
                        createTime:new Date(info.ChangeDate).valueOf(),
                        isRead:false
                    }
                    await dynamicsData.createDataByParam(addChangeInfo);
                }
            }
        }
        await enterInfo.save();
    } else {
        console.log(`企查查获取不到该企业数据：${uscc}`);
    }
}

/**
 * 准入尽职调查
 * 逻辑：增量更新
 * @param uscc 
 * @returns 
 */
async function 准入尽职调查(uscc) {
    let enterpriseInfo = await findEnterpriseByUscc(uscc);
    let {header, query} = getReqParam(uscc);
    let qccselectdata:any = await getQcc('https://api.qichacha.com/AcctScan/GetInfo', query, header);

    if (qccselectdata.Data) {

        for (let i = 0; i < qccselectdata.Data.length; i++) {
            let info = qccselectdata.Data[i];
            
            let item = await enterpriseRiskData.findOnce({uscc, title:info.Title});
            if (!item || !item.id) {
                let addInfo:any = {
                    id:randomId("risk"),  
                    uscc,
                    fuHuaQiUscc:enterpriseInfo.fuHuaQiUscc,
                    enterpriseName:enterpriseInfo.name,
                    title:info.Title,
                    description:info.Description,
                    passage:info.Passage,
                    riskType:info.RiskType,
                };
                if (info.TypeCode) addInfo.typeCode = parseInt(info.TypeCode);
                await enterpriseRiskData.createOnce(addInfo);
            }   
        }
        
    }
    
}


/**
 * 失信核查
 * 逻辑：全量更新 删除之前所有的 再添加
 * @param uscc 
 * @returns 
 */
async function 失信核查(uscc) {
    let enterpriseInfo = await findEnterpriseByUscc(uscc);

    let {header, query} = getReqParam(uscc);
    let qccselectdata:any = await getQcc('https://api.qichacha.com/ShixinCheck/GetList', query, header);

    await dishonestPersonData.removeMany({uscc});
    if (qccselectdata.Data) {
        let addList = [];
        qccselectdata.Data.forEach(info => {
            let addInfo:any = {
                id:randomId("dis"),
                uscc,
                fuHuaQiUscc:enterpriseInfo.fuHuaQiUscc,
                enterpriseName:enterpriseInfo.name,
                anno:info.Anno,
                executegov:info.Executegov,
                executestatus:info.Executestatus,
                executeno:info.Executeno,
                actionRemark:info.ActionRemark,
                amount:info.Amount,
            };

            addList.push(addInfo);
        });
        await dishonestPersonData.addMany(addList);
    }
    
}

/**
 * 严重违法核查
 * 逻辑：全量更新 删除之前所有的 再添加
 * @param uscc 
 * @returns 
 */
async function 严重违法核查(uscc) {
    let enterpriseInfo = await findEnterpriseByUscc(uscc);
    let {header, query} = getReqParam(uscc);
    let qccselectdata:any = await getQcc('https://api.qichacha.com/SeriousIllegalCheck/GetList', query, header);

    await illegalityData.removeMany({uscc});
    if (qccselectdata.Data) {
        let addList = [];
        qccselectdata.Data.forEach(info => {
            let addInfo:any = {
                id:randomId("ill"),
                uscc,
                fuHuaQiUscc:enterpriseInfo.fuHuaQiUscc,
                enterpriseName:enterpriseInfo.name,
                type:info.Type,
                addReason:info.AddReason,
                addOffice:info.AddOffice,
                removeReason:info.RemoveReason,
                removeOffice:info.RemoveOffice,
            };

            addList.push(addInfo);
        });
        await illegalityData.addMany(addList);
    }
    
}

/**
 * 科创分
 * 有则更新，无则添加
 * @param uscc 
 * @returns 
 */
async function 科创分(uscc) {
    let enterpriseInfo = await findEnterpriseByUscc(uscc);
    let {header, query} = getReqParam(uscc);
    let qccselectdata:any = await getQcc('https://api.qichacha.com/TechScore/GetInfo', query, header);

    if (qccselectdata.Data) {
        let kcf = {
            industry:qccselectdata.Data.Industry.Industry,
            subIndustry:qccselectdata.Data.Industry.SubIndustry,
            industryRanking:qccselectdata.Data.IndustryRanking,
            score:qccselectdata.Data.Score,
        };

        enterpriseInfo.kcf = kcf;
    }
    
}


async function 资质证书详情查询(id) {
    let {header, query} = getReqParam(id, "certId");
    let qccselectdata:any = await getQcc('https://api.qichacha.com/ECICertification/GetCertificationDetailById', query, header);
    
    return qccselectdata;
}


async function 资质证书查询(uscc) {
     let enterpriseInfo = await findEnterpriseByUscc(uscc);
    let {header, query} = getReqParam(uscc);
    let qccselectdata:any = await getQcc('https://api.qichacha.com/ECICertification/SearchCertification', query, header);
    // let enterpriseInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.企业基础信息表, {uscc}, ["eId"]);
    // let eId = enterpriseInfo.eId;

    await qualificationsData.removeMany({uscc});
    if (Array.isArray(qccselectdata)) {
        let addList = [];
        for (let i = 0; i < qccselectdata.length; i++) {
            let qcc资质证书详情 = {};
            let info = qccselectdata[i];
            let qcc资质证书Id = info.Id;
            qccselectdata[i]["qcc资质证书详情"] = await 资质证书详情查询(qcc资质证书Id);

        }

        qccselectdata.forEach(info => {
            let addInfo:any = {
                 uscc,
                fuHuaQiUscc:enterpriseInfo.fuHuaQiUscc,
                enterpriseName:enterpriseInfo.name,
                id:randomId('qua'),
                name:info.Name,
                type:info.Type,
                no:info.No,
                typeDesc:info.TypeDesc,
                institutionList:info.InstitutionList ? info.InstitutionList[0] :"",
                status:info.Status,
            };

            if (info.StartDate) addInfo.startDate = new Date(info.StartDate).valueOf();
            if (info.EndDate) addInfo.endDate = new Date(info.EndDate).valueOf();

            addList.push(addInfo);
        });
        await qualificationsData.addMany( addList);
    }
    
    return qccselectdata;
}
