/**
 * 月度表单逻辑层 
 * 需求说明     1.月度填报必须填值才能提交
 * 逻辑说明     1.月度表单在孵化器任务生成时一起生成
 *              2.草稿锁(draftLock)默认=false  draftLock=true表示这个任务被提交了 草稿锁用于判断孵化器有没有提交
 *              3.提交状态(isUpdate)默认=false 这个字段=true代表编辑过  这个字段=false表示不存在此任务
 *                可以通过这个字段判断孵化器有没有填报
 *              4.不可以创建数据 因为数据一个月只能有一个 并且这一个系统会生成
 *              5.创建和删除按钮触发 相当于更新了isUpdate的状态
 */

import { ERRORENUM } from "../../../config/errorEnum";
import { BizError } from "../../../util/bizError";
import * as monthData from "../../../data/fuHuaQi/monthTask/monthTable";
import { extractData } from "../../../util/piecemeal";
import * as sysTools from "../../../tools/system";
import { findAllFuHuaQiOperationNameMap } from "../../../data/fuHuaQi/fuhuaqi";
import * as splitResultConfig from "../../../config/splitResultConfig";
import { OUTPUTTYPE } from "../../../config/enum";
import moment = require("moment");

/**
 * 新添加孵化器月度填报
 * 小程序端
 * 逻辑是 将填报编辑状态修改为true 当库里=true时
 * @param uscc 孵化器的统一信用代码
 * @param occupancyRate 本月出租率
 * @returns {isSuccess:true/false}
 */
export async function createReport(uscc:string, occupancyRate:number) {
    const TaskId = sysTools.getTaskId(uscc);
    let dataBaseInfo = await monthData.findMonthTableByTaskId(TaskId);

    /**不可以重复创建 */
    if ( dataBaseInfo.isUpdate ) throw new BizError(ERRORENUM.该孵化器月度填报已存在, `${uscc}重复提交了月度填报值为${occupancyRate}`);

    /**更新状态和数据 */
    dataBaseInfo.isUpdate = true;
    dataBaseInfo.occupancyRate = occupancyRate;

    await dataBaseInfo.save();
    
    return {isSuccess:true};
}


/**
 * 修改孵化器月度报表
 * 小程序端
 * @param uscc 孵化器的统一信用代码
 * @param occupancyRate 本月出租率
 * @returns {isSuccess:true/false}
 */
export async function updateReport(uscc:string, occupancyRate:number) {
    const TaskId = sysTools.getTaskId(uscc);

    let dataBaseInfo = await monthData.findMonthTableByTaskId(TaskId);

    const thisTaskMonth = moment().subtract(1, 'months').month() + 1;//数据填报月份
    const YearNumber =moment().subtract(1, 'months').year();//数据填报年份

    if (dataBaseInfo.month != thisTaskMonth || dataBaseInfo.year != YearNumber) {
        if (!dataBaseInfo || !dataBaseInfo.taskId) throw new BizError(ERRORENUM.未找到数据, `未找到${uscc}的月度报表` );
        if ( !dataBaseInfo.isUpdate ) throw new BizError(ERRORENUM.未找到数据, `${uscc}还没有创建数据就修改了${occupancyRate}`);
        if ( dataBaseInfo.draftLock ) throw new BizError(ERRORENUM.该任务已提交, `${uscc}已经提交了月度报表又 修改 数据${occupancyRate}`);
    }

    dataBaseInfo.occupancyRate = occupancyRate;
    await dataBaseInfo.save();

    return {isSuccess:true};
}


/**
 * 删除孵化器月度报表
 * 小程序端
 * @param uscc 孵化器的统一信用代码
 * @returns  {isSuccess:true/false}
 */
export async function deleteReport(uscc:string) {
    const TaskId = sysTools.getTaskId(uscc);

    let dataBaseInfo = await monthData.findMonthTableByTaskId(TaskId);
    if ( !dataBaseInfo || !dataBaseInfo.isUpdate) throw new BizError(ERRORENUM.未找到数据, `未找到${uscc}的月度报表`);
    if ( dataBaseInfo.draftLock ) throw new BizError(ERRORENUM.该任务已提交, `${uscc}已经提交了月度报表又 删除 数据`);

    dataBaseInfo.isUpdate = false; //修改编辑状态

    await dataBaseInfo.save();

    return {isSuccess:true};
}


/**
 * 根据uscc查询月度报表
 * 小程序端  回显
 * 不能回显已经提交的任务
 * @param uscc 企业统一信用代码
 * @returns {occupancyRate:0} occupancyRate 本月出租率
 */
export async function getMonthByUscc(uscc:string) {
    const TaskId = sysTools.getTaskId(uscc);

    let dataBaseInfo = await monthData.findMonthTableByTaskId(TaskId);
    const thisTaskMonth = moment().subtract(1, 'months').month() + 1;//数据填报月份
    const YearNumber =moment().subtract(1, 'months').year();//数据填报年份
    if (dataBaseInfo.month != thisTaskMonth || dataBaseInfo.year != YearNumber) {
        if ( !dataBaseInfo.isUpdate ) throw new BizError(ERRORENUM.未找到数据, `未找到${uscc}的月度报表` );
        if ( dataBaseInfo.draftLock ) throw new BizError(ERRORENUM.该任务已提交, `${uscc}已经提交了月度报表又 回显 数据`);
    }
    let data = extractData(splitResultConfig.MonthConfig, dataBaseInfo, false);

    return { data };
}

