/**
 * 企业融资情况
 * 作者：lxm
 */

import { EnterpriseCreateFinancingDataConfig, EnterpriseCreateFinancingParamSubConfig, EnterpriseUpdateFinancingDataConfig } from "../../config/eccFormParamConfig";
import { FINANCINGROUNDS, FUHUAQILNVESTMENTSTYLE } from "../../config/enum";
import { ERRORENUM } from "../../config/errorEnum";
import { EnterpriseFinancingListDataConfig } from "../../config/splitResultConfig";
import { findEnterpriseByUscc } from "../../data/enterprise/enterprise";
import { addFinancingInfo, deleteFinancingById, getFinancingById, selectFinancingListByUscc } from "../../data/enterprise/financingInfo";
import { getFinancingId } from "../../tools/system";
import { BizError } from "../../util/bizError";
import { checkChange, extractData } from "../../util/piecemeal";
import { eccEnumValue } from "../../util/verificationEnum";
import { eccFormParam } from "../../util/verificationParam";


/**
 * 获取企业融资情况信息列表
 * @param uscc 企业统一信用代码
 */
export async function getEnterpriseFinancingList(uscc:string) {
    let financingList = await selectFinancingListByUscc(uscc);

    let resultList = [];
    
    financingList.forEach(info => {
        let updateInfo = extractData(EnterpriseFinancingListDataConfig, info, false);
        resultList.push(updateInfo);
    });

    let enterpriseInfo = await findEnterpriseByUscc(uscc);

    if (!enterpriseInfo.tipsFinancingInfo ) {
        enterpriseInfo.tipsFinancingInfo = true;
        await enterpriseInfo.save();
    }

    return { dataList:resultList };
}


/**
 * 添加融资信息
 * @param uscc 企业统一信用代码
 * @param form 表单
 * @returns 
 */ 
export async function addEnterpriseFinancing(uscc:string, form) {
    eccFormParam("企业录入企业融资信息", EnterpriseCreateFinancingDataConfig, form);
    eccEnumValue( "录入企业融资信息", "financingRounds", FINANCINGROUNDS, form.financingRounds);

    if (form.fuHuaQiInvestment) {
        /**如果当孵化器是否投资填了true 则校验 fuHuaQiInvestmentAmount， fuHuaQiInvestmentStyle 两个字段 */
        let subCheckName = "录入企业融资信息_孵化器是否参与";
        let subCheckData = {
            fuHuaQiInvestmentAmount:form.fuHuaQiInvestmentAmount,
            fuHuaQiInvestmentStyle:form.fuHuaQiInvestmentStyle,
        };
        eccFormParam(subCheckName, EnterpriseCreateFinancingParamSubConfig, subCheckData);
        /**校验投资方式是否符合枚举规则 */
        eccEnumValue("添加企业融资信息", "fuHuaQiInvestmentStyle", FUHUAQILNVESTMENTSTYLE, form.fuHuaQiInvestmentStyle);
    } else {
        /**如果 没有填这里给默认数据*/
        form.fuHuaQiInvestmentAmount = 0;
        form.fuHuaQiInvestmentStyle = 0;
    } 

    let enterpriseInfo = await findEnterpriseByUscc(uscc);
    let addInfo:any = {
        id:getFinancingId(uscc), uscc, name:enterpriseInfo.name, createTime:new Date().valueOf(), type:2
    };

    await addFinancingInfo(Object.assign(addInfo, form));

    return {isSuccess:true};
}


/**
 * 获取 企业融资情况
 * 回显
 * @param uscc 企业统一信用代码
 * @param id 融资数据标识
 */
export async function getEnterpriseFinancingInfo(uscc:string, id:string) {
    let financingInfo = await getFinancingById(id);
    
    if (!financingInfo) throw new BizError(ERRORENUM.参数错误, `获取企业融资情况 id在库里找不到`);
    if ( financingInfo.uscc != uscc ) throw new BizError(ERRORENUM.只能修改本企业信息, `${uscc} 想修改 ${id}`);

    let updateInfo = extractData(EnterpriseFinancingListDataConfig, financingInfo, false);

    return updateInfo;
}


/**
 * 修改企业融资数据
 * @param uscc 企业统一信用代码
 * @param form 表单
 * @returns 
 */
export async function updateEnterpriseFinancingInfo(uscc:string, form) {
    eccFormParam("企业修改企业融资信息", EnterpriseUpdateFinancingDataConfig, form);
    eccEnumValue( "录入企业融资信息", "financingRounds", FINANCINGROUNDS, form.financingRounds);

    if (form.fuHuaQiInvestment) {
        /**如果当孵化器是否投资填了true 则校验 fuHuaQiInvestmentAmount， fuHuaQiInvestmentStyle 两个字段 */
        let subCheckName = "录入企业融资信息_孵化器是否参与";
        let subCheckData = {
            fuHuaQiInvestmentAmount:form.fuHuaQiInvestmentAmount,
            fuHuaQiInvestmentStyle:form.fuHuaQiInvestmentStyle,
        };
        eccFormParam(subCheckName, EnterpriseCreateFinancingParamSubConfig, subCheckData);
        /**校验投资方式是否符合枚举规则 */
        eccEnumValue("添加企业融资信息", "fuHuaQiInvestmentStyle", FUHUAQILNVESTMENTSTYLE, form.fuHuaQiInvestmentStyle);
    } else {
        /**如果 没有填这里给默认数据*/
        form.fuHuaQiInvestmentAmount = 0;
        form.fuHuaQiInvestmentStyle = 0;
    } 

    let financingInfo = await getFinancingById(form.id);

    if (!financingInfo) throw new BizError(ERRORENUM.参数错误, `修改企业信息的时候 id在库里找不到`);
    if ( financingInfo.uscc != uscc ) throw new BizError(ERRORENUM.只能修改本企业信息, `${uscc} 想修改 ${form.id}`);

    let changeList = checkChange(form, financingInfo);
    if ( !changeList.length ) throw new BizError(ERRORENUM.数据无更新, `企业更新融资数据`, `${form.id}数据无更新`);
    changeList.forEach(key => {
        if (key != "id") {
            financingInfo[key] = form[key];
        }
    });

    await financingInfo.save();

    return {isSuccess:true};
}


/**
 * 删除企业融资数据
 * @param uscc 企业统一信用代码
 * @param id 唯一标识
 * @returns 
 */
export async function deleteEnterpriseFinancingInfo(uscc:string, id:string) {
    let financingInfo = await getFinancingById(id);
    
    if (!financingInfo) throw new BizError(ERRORENUM.参数错误, `获取企业融资情况 id在库里找不到`);
    if ( financingInfo.uscc != uscc ) throw new BizError(ERRORENUM.只能删除本企业信息, `${uscc} 想修改 ${id}`);

    await deleteFinancingById(id);

    return {isSuccess:true};
}