/**
 * 小程序端 孵化器入口 企业融资相关逻辑
 * 作者: lxm
 * 包括新融资信息的增删改查
 * 
 */
import * as eccFormParamConfig from "../../../config/eccParam/fuHuaQi";
import { ERRORENUM } from "../../../config/errorEnum";
import * as financingData from "../../../data/fuHuaQi/monthTask/financing";
import * as splitResultConfig from "../../../config/splitResultConfig";
import { BizError } from "../../../util/bizError";
import * as taskTool from "../../../tools/taskTool";
import * as enterpriseData from "../../../data/enterprise/enterprise";
import * as configEnum from "../../../config/enum";
import { eccFormParam } from "../../../util/verificationParam";
import { eccEnumValue } from "../../../util/verificationEnum";
import { checkChange, extractData } from "../../../util/piecemeal";


/**
 * 添加企业融资信息
 * 小程序端
 * @param uscc 孵化器统一信用代码
 * @param param 表单配置
 * @returns {isSuccess:true/false}
 */
export async function createFinancingInfo(uscc:string, param) {
    /**校验表单参数*/
    eccFormParam("录入企业融资信息", eccFormParamConfig.FinancingParamConfig, param);
    eccEnumValue( "录入企业融资信息", "financingRounds", configEnum.FINANCINGROUNDS, param.financingRounds);
    if (param.fuHuaQiInvestment) {
        /**如果当孵化器是否投资填了true 则校验 fuHuaQiInvestmentAmount， fuHuaQiInvestmentStyle 两个字段 */
        let subCheckName = "录入企业融资信息_孵化器是否参与";
        let subCheckData = {
            fuHuaQiInvestmentAmount:param.fuHuaQiInvestmentAmount,
            fuHuaQiInvestmentStyle:param.fuHuaQiInvestmentStyle,
        };
        eccFormParam(subCheckName, eccFormParamConfig.FinancingParamSubConfig, subCheckData);
        /**校验投资方式是否符合枚举规则 */
        eccEnumValue("添加企业融资信息", "fuHuaQiInvestmentStyle", configEnum.FUHUAQILNVESTMENTSTYLE, param.fuHuaQiInvestmentStyle);
    } else {
        /**如果 没有填这里给默认数据*/
        param.fuHuaQiInvestmentAmount = 0;
        param.fuHuaQiInvestmentStyle = 0;
    }

    /**校验企业是否存在 */
    let enterpriseInfo = await enterpriseData.findEnterpriseInfoByName(param.name);
    if (!enterpriseInfo) throw new BizError(ERRORENUM.该企业不存在, uscc, param.name);

    /** 前端无需传入企业的uscc  库中需要企业的uscc*/
    param.uscc = enterpriseInfo.uscc;//
    const TaskId = taskTool.getTaskId(uscc);
    
    /**一个企业一个月只能填报一次融资信息 */
    let dataBaseInfo = await financingData.findFinancingInfoByTaskIdAndSucc(TaskId, param.uscc);
    if ( dataBaseInfo&& dataBaseInfo.uscc) throw new BizError(ERRORENUM.该企业当月数据已存在, `${param.uscc}的融资信息已经存在`);

    await financingData.createFinancing(uscc, TaskId, enterpriseInfo.uscc, enterpriseInfo.logonTime,enterpriseInfo.industry || [], param);

    return {isSuccess:true};
}


/**
 * 更新企业融资信息
 * 小程序端
 * 只能更新本月未提交的并且已经创建的企业融资信息
 * @param uscc 孵化器统一信用代码
 * @param param 表单配置
 * @returns {isSuccess:true/false}
 */
export async function updateFinancingInfo(uscc:string, param) {
    /**校验表单参数 */
    eccFormParam( "更新融资企业信息", eccFormParamConfig.FinancingParamUpdateConfig, param );
    if (param.fuHuaQiInvestment) {
        /**如果当孵化器是否投资填了true 则校验 fuHuaQiInvestmentAmount， fuHuaQiInvestmentStyle 两个字段 */
        let subCheckName = "录入企业融资信息_孵化器是否参与";
        let subCheckData = {
            fuHuaQiInvestmentAmount:param.fuHuaQiInvestmentAmount,
            fuHuaQiInvestmentStyle:param.fuHuaQiInvestmentStyle,
        };
        eccFormParam(subCheckName, eccFormParamConfig.FinancingParamSubConfig, subCheckData);
        /**校验投资方式是否符合枚举规则 */
        eccEnumValue("修改企业融资信息", "fuHuaQiInvestmentStyle", configEnum.FUHUAQILNVESTMENTSTYLE, param.fuHuaQiInvestmentStyle);
    } else {
        /**如果 没有填这里给默认数据*/
        param.fuHuaQiInvestmentAmount = 0;
        param.fuHuaQiInvestmentStyle = 0;
    }

    /**找到该企业的本月融资数据 */
    const TaskId = taskTool.getTaskId(uscc);
    let dataBaseInfo = await financingData.findFinancingInfoByTaskIdAndSucc(TaskId, param.uscc);
    if (!dataBaseInfo || !dataBaseInfo.uscc) throw new BizError(ERRORENUM.未找到数据, `库中不存在${TaskId}的任务`);
   
    /**匹配修改 */
    let changeList = checkChange(param, dataBaseInfo);
    if ( !changeList.length ) throw new BizError(ERRORENUM.数据无更新, `${param.uscc}数据无更新`);
    changeList.forEach(key => {
        dataBaseInfo[key] = param[key];
    });

    await dataBaseInfo.save();

    return {isSuccess:true};
}


/**
 * 查询本月提交的企业融资信息
 * 小程序端 回显
 * @param fuHuaQiUscc 孵化器统一信用代码
 * @param uscc 企业统一信用代码
 * @returns {data:{}}  表单
 */
export async function getEnterpriseFinancingByUscc(fuHuaQiUscc:string, uscc:string) {
    /**找到该企业的本月融资数据 */
    const TaskId = taskTool.getTaskId(fuHuaQiUscc);
    let dataBaseInfo = await financingData.findFinancingInfoByTaskIdAndSucc(TaskId, uscc);
    if (!dataBaseInfo || !dataBaseInfo.uscc) throw new BizError(ERRORENUM.未找到数据, `库中不存在${uscc}这个企业的本次融资数据`);

    /**截取返回字段 */
    let data = extractData(splitResultConfig.EnterpriseFinancingInfoConfig, dataBaseInfo, false);

    return {data};
}


/**
 * 根据企业id删除草稿企业
 * 小程序端
 * @param fuHuaQiUscc 孵化器统一信用代码
 * @param uscc 企业统一信用代码
 * @returns {isSuccess:true/false}
 */
export async function deleteEnterpriseFinancingByUscc(fuHuaQiUscc:string, uscc:string) {
    /**找到该企业的本月融资数据 */
    const TaskId = taskTool.getTaskId(fuHuaQiUscc);
    let dataBaseInfo = await financingData.findFinancingInfoByTaskIdAndSucc(TaskId, uscc);
    if (!dataBaseInfo || !dataBaseInfo.uscc) throw new BizError(ERRORENUM.未找到数据, `库中不存在${uscc}这个企业`);

    /**删除限制 */
    if (dataBaseInfo.draftLock) throw new BizError(ERRORENUM.已入库的数据不能删除, uscc);

    await financingData.deleteEnterpriseFinancing(uscc, TaskId);

    return {isSuccess:true};
}


/**
 * 获取本孵化器下的所有企业信息
 * 小程序端  用于下拉框使用
 * @param uscc 孵化器统一信用代码
 * @returns []  name:企业名称, logonAdd:注册地址, operatingAdd:经营地址
 */
export async function getFuHuaQiEnterpriseForSelect(uscc:string) {
    /**找到该孵化器下已经提交入库的企业列表 */
    let enterpriseList = await enterpriseData.findSubmittedEnterpriseListByFuHuaQiUscc(uscc);

    /**拼接返回 */
    let dataList = [];
    enterpriseList.forEach( info => {
        if (info.state != configEnum.FUHUASTATE.迁出) {
            dataList.push({
                name:info.name,
                logonAddress:info.logonAddress, 
                operatingAddress:info.operatingAddress
            });
        }
    });
    
    return dataList;
}

