/**
 * 资质
 */

import moment = require("moment");
import { OPERATIONALDATATYPE, TABLEID, TABLENAME } from "../config/enum/dbEnum";
import { operationalData, selectData } from "../data/operationalData";
import { EnterpriseAwardUpdateConfig, EnterpriseIPRUpdateConfig, EnterprisePatentUpdateConfig, EnterpriseQualificationUpdateConfig } from "../config/eccParam/enterprise";
import { eccFormParam } from "../util/verificationParam";
import { getMySqlMs, randomId } from "../tools/system";
import { BizError } from "../util/bizError";
import { ERRORENUM } from "../config/enum/errorEnum";
import * as enumConfig from "../config/enum/enum";
import { checkChange, extractData } from "../util/piecemeal";
import { changeEnumValue, eccEnumValue } from "../util/verificationEnum";
import { EnterpriseQualificationInfoResConfig } from "../config/splitResult/enterprise";


//------------------------------------------------荣誉奖项

/**
 * 荣誉奖项-列表
 * 回显
 * @param uscc 
 */
export async function enterpriseHonorInfo(eId) {
    let filesList = ["hId", "awardName", "awardingUnit", "awardTime", "awardLevel", "awardImg"];
    let iprList = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.企业荣誉, {eId}, filesList);
    
    let dataList:any[] = [];
    iprList.forEach(item => {
        let {hId, awardName, awardingUnit, awardTime, awardLevel, awardImg} = item;
        dataList.push({
            hId,
            awardName,
            awardingUnit,
            awardTime: moment(awardTime).format("YYYY"),
            awardLevel:awardLevel,
            awardImg:JSON.parse(awardImg),
        });
    });

    //倒序排列
    dataList.sort(function(a, b) {
        return parseInt(moment(b.awardTime).format("YYYY")) - parseInt(moment(a.awardTime).format("YYYY"));
    });

    return {dataList};
}



/**
 * 荣誉奖项-修改
 * @param uscc 企业统一信用代码
 * @param param 表单
 * @returns 
 */
export async function updateEnterpriseHonorInfo(eId, hId, param) {
    const FuncName = "企业修改荣誉奖项"
    eccFormParam(FuncName, EnterpriseAwardUpdateConfig, param );
    param.awardTime = getMySqlMs(param.awardTime);
    param.awardImg = JSON.stringify(param.awardImg);
    
    let filesList = [ "awardName", "awardingUnit", "awardTime", "awardLevel", "awardImg"];
    let iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.企业荣誉, {hId, eId}, filesList)
    if (!iprInfo || !iprInfo.awardName) throw new BizError(ERRORENUM.当前数据不存在); 
   
    await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.企业荣誉, param, {hId, eId});
    
    return {isSuccess:true};
}


/**
 * 荣誉奖项-添加
 * @param uscc 企业统一信用代码
 * @param param 表单
 * @returns 
 */
export async function createEnterpriseHonorInfo(eId, param) {
    const FuncName = "企业创建荣誉奖项"
    eccFormParam(FuncName, EnterpriseAwardUpdateConfig, param );
   
    let addInfo = {
        eId,
        hId:randomId(TABLEID.企业荣誉),
        awardName:param.awardName,
        awardingUnit:param.awardingUnit,
        awardTime:getMySqlMs(param.awardTime),
        awardLevel:param.awardLevel,
        awardImg:JSON.stringify(param.awardImg),
    };

    await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.企业荣誉, addInfo, {});
    
    return {isSuccess:true};
}


/**
 * 荣誉奖项-删除
 * @param param0 
 * @returns 
 */
export async function deleteEnterpriseHonorInfo(eId, hId) {
    let filesList = [ "awardName", "awardingUnit", "awardTime", "awardLevel", "awardImg"];
    let iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.企业荣誉, {hId, eId}, filesList)
    if (!iprInfo || !iprInfo.awardName) throw new BizError(ERRORENUM.当前数据不存在);

    await operationalData(OPERATIONALDATATYPE.删除, TABLENAME.企业荣誉, {}, {hId, eId});
    
    return {isSuccess:true};
}


//=====================================================知识产权

/**
 * 知识产权信息统计
 * @param uscc 
 */
 export async function enterpriseIPRCount(eId) {
    let 商标 = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.商标, {eId}, ["tmId", "eId", "qccId", "regNo", "name", "categoryId", "category", "person", "imageUrl", "flowStatusDesc", "RegDate"]);
    let 作品著作权 = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.作品著作权, {eId}, ["crId", "eId", "owner", "category", "name", "registerNo", "registerDate", "publishDate", "finishDate"]);
    let 软件著作权 = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.软件著作权, {eId}, ["scId", "eId", "category", "publishDate", "versionNo", "registerNo", "registerAperDate", "name", "shortName", "owner", "finishDevelopDate"]);
    let 专利 = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.专利, {eId}, ["patentId", "eId", "category", "kindcode", "applicationNumber", "applicationDate", "publicationNumber", "publicationDate", "legalStatusDesc", "title", "agency", "kindCodeDesc", "IPCDesc", "inventorStringList", "assigneestringList"]);

    let 知识产权Map:any = {};
    知识产权Map[enumConfig.IPRALLTYPE.商标信息] = 商标;
    知识产权Map[enumConfig.IPRALLTYPE.作品著作权] = 作品著作权;
    知识产权Map[enumConfig.IPRALLTYPE.软件著作权] = 软件著作权;
    知识产权Map[enumConfig.IPRALLTYPE.外观设计专利] = [];
    知识产权Map[enumConfig.IPRALLTYPE.实用新型专利] = [];
    知识产权Map[enumConfig.IPRALLTYPE.发明专利] = [];
    
    //1.发明公布，2.发明授权，3.实用新型，4.外观设计
    专利.forEach( info => {
        let {kindcode} = info;
        if (kindcode == enumConfig.KUNDCODE.发明公布) {
            知识产权Map[enumConfig.IPRALLTYPE.发明专利].push(info);
        } else if (kindcode == enumConfig.KUNDCODE.发明授权) {
            知识产权Map[enumConfig.IPRALLTYPE.发明专利].push(info);
        } else if (kindcode == enumConfig.KUNDCODE.实用新型) {
            知识产权Map[enumConfig.IPRALLTYPE.实用新型专利].push(info);
        } else if (kindcode == enumConfig.KUNDCODE.外观设计) {
            知识产权Map[enumConfig.IPRALLTYPE.外观设计专利].push(info);
        }
    })

    let iprYearInfo = {};
    let iprConf = "";
    for (let key in 知识产权Map) {
        if (parseInt(key) == enumConfig.IPRALLTYPE.商标信息) iprConf = "RegDate";
        if (parseInt(key) == enumConfig.IPRALLTYPE.作品著作权) iprConf = "registerDate";
        if (parseInt(key) == enumConfig.IPRALLTYPE.软件著作权) iprConf = "registerAperDate";
        if (parseInt(key) == enumConfig.IPRALLTYPE.外观设计专利 || parseInt(key) == enumConfig.IPRALLTYPE.商标信息 || parseInt(key) == enumConfig.IPRALLTYPE.商标信息) iprConf = "publicationDate";

        let 知识产权Data = 知识产权Map[key];
        知识产权Data.forEach(patents => {
            let year = moment(patents[iprConf]).format("YYYY");
            if (!iprYearInfo[year]) iprYearInfo[year] = 0;
            iprYearInfo[year] += 1;
        })
    }

    let dataList = [];
    for (let key in iprYearInfo) {
        dataList.push({
            year:key,
            count:iprYearInfo[key]
        })
    }

    //倒序
    dataList.sort(function(a, b) {
        return parseInt(b.year) - parseInt(a.year);
    });

    return {dataList};
}


/**
 * 知识产权信息-列表
 * @param uscc 
 */
export async function enterpriseIPRList(eId, iprType) {
    let iprList = [];
    let iprColumn = [];
    let iprConf = [];
    if (iprType == enumConfig.IPRALLTYPE.商标信息) {
        iprColumn = ["tmId", "name", "RegDate", "imageUrl"];
        iprList = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.商标, {eId}, iprColumn);
        iprConf = ["tmId", "name", "RegDate", "imageUrl"]
    }

    if (iprType == enumConfig.IPRALLTYPE.作品著作权) {
        iprColumn = ["crId", "name", "registerDate", "iprUrl"];
        iprList = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.作品著作权, {eId}, iprColumn);
        iprConf = ["crId", "name", "registerDate", "iprUrl"]
    }

    if (iprType == enumConfig.IPRALLTYPE.软件著作权) {
        iprColumn = ["scId", "registerAperDate", "name", "iprUrl"];
        iprList = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.软件著作权, {eId}, iprColumn);
        iprConf = ["scId", "name", "registerAperDate", "iprUrl"]
    }

    if (iprType == enumConfig.IPRALLTYPE.外观设计专利) {
        iprColumn = ["patentId", "publicationDate", "title", "iprUrl"];
        iprList = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.专利, {eId, kindcode:enumConfig.KUNDCODE.外观设计}, iprColumn);
        iprConf = ["patentId", "title", "publicationDate", "iprUrl"]
    }

    if (iprType == enumConfig.IPRALLTYPE.实用新型专利) {
        iprColumn = ["patentId", "eId", "category", "kindcode", "applicationNumber", "applicationDate", "publicationNumber", "publicationDate", "legalStatusDesc", "title", "agency", "kindCodeDesc", "IPCDesc", "inventorStringList", "assigneestringList", "iprUrl"];
        iprList = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.专利, {eId, kindcode:enumConfig.KUNDCODE.实用新型}, iprColumn);
        iprConf = ["patentId", "title", "publicationDate", "iprUrl"]
    }

    if (iprType == enumConfig.IPRALLTYPE.发明专利) {
        iprColumn = ["patentId", "eId", "category", "kindcode", "applicationNumber", "applicationDate", "publicationNumber", "publicationDate", "legalStatusDesc", "title", "agency", "kindCodeDesc", "IPCDesc", "inventorStringList", "assigneestringList", "iprUrl"];
        iprList = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.专利, {eId, kindcode:{"%between%":[enumConfig.KUNDCODE.发明公布, enumConfig.KUNDCODE.发明授权]}}, iprColumn);
        iprConf = ["patentId", "title", "publicationDate", "iprUrl"]
    }

    let dataList:any = [];
    if (iprList && iprList.length) {
        iprList.forEach( info => {
            let iprTime = "";
            if (info[iprConf[2]]) iprTime = moment(info[iprConf[2]]).format("YYYY-MM-DD");
            dataList.push({
                iprId:info[iprConf[0]],
                iprName:info[iprConf[1]],
                iprTime,
                timeNumber:new Date(info[iprConf[2]]).valueOf(),
                iprUrl:JSON.parse(info[iprConf[3]]),
            })
        })
    }

    //倒序
    dataList.sort(function(a, b) {
        return b.timeNumber - a.timeNumber;
    });

    return {dataList};
}


/**
 * 知识产权信息-回显
 * @param uscc 
 */
export async function enterpriseIPRInfo(eId, iprType, iprId) {
    let iprInfo = {};
    let iprColumn = [];
    let iprConf = [];
    if (iprType == enumConfig.IPRALLTYPE.商标信息) {
        iprColumn = ["tmId", "name", "RegDate", "imageUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.商标, {eId, tmId:iprId}, iprColumn);
        iprConf = ["tmId", "name", "RegDate", "imageUrl"]
    }

    if (iprType == enumConfig.IPRALLTYPE.作品著作权) {
        iprColumn = ["crId", "name", "registerDate", "iprUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.作品著作权, {eId, crId:iprId}, iprColumn);
        iprConf = ["crId", "name", "registerDate", "iprUrl"]
    }

    if (iprType == enumConfig.IPRALLTYPE.软件著作权) {
        iprColumn = ["scId", "registerAperDate", "name", "iprUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.软件著作权, {eId, scId:iprId}, iprColumn);
        iprConf = ["scId", "name", "registerAperDate", "iprUrl"]
    }

    if (iprType == enumConfig.IPRALLTYPE.外观设计专利 || iprType == enumConfig.IPRALLTYPE.实用新型专利 || iprType == enumConfig.IPRALLTYPE.发明专利) {
        iprColumn = ["patentId", "publicationDate", "title", "iprUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.专利, {eId, patentId:iprId}, iprColumn);
        iprConf = ["patentId", "title", "publicationDate", "iprUrl"]
    }

    let dataInfo = {};
    if (Object.keys(iprInfo).length) {
        let iprTime = "";
        if (iprInfo[iprConf[2]]) iprTime = moment(iprInfo[iprConf[2]]).format("YYYY-MM-DD");
        dataInfo = {
            iprId:iprInfo[iprConf[0]],
            iprName:iprInfo[iprConf[1]],
            iprTime,
            iprUrl:JSON.parse(iprInfo[iprConf[3]]),
        }
    }
    
    return {dataInfo};
}


/**
 * 知识产权信息-添加
 * @param eId 
 * @param param 
 * @returns 
 */
export async function createIPRInfo(eId, iprType, param) {
    const FuncName = `企业创建知识产权-${changeEnumValue(enumConfig.IPRALLTYPE, iprType)}`
    eccFormParam(FuncName, EnterpriseIPRUpdateConfig, param );

    let addInfo:any = {};
    if (iprType == enumConfig.IPRALLTYPE.商标信息) {
        addInfo = {
            eId,
            tmId:randomId(TABLEID.商标),
            name:param.iprName,
            RegDate:getMySqlMs(param.iprTime), //注册公告时间
            imageUrl:JSON.stringify(param.iprUrl),
        }
        await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.商标, addInfo, {});
    }

    if (iprType == enumConfig.IPRALLTYPE.作品著作权) {
        addInfo = {
            eId,
            crId:randomId(TABLEID.作品著作权),
            name:param.iprName,
            registerDate:getMySqlMs(param.iprTime), //登记日期
            iprUrl:JSON.stringify(param.iprUrl),
        }
        await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.作品著作权, addInfo, {});
    }

    if (iprType == enumConfig.IPRALLTYPE.软件著作权) {
        addInfo = {
            eId,
            scId:randomId(TABLEID.软件著作权),
            name:param.iprName,
            registerAperDate:getMySqlMs(param.iprTime), //登记批准日期
            iprUrl:JSON.stringify(param.iprUrl),
        }
        await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.软件著作权, addInfo, {});
    }

    if (iprType == enumConfig.IPRALLTYPE.外观设计专利) {
        addInfo = {
            eId,
            patentId:randomId(TABLEID.专利),
            kindcode:enumConfig.KUNDCODE.外观设计,
            title:param.iprName,
            publicationDate:getMySqlMs(param.iprTime), //公开日期
            iprUrl:JSON.stringify(param.iprUrl),
        }
        await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.专利, addInfo, {});
    }

    if (iprType == enumConfig.IPRALLTYPE.实用新型专利) {
        addInfo = {
            eId,
            patentId:randomId(TABLEID.专利),
            kindcode:enumConfig.KUNDCODE.实用新型,
            title:param.iprName,
            publicationDate:getMySqlMs(param.iprTime), //公开日期
            iprUrl:JSON.stringify(param.iprUrl),
        }
        await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.专利, addInfo, {});
    }

    if (iprType == enumConfig.IPRALLTYPE.发明专利) {
        addInfo = {
            eId,
            patentId:randomId(TABLEID.专利),
            kindcode:enumConfig.KUNDCODE.发明公布,
            title:param.iprName,
            publicationDate:getMySqlMs(param.iprTime), //公开日期
            iprUrl:JSON.stringify(param.iprUrl),
        }
        await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.专利, addInfo, {});
    }
    
    return {isSuccess:true};
}


/**
 * 知识产权信息-修改
 * @param eId 
 * @param param 
 * @returns 
 */
export async function enterpriseIPRUpdate(eId, iprType, iprId, param) {
    const FuncName = `企业修改知识产权-${changeEnumValue(enumConfig.IPRALLTYPE, iprType)}`;
    eccFormParam(FuncName, EnterpriseIPRUpdateConfig, param );

    let iprInfo = {};
    let iprColumn = [];
    
    let updateInfo:any = {};
    if (iprType == enumConfig.IPRALLTYPE.商标信息) {
        iprColumn = ["tmId", "name", "RegDate", "imageUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.商标, {eId, tmId:iprId}, iprColumn);
        if (!iprInfo) throw new BizError(ERRORENUM.当前数据不存在);

        updateInfo = {
            name:param.iprName,
            RegDate:getMySqlMs(param.iprTime), //注册公告时间
            imageUrl:JSON.stringify(param.iprUrl),
        }
        await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.商标, updateInfo, {eId, tmId:iprId});
    }

    if (iprType == enumConfig.IPRALLTYPE.作品著作权) {
        iprColumn = ["crId", "name", "registerDate", "iprUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.作品著作权, {eId, crId:iprId}, iprColumn);
        if (!iprInfo) throw new BizError(ERRORENUM.当前数据不存在);
        
        updateInfo = {
            name:param.iprName,
            registerDate:getMySqlMs(param.iprTime), //创作完成日期
            iprUrl:JSON.stringify(param.iprUrl),
        }
        await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.作品著作权, updateInfo, {eId, crId:iprId});
    }

    if (iprType == enumConfig.IPRALLTYPE.软件著作权) {
        iprColumn = ["scId", "registerAperDate", "name", "iprUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.软件著作权, {eId, scId:iprId}, iprColumn);
        if (!iprInfo) throw new BizError(ERRORENUM.当前数据不存在);
        
        updateInfo = {
            name:param.iprName,
            registerAperDate:getMySqlMs(param.iprTime), //发布日期
            iprUrl:JSON.stringify(param.iprUrl),
        }
        await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.软件著作权, updateInfo, {eId, scId:iprId});
    }

    if (iprType == enumConfig.IPRALLTYPE.外观设计专利) {
        iprColumn = ["patentId", "publicationDate", "title", "iprUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.专利, {eId, patentId:iprId}, iprColumn);
        if (!iprInfo) throw new BizError(ERRORENUM.当前数据不存在);

        updateInfo = {
            kindcode:enumConfig.KUNDCODE.外观设计,
            title:param.iprName,
            publicationDate:getMySqlMs(param.iprTime), //公开日期
            iprUrl:JSON.stringify(param.iprUrl),
        }
        await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.专利, updateInfo, {eId, patentId:iprId});
    }

    if (iprType == enumConfig.IPRALLTYPE.实用新型专利) {
        iprColumn = ["patentId", "publicationDate", "title", "iprUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.专利, {eId, patentId:iprId}, iprColumn);
        if (!iprInfo) throw new BizError(ERRORENUM.当前数据不存在);

        updateInfo = {
            kindcode:enumConfig.KUNDCODE.实用新型,
            title:param.iprName,
            publicationDate:getMySqlMs(param.iprTime), //公开日期
            iprUrl:JSON.stringify(param.iprUrl),
        }
        await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.专利, updateInfo, {eId, patentId:iprId});
    }

    if (iprType == enumConfig.IPRALLTYPE.发明专利) {
        iprColumn = ["patentId", "publicationDate", "title", "iprUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.专利, {eId, patentId:iprId}, iprColumn);
        if (!iprInfo) throw new BizError(ERRORENUM.当前数据不存在);

        updateInfo = {
            kindcode:enumConfig.KUNDCODE.发明公布,
            title:param.iprName,
            publicationDate:getMySqlMs(param.iprTime), //公开日期
            iprUrl:JSON.stringify(param.iprUrl),
        }
        await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.专利, updateInfo, {eId, patentId:iprId});
    }

    return {isSuccess:true};
}


/**
 * 知识产权信息-删除
 * @param eId 
 * @param iprType 
 * @param iprId 
 * @returns 
 */
export async function enterpriseIPRDel(eId, iprType, iprId) {
    let iprInfo = {};
    let iprColumn = [];
    
    if (iprType == enumConfig.IPRALLTYPE.商标信息) {
        iprColumn = ["tmId", "name", "RegDate", "imageUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.商标, {eId, tmId:iprId}, iprColumn);
        if (!iprInfo) throw new BizError(ERRORENUM.当前数据不存在);

        await operationalData(OPERATIONALDATATYPE.删除, TABLENAME.商标, {}, {eId, tmId:iprId});
    }

    if (iprType == enumConfig.IPRALLTYPE.作品著作权) {
        iprColumn = ["crId", "name", "registerDate", "iprUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.作品著作权, {eId, crId:iprId}, iprColumn);
        if (!iprInfo) throw new BizError(ERRORENUM.当前数据不存在);
        
        await operationalData(OPERATIONALDATATYPE.删除, TABLENAME.作品著作权, {}, {eId, crId:iprId});
    }

    if (iprType == enumConfig.IPRALLTYPE.软件著作权) {
        iprColumn = ["scId", "registerAperDate", "name", "iprUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.软件著作权, {eId, scId:iprId}, iprColumn);
        if (!iprInfo) throw new BizError(ERRORENUM.当前数据不存在);
        
        await operationalData(OPERATIONALDATATYPE.删除, TABLENAME.软件著作权, {}, {eId, scId:iprId});
    }

    if (iprType == enumConfig.IPRALLTYPE.外观设计专利) {
        iprColumn = ["patentId", "publicationDate", "title", "iprUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.专利, {eId, patentId:iprId}, iprColumn);
        if (!iprInfo) throw new BizError(ERRORENUM.当前数据不存在);

        await operationalData(OPERATIONALDATATYPE.删除, TABLENAME.专利, {}, {eId, patentId:iprId});
    }

    if (iprType == enumConfig.IPRALLTYPE.实用新型专利) {
        iprColumn = ["patentId", "publicationDate", "title", "iprUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.专利, {eId, patentId:iprId}, iprColumn);
        if (!iprInfo) throw new BizError(ERRORENUM.当前数据不存在);

        await operationalData(OPERATIONALDATATYPE.删除, TABLENAME.专利, {}, {eId, patentId:iprId});
    }

    if (iprType == enumConfig.IPRALLTYPE.发明专利) {
        iprColumn = ["patentId", "publicationDate", "title", "iprUrl"];
        iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.专利, {eId, patentId:iprId}, iprColumn);
        if (!iprInfo) throw new BizError(ERRORENUM.当前数据不存在);

        await operationalData(OPERATIONALDATATYPE.删除, TABLENAME.专利, {}, {eId, patentId:iprId});
    }

    return {isSuccess:true};
}



//====================================专利

// /**
//  * 专利信息-列表
//  * 回显
//  * @param uscc 
//  */
// export async function enterprisePatentInfo(eId:string) {
//     let filesList = ["pId", "year", "alienPatent", "classIPatent", "secondClassPatent", "thirdPentent"];
//     let iprList = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.企业专利表, {eId}, filesList);

//     let dataList:any[] = [];
//     iprList.forEach(item => {
//         let {pId, year, number, alienPatent, classIPatent, secondClassPatent, thirdPentent} = item;
//         dataList.push({
//             pId,
//             year,
//             yearStr:moment(year).format("YYYY"),
//             number,
//             alienPatent,
//             classIPatent,
//             secondClassPatent,
//             thirdPentent
//         });
//     });

//     return {dataList};
// }



// /**
//  * 专利信息-修改
//  * @param uscc 企业统一信用代码
//  * @param param 表单
//  * @returns 
//  */
// export async function updateEnterprisePatentInfo(eId, pId, param) {
//     const FuncName = "企业修改知识产权"
//     eccFormParam(FuncName, EnterprisePatentUpdateConfig, param );

//     let filesList = ["pId", "year", "alienPatent", "classIPatent", "secondClassPatent", "thirdPentent"];
//     let iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.企业专利表, {pId, eId}, filesList)
//     if (!iprInfo || !iprInfo.year) throw new BizError(ERRORENUM.当前数据不存在); 
//     let disInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.企业专利表, {year:getMySqlMs(param.year), eId}, ["eId", "pId"])
//     if (disInfo && disInfo.eId && disInfo.pId != pId) throw new BizError(ERRORENUM.存在重复的年份数据); 
//     /**修改字段 */
//     let changeList = checkChange(param, iprInfo);
//     if ( !changeList.length ) return {isSuccess:true};
   
//     await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.企业专利表, param, {pId, eId});
    
//     return {isSuccess:true};
// }


// /**
//  * 专利信息-添加
//  * @param uscc 企业统一信用代码
//  * @param param 表单
//  * @returns 
//  */
// export async function createEnterprisePatentInfo(eId, param) {
//     const FuncName = "企业创建知识产权"
//     eccFormParam(FuncName, EnterprisePatentUpdateConfig, param );
   
//     let disInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.企业专利表, {year:getMySqlMs(param.year), eId}, ["eId"])
//     if (disInfo && disInfo.eId) throw new BizError(ERRORENUM.存在重复的年份数据); 

//     let addInfo = {
//         eId,
//         year:getMySqlMs(param.year),
//         alienPatent:param.alienPatent,
//         classIPatent:param.classIPatent,
//         secondClassPatent:param.secondClassPatent,
//         thirdPentent:param.thirdPentent,
//         pId:randomId(TABLEID.企业专利表)
//     };

//     await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.企业专利表, addInfo, {});
    
//     return {isSuccess:true};
// }


// /**
//  * 专利信息-删除
//  * @param param0 
//  * @returns 
//  */
// export async function deleteEnterprisePatentInfo(eId, pId) {
//     let filesList = ["pId", "year", "alienPatent", "classIPatent", "secondClassPatent", "thirdPentent"];
//     let iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.企业专利表, {pId, eId}, filesList)
//     if (!iprInfo || !iprInfo.year) throw new BizError(ERRORENUM.当前数据不存在);

//     await operationalData(OPERATIONALDATATYPE.删除, TABLENAME.企业专利表, {}, {pId, eId});
    
//     return {isSuccess:true};
// }



//====================================资质认证

/**
 * 资质信息-回显
 * 回显
 * @param uscc 
 */
export async function enterpriseQualificationInfo(eId) {
    let qualificationData = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.企业资质, {eId}, []);
    
    // let qualificationList = [];
    // if (qualificationData.goPublicSector) {
    //     let qualification = JSON.parse(qualificationData.goPublicSector);
    //     for(let i = 0; i < qualification.length; i++){
    //         let qualificationType = changeEnumValue(enumConfig.LISTINGSITUATION, qualification[i]);
    //         qualificationList.push(qualificationType);
    //     }
    // }

    let result = {
        kxState: 0,
        kxTime: "",
        kxNumber: "",
        kxImg: [],

        zjtxState: 0,
        zjtxTime: "",
        zjtxImg: [],

        // xjrState: 0,
        // xjrTime: "",
        // xjrImg: [],

        xjrPyState: 0,
        xjrPyTime: "",
        xjrPyImg: [],

        gxjsState: 0,
        gaoXinJiShuTime: "",
        gaoXinJiShuImg: [],

        cxState: 0,
        cxTime: "",
        cxImg: [],

        // listedState: 0,
        // goPublicSector: [],
        // goPublicTime: "",
        // other: "",
    };
    if (qualificationData.qId) {
        result = {
            kxState: qualificationData.kxState || 0,//新增科小状态
            kxTime: qualificationData.kxTime,
            kxNumber: qualificationData.kxNumber || '',//新增科小编号
            kxImg: qualificationData.kxImg ? JSON.parse(qualificationData.kxImg) :[],//新增科编图片
            zjtxState: qualificationData.zjtxState || 0,
            zjtxTime: qualificationData.zjtxTime,
            zjtxImg: qualificationData.zjtxImg ? JSON.parse(qualificationData.zjtxImg) : [],//新增专精特新证书
            // xjrState: qualificationData.xjrState || 0,
            // xjrTime: qualificationData.xjrTime,
            // xjrImg: qualificationData.xjrImg ? JSON.parse(qualificationData.xjrImg) : [],//新增小巨人证书
            xjrPyState: qualificationData.xjrPyState || 0,
            xjrPyTime: qualificationData.xjrPyTime,
            xjrPyImg: qualificationData.zjtxImg ? JSON.parse(qualificationData.xjrPyImg) : [],//新增小巨人培育证书
            gxjsState: qualificationData.gxjsState || 0,
            gaoXinJiShuTime: qualificationData.gaoXinJiShuTime,
            gaoXinJiShuImg: qualificationData.gaoXinJiShuImg ? JSON.parse(qualificationData.gaoXinJiShuImg) : [],//新增高新技术证书
            cxState: qualificationData.cxState || 0,//新增创新状态
            cxTime: qualificationData.cxTime,
            cxImg: qualificationData.cxImg ? JSON.parse(qualificationData.cxImg) :[],//新增科编图片
            // listedState: qualificationData.listedState || 0,
            // goPublicSector: JSON.parse(qualificationData.goPublicSector) || [],
            // goPublicTime: qualificationData.goPublicTime,
            // other: qualificationData.other || '',//新增其他
        };
    }

    return result;
}


/**
 * 资质信息-修改
 * @param uscc 企业统一信用代码
 * @param param 表单
 * @returns 
 */
export async function updateEnterpriseQualificationInfo(eId, param) {
    const FuncName = "企业修改资质信息";
   
    eccFormParam(FuncName, EnterpriseQualificationUpdateConfig, param );

    if (param.goPublicSector) eccEnumValue(FuncName, 'goPublicSector', enumConfig.LISTINGSITUATION, param.goPublicSector);
        
    let filesList = ["eId", "kxState","kxTime","kxNumber","kxImg", "zjtxState","zjtxTime","zjtxImg",
        "xjrPyState", "xjrPyTime","xjrPyImg",
        "gxjsState","gaoXinJiShuTime","gaoXinJiShuImg","cxState", "cxTime","cxImg"];

    let iprInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.企业资质, {eId}, filesList)

    param.kxNumber = param.kxNumber ? param.kxNumber : null;
    // param.goPublicSector = param.goPublicSector ? JSON.stringify(param.goPublicSector) : "[]";
    param.kxImg = param.kxImg ? JSON.stringify(param.kxImg) : "[]";
    param.zjtxImg = param.zjtxImg ? JSON.stringify(param.zjtxImg) : "[]";
    param.xjrImg = param.xjrImg ? JSON.stringify(param.xjrImg) : "[]";
    param.xjrPyImg = param.xjrPyImg ? JSON.stringify(param.xjrPyImg) : "[]";
    param.gaoXinJiShuImg = param.gaoXinJiShuImg ? JSON.stringify(param.gaoXinJiShuImg) : "[]";
    param.cxImg = param.cxImg ? JSON.stringify(param.cxImg) : "[]";
    param.kxTime = param.kxTime ? getMySqlMs(param.kxTime) : null;
    param.zjtxTime = param.zjtxTime ? getMySqlMs(param.zjtxTime) : null;
    param.xjrTime = param.xjrTime ? getMySqlMs(param.xjrTime) : null;
    param.xjrPyTime = param.xjrPyTime ? getMySqlMs(param.xjrPyTime) : null;
    param.gaoXinJiShuTime = param.gaoXinJiShuTime ? getMySqlMs(param.gaoXinJiShuTime) : null;
    param.cxTime = param.cxTime ? getMySqlMs(param.cxTime) : null;
    // param.goPublicTime = param.goPublicTime ? getMySqlMs(param.goPublicTime) : null;

    if (!iprInfo || !iprInfo.eId) {
        param.qId = randomId(TABLEID.企业资质);
        param.eId = eId;
        await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.企业资质, param, {});
    } else {
        if (param && Object.keys(param).length > 0) {
            await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.企业资质, param, {eId});
        }
    }
    
    return {isSuccess:true};
}





