import { Op, Sequelize } from "sequelize";
import { ERRORENUM } from "../config/errorEnum";
import { mysqlModelMap } from "../model/sqlModelBind";
import { BizError } from "../util/bizError";


/**
 * where条件查询参数
 * @param param 
 * %or%: 或者条件 [{"列名": 条件}, {"列名":条件}]
 * %like%：模糊查询 {列名: {"%like%": }}
 * %gt%：大于 {列名: {"%gt%": }}
 * %gte%：大于等于 {列名: {"%gte%": }}
 * %lt%：小于 {列名: {"%lt%": }}
 * %lte%：小于等于 {列名: {"%lte%": }}
 * %between%：查询范围内数据 {列名: {"%between%": ["开始参数", "结束参数"]}} ---BETWEEN 开始参数 AND 结束参数 列>开始参数 and 列<结束参数
 * %notBetween%：查询不在范围内数据 {列名: {"%notBetween%": ["开始参数", "结束参数"]}} ---NOT BETWEEN 开始参数 AND 结束参数
 * %orderDesc%: order by DESC {"%orderDesc%": "列名"}
 * %limit%: {"%limit%": 数量}
 * %literal%: 拼接查询sql条件，多表联查使用时记得指定子表字段 {"%literal%": `sql语句`} 例如：enterprise_fuhua.startTime BETWEEN "" and "" and (state != "" or state == "")
 * @param column 
 * @returns 
 */
// function analysisParamToWhere(param, column) {
//     let where = {};
//     let order = [];
//     let group = "";
//     let literal = "";
//     let limit = 0;
//     for (let key in param) {  
//         if (key == "%or%") {
//             where[Op.or] = param["%or%"];
//         } else if (typeof param[key] == "object") {
//             where[key] = {};
//             for (let whereKey in param[key]){
//                 switch(whereKey) {
//                     case "%like%":
//                         where[key][Op.like] = `%${param[key]["%like%"]}%`;
//                     break;
//                     case "%gt%":
//                         where[key][Op.gt] = param[key]["%gt%"];
//                     break;
//                     case "%gte%":
//                         where[key][Op.gte] = param[key]["%gte%"];
//                     break;
//                     case "%lt%":
//                         where[key][Op.lt] = param[key]["%lt%"];
//                     break;
//                     case "%lte%":
//                         where[key][Op.lte] = param[key]["%lte%"];
//                     break;
//                     case "%between%":
//                         where[key][Op.between] = param[key]["%between%"];
//                     break;
//                     case "%notBetween%":
//                         where[key][Op.notBetween] = param[key]["%notBetween%"];
//                     break;
//                     case "%in%"://在范围内数据
//                         where[key][Op.in] = param[key]["%in%"];
//                     break;
//                     case "%notIn%"://不在范围内数据
//                         where[key][Op.notIn] = param[key]["%notIn%"];
//                     break;
//                     case "%ne%"://不等于
//                         where[key][Op.ne] = param[key]["%ne%"];
//                     break;
//                     case "%regexp%":
//                         where[key][Op.regexp] = param[key]["%regexp%"];
//                     break;
//                 }
//             }
//         } else {
//             switch (key) {
//                 case "%orderDesc%":
//                     order = [[Sequelize.col(param[key]), "DESC"]];
//                 break;
//                 case "%orderAsc%":
//                     order = [[Sequelize.col(param[key]), "ASC"]];
//                 break;
//                 case "%limit%":
//                     limit = param[key];
//                 break;
//                 case "%group%":
//                     group = param[key];
//                 break;
//                 case "%literal%":
//                     literal = param["%literal%"];
//                 break;
//                 default: where[key] = param[key];
//             }
//         }
//     }

//     let selectParam:any = {where};
//     if (column && column.length) selectParam.attributes = column;
//     if (order && order.length) selectParam.order = order;
//     if (limit) selectParam.limit = limit;
//     if (group) selectParam.group = group;
//     if (literal) selectParam.where = Sequelize.literal(literal);
//     return selectParam;
// }
function analysisParamToWhere(param, column) {
    let where = {};
    let order = [];
    let group = "";
    let literal = "";
    let limit = 0;
    
    for (let key in param) {  
        // 处理 %or% 条件
        if (key == "%or%") {
            // 递归处理 or 中的每个条件
            const orConditions = param["%or%"].map(condition => {
                return processCondition(condition);
            });
            // 过滤掉空对象
            const validConditions = orConditions.filter(cond => 
                cond && Object.keys(cond).length > 0
            );
            if (validConditions.length > 0) {
                where[Op.or] = validConditions;
            }
        } else {
            // 处理其他条件
            const result = processCondition({[key]: param[key]});
            // 合并结果，跳过特殊键
            if (result && Object.keys(result).length > 0) {
                Object.assign(where, result);
            }
        }
    }

    let selectParam:any = {where};
    if (column && column.length) selectParam.attributes = column;
    if (order && order.length) selectParam.order = order;
    if (limit) selectParam.limit = limit;
    if (group) selectParam.group = group;
    if (literal) selectParam.where = Sequelize.literal(literal);
    return selectParam;

    // 辅助函数：处理单个条件
    function processCondition(condition) {
        const result = {};
        
        for (let key in condition) {
            const value = condition[key];
            
            // 处理 order/group/limit/literal 等特殊键
            switch(key) {
                case "%orderDesc%":
                    order = [[Sequelize.col(value), "DESC"]];
                    break;
                case "%orderAsc%":
                    order = [[Sequelize.col(value), "ASC"]];
                    break;
                case "%limit%":
                    limit = value;
                    break;
                case "%group%":
                    group = value;
                    break;
                case "%literal%":
                    literal = value;
                    break;
                default:
                    // 处理操作符对象
                    if (typeof value === "object" && value !== null && !Array.isArray(value)) {
                        const operatorObj = {};
                        let hasOperator = false;
                        
                        for (let opKey in value) {
                            const opValue = value[opKey];
                            
                            switch(opKey) {
                                case "%like%":
                                    operatorObj[Op.like] = `%${opValue}%`;
                                    hasOperator = true;
                                    break;
                                case "%gt%":
                                    operatorObj[Op.gt] = opValue;
                                    hasOperator = true;
                                    break;
                                case "%gte%":
                                    operatorObj[Op.gte] = opValue;
                                    hasOperator = true;
                                    break;
                                case "%lt%":
                                    operatorObj[Op.lt] = opValue;
                                    hasOperator = true;
                                    break;
                                case "%lte%":
                                    operatorObj[Op.lte] = opValue;
                                    hasOperator = true;
                                    break;
                                case "%between%":
                                    operatorObj[Op.between] = opValue;
                                    hasOperator = true;
                                    break;
                                case "%notBetween%":
                                    operatorObj[Op.notBetween] = opValue;
                                    hasOperator = true;
                                    break;
                                case "%in%":
                                    operatorObj[Op.in] = opValue;
                                    hasOperator = true;
                                    break;
                                case "%notIn%":
                                    operatorObj[Op.notIn] = opValue;
                                    hasOperator = true;
                                    break;
                                case "%ne%":
                                    operatorObj[Op.ne] = opValue;
                                    hasOperator = true;
                                    break;
                                case "%regexp%":
                                    operatorObj[Op.regexp] = opValue;
                                    hasOperator = true;
                                    break;
                            }
                        }
                        
                        if (hasOperator) {
                            result[key] = operatorObj;
                        }
                    } else {
                        // 直接赋值（包括字符串 "[]"）
                        result[key] = value;
                    }
            }
        }
        
        return result;
    }
}


/**
 * 查询单个数据
 * @param tableModel 表对象
 * @param param 
 * @returns 
 */
export async function selectOneDataByParam(tableModel, param, column) {
    let selectParam = analysisParamToWhere(param, column);
    let data = await tableModel.findOne(selectParam);
    data = data || {};
    return {data};
}


/**
 * 查询多个数据
 * @param tableName 表对象
 * @param param 
 * @returns 
 */
export async function selectDataListByParam(tableModel, param, column) {
    let selectParam = analysisParamToWhere(param, column);
    let data = await tableModel.findAll(selectParam);
    return {data};
}


/**
 * 分页查询
 * @param tableModel 
 * @param param 
 * @param pageNumber 
 * @param pageSize 
 * @returns 
 */
export async function selectDataListToPageByParam(tableModel, param, column, pageNumber:number, pageSize:number) {
    let selectParam:any = analysisParamToWhere(param, column);
    selectParam.limit = pageSize || 10;
    selectParam.offset = (pageNumber-1)*10;
    let data = await tableModel.findAll(selectParam);
    return {data};
}

export async function selectDataCountByParam(tableModel, param) {
    let selectParam:any = analysisParamToWhere(param, []);
    let data = await tableModel.count(selectParam);
    return {data};
}


export async function associationSelect(tableName:string, param) {
    let model = mysqlModelMap[tableName];
    if (!model) throw new BizError(ERRORENUM.不存在表);

    let data = await model.aggragateData(param);

    return {data};
}


/**
 * 多表联查 列表
 * @param tableModel 
 * @param includeConf {"表名":["",""] } 
 * @param param 
 * @param column 
 * @returns 
 */
export async function selectDataToTableAssociation(tableModel, includeConf, param, column) {
    let include = [];
    for (let tableName in includeConf) {
        if (!mysqlModelMap[tableName]) throw new BizError(ERRORENUM.不存在表, `尝试进行多表联查，但是不存在${tableName}`);
        let {where, column} = includeConf[tableName];
        let includeInfomation = analysisParamToWhere(where, column);
        includeInfomation.model = mysqlModelMap[tableName];
        include.push(includeInfomation);
    }

    let selectParam:any = analysisParamToWhere(param, column);
    selectParam.include = include;

    let data = await tableModel.findAll(selectParam);
    return {data};
}



/**
 * 多表联查 分页
 * @param tableModel 
 * @param includeConf {"表名":["",""] } 
 * @param param 
 * @param column 
 * @returns 
 */
export async function selectDataToTableAssociationToPage(tableModel, includeConf, param, column, pageNumber:number, pageSize:number) {
    let include = [];
    for (let tableName in includeConf) {
        if (!mysqlModelMap[tableName]) throw new BizError(ERRORENUM.不存在表, `尝试进行多表联查，但是不存在${tableName}`);
        let {where, column, required} = includeConf[tableName];
        let includeInfomation = analysisParamToWhere(where, column);
        includeInfomation.model = mysqlModelMap[tableName];
        // if (required != undefined) {
        //     includeInfomation.required = required;
        // }
        include.push(includeInfomation);
    }

    let selectParam:any = analysisParamToWhere(param, column);
    selectParam.include = include;
    selectParam.limit = pageSize || 10;
    selectParam.offset = (pageNumber-1)*10;

    let data = await tableModel.findAll(selectParam);
    return {data};
}


/**
 * 多表查询单个
 * @param tableModel 
 * @param includeConf 
 * @param param 
 * @param column 
 * @returns 
 */
export async function selectOneDataToTableAssociation(tableModel, includeConf, param, column) {
    let include = [];
    for (let tableName in includeConf) {
        if (!mysqlModelMap[tableName]) throw new BizError(ERRORENUM.不存在表, `尝试进行多表联查，但是不存在${tableName}`);
        let {where, column} = includeConf[tableName];
        let includeInfomation = analysisParamToWhere(where, column);
        includeInfomation.model = mysqlModelMap[tableName];
        include.push(includeInfomation);
    }

    let selectParam:any = analysisParamToWhere(param, column);
    selectParam.include = include;

    let data = await tableModel.findOne(selectParam);
    data = data || {};
    return {data};
}


