
import { OPERATIONALDATATYPE } from "../config/enum/dbEnum";
import { ERRORENUM } from "../config/enum/errorEnum";
import { systemConfig } from "../config/serverConfig";
import { BizError } from "../util/bizError";
import { post } from "../util/request";


/**
 * 操作数据库 新增  修改  删除
 * @param url url地址 根据枚举获取值
 * @param tableName 表名
 * @param data 数据
 * @param param 条件
 */
export async function operationalData(url:string, tableName:string, data, param) {
    let header = {table:tableName, sign:systemConfig.dbSign};
    let queray:any = {};

    if (url == OPERATIONALDATATYPE.增加) {
        queray.data = data;
    } else if (url == OPERATIONALDATATYPE.修改) {
        queray.data = data;
        queray.param = param;
    } else if (url == OPERATIONALDATATYPE.删除) {
        queray.param = param;
    } else {
        throw new BizError(ERRORENUM.该方法仅可进行数据操作, '使用操作数据库的方法进行查询调用');
    }

    let result:any = await post(`${systemConfig.dbPath}${url}`, queray, header );
    if (result.code != 200) throw new BizError(ERRORENUM.数据操作失败, result.code);
    if (!result.data || !result.data.isSuccess) throw new BizError(ERRORENUM.数据操作失败, JSON.stringify(result.data));

    return true;
}


/**
 * 查询
 * @param url url地址 根据枚举获取值
 * @param tableName 表名
 * @param param 条件
 * @param pageNumber 分页传入的页数  非分页可不传
 * @param pageSize 分页传入的单页大小  非分页可不传
 */
export async function selectData(url, tableName, param, column,  pageNumber?, pageSize?) {
    if (url != OPERATIONALDATATYPE.查询单个 && url != OPERATIONALDATATYPE.查询多个 && url != OPERATIONALDATATYPE.查询数据量 &&  url != OPERATIONALDATATYPE.分页查询) {
        throw new BizError(ERRORENUM.该方法仅可进行查询操作, '该方法仅可进行查询操作');
    }
    let header = {table:tableName, sign:systemConfig.dbSign};
    let queray:any = {param};
    if (column && column.length) {
        queray.column = column;
    }

    if (url == OPERATIONALDATATYPE.分页查询) {
        if (!pageNumber) throw new BizError(ERRORENUM.分页请设置当前页数, `pageNumber:${pageNumber};pageSize:${pageSize}`);
        queray.pageNumber = pageNumber;
        queray.pageSize = pageSize || 10;
    }

    let result:any = await post(`${systemConfig.dbPath}${url}`, queray, header );
    if (result.code != 200) throw new BizError(ERRORENUM.数据查询失败, result.code);
    if (!result.data || result.data.data == null || result.data.data == undefined) throw new BizError(ERRORENUM.数据查询失败, JSON.stringify(result.data));
    return result.data.data;
}


/**
 * 多表联查
 * @param url 
 * @param tableName 
 * @param param 
 * @param column 
 * @param includeConf 
 * @param pageNumber 
 * @param pageSize 
 * @returns 
 */
export async function selectManyTableData(url, tableName, param, column,  includeConf, pageNumber?, pageSize?) {
    if (url != OPERATIONALDATATYPE.多表联查 && url != OPERATIONALDATATYPE.多表分页 && url != OPERATIONALDATATYPE.多表单个) {
        throw new BizError(ERRORENUM.该方法仅可进行联合查询操作, '该方法仅可进行联合查询操作');
    }
    let header = {table:tableName, sign:systemConfig.dbSign};
    let queray:any = {param, includeConf};
    if (column && column.length) {
        queray.column = column;
    }
    
    if (url == OPERATIONALDATATYPE.多表分页) {
        if (!pageNumber) throw new BizError(ERRORENUM.分页请设置当前页数, `pageNumber:${pageNumber};pageSize:${pageSize}`);
        queray.pageNumber = pageNumber;
        queray.pageSize = pageSize || 10;
    }

    let result:any = await post(`${systemConfig.dbPath}${url}`, queray, header );
    if (result.code != 200) throw new BizError(ERRORENUM.数据联合查询失败, result.code);
    if (!result.data || !result.data.data) throw new BizError(ERRORENUM.数据联合查询失败, JSON.stringify(result.data));
    return result.data.data;
}

/**
 * 数组分页函数
 * @param {Array} data - 原始数组
 * @param {Number} page - 当前页码（从1开始）
 * @param {Number} pageSize - 每页数据量
 * @returns {Object} 包含分页数据和元信息的对象
 */
export async function paginateArray(data, page = 1, pageSize = 10) {
    // 校验参数合法性
    page = Math.max(1, page) || 1;
    pageSize = Math.max(1, pageSize) || 10;
  
    // 计算分页索引
    const startIndex = (page - 1) * pageSize;
    const endIndex = startIndex + pageSize;
  
    // 切割数据并生成结果
    const paginatedData = data.slice(startIndex, endIndex);
    
    return {
      data: paginatedData,
      meta: {
        currentPage: page, //当前页码
        pageSize: pageSize, //每页数据量
        totalItems: data.length, //总条数
        totalPages: Math.ceil(data.length / pageSize), //总页数
        hasNextPage: endIndex < data.length, // 是否有下一页
        hasPrevPage: startIndex > 0 // 是否有上一页
      }
    };
  }


