/**
 * zip文件下载
 */

import * as asyncHandler from 'express-async-handler';
import * as zipBiz from '../biz/createZip';
import { checkUser } from '../middleware/user';
import * as fs from 'fs-extra';



export function setRouter(httpServer) {
    httpServer.post('/admin/zip/download', checkUser, asyncHandler(downloadZipfiles));
}


/**
 * 下载企业文件ZIP包
 * @param req 
 * @param res 
 */
async function downloadZipfiles(req, res) {
    try {
        console.log('收到ZIP下载请求，参数:', req.body);
        let { eId } = req.body;
        
        if (!eId) {
            return res.status(400).json({ error: '缺少必要参数eId' });
        }
        
        let result = await zipBiz.downloadEnterpriseFilesZip(eId);
        
        console.log('ZIP文件创建成功:', result.fileName);
        
        // 设置响应头
        res.setHeader('Content-Type', 'application/zip');
        res.setHeader('Content-Disposition', `attachment; filename=${encodeURIComponent(result.fileName)}`);
        res.setHeader('Access-Control-Expose-Headers', 'Content-Disposition');
        
        // 检查文件是否存在
        if (!await fs.pathExists(result.filePath)) {
            throw new Error('ZIP文件未找到: ' + result.filePath);
        }
        
        // 获取文件状态
        const stats = await fs.stat(result.filePath);
        res.setHeader('Content-Length', stats.size);
        
        console.log(`准备发送文件: ${result.fileName}, 大小: ${stats.size} bytes`);
        
        // 发送文件
        res.sendFile(result.filePath, (err) => {
            if (err) {
                console.error('发送文件失败:', err);
                if (!res.headersSent) {
                    res.status(500).json({ error: '文件发送失败', message: err.message });
                }
            } else {
                console.log('文件发送成功');
            }
            
            // 可选：发送完成后删除临时文件
            setTimeout(async () => {
                try {
                    if (await fs.pathExists(result.filePath)) {
                        await fs.remove(result.filePath);
                        console.log('临时文件已清理:', result.filePath);
                    }
                } catch (cleanupError) {
                    console.error('清理临时文件失败:', cleanupError);
                }
            }, 1000);
        });
        
    } catch (error) {
        console.error('下载文件失败:', error);
        if (!res.headersSent) {
            res.status(500).json({ 
                error: '下载失败', 
                message: error.message,
                stack: process.env.NODE_ENV === 'development' ? error.stack : undefined
            });
        }
    }
}



