/**
 * 小程序
 */

import { ERRORENUM } from '../../config/errorEnum';
import * as partyMemberData from '../../data/partyMember';
import * as sysTools from "../../tools/system";
import { BizError } from '../../util/bizError';

/**
 * 党员入口登录
 * @param loginId 登录账号
 * @param pwd 登录密码
 * @returns loginUserInfo: {loginId, firstLogin, token} 登录后的信息
 */
export async function login(loginId:string, pwd:string) {
    let loginInfo = await partyMemberData.findPartyMemberByLoginId(loginId);
    if (!loginInfo) throw new BizError(ERRORENUM.账号不存在);

    let checkPwd = sysTools.getPwdMd5(loginInfo.loginId, pwd);
    if (loginInfo.pwd != checkPwd) throw new BizError(ERRORENUM.密码错误);

    const Token = sysTools.getToken(loginId);

    let loginUserInfo = {
        loginId: loginInfo.loginId,
        firstLogin: !loginInfo.firstLoginIsChangePwd,
        token: Token
    };

    loginInfo.token = Token;
    loginInfo.tokenMs = new Date().valueOf;

    await loginInfo.save();

    return loginUserInfo;
}


/**
 * 首次登录修改密码
 * @param loginId 登录账号
 * @param pwd 新密码
 * @param confirmPwd 确认密码 
 * @returns {isSuccess: true}
 */
export async function firstLoginChangePwd(loginId:string, pwd:string, confirmPwd:string) {
    if (pwd != confirmPwd) throw new BizError(ERRORENUM.密码不一致);
    if (pwd.search(/^[A-Za-z0-9]{6,18}$/) < 0) throw new BizError(ERRORENUM.密码只能由6至18位字符和数字组成);

    let partyMemberInfo = await partyMemberData.findPartyMemberByLoginId(loginId);
    if (partyMemberInfo) throw new BizError(ERRORENUM.未找到数据);
    if (partyMemberInfo.firstLoginIsChangePwd) throw new BizError(ERRORENUM.不能重复修改密码, `重复调用了首次登录之后的修改密码接口${loginId}`);

    partyMemberInfo.pwd = sysTools.getPwdMd5(loginId, sysTools.md5PwdStr(pwd));
    partyMemberInfo.firstLoginChangePwd = true;
    await partyMemberInfo.save();

    return {isSuccess: true};
}


/**
 * 修改密码
 * 小程序端 2.0
 * @param uscc 信用代码
 * @param pwd 原密码  md5之后的
 * @param newPwd 新密码  未md5
 * @param confirmPwd 确认新密码  未md5
 * @returns {isSuccess:true/false}
 */
 export async function changePassword(loginId:string, pwd:string, newPwd:string, confirmPwd:string ) {
    if (newPwd != confirmPwd) throw new BizError(ERRORENUM.密码不一致);
    if (newPwd.search(/^[A-Za-z0-9]{6,18}$/) < 0) throw new BizError(ERRORENUM.密码只能由6至18位字符和数字组成);

    let partyMemberInfo = await partyMemberData.findPartyMemberByLoginId(loginId);
    if (!partyMemberInfo) throw new BizError(ERRORENUM.未找到数据);

    /**由于pwd是md5之后的 所以这里md5一次即可 */
    let checkPwd = sysTools.getPwdMd5(partyMemberInfo.uscc, pwd);
    if (partyMemberInfo.pwd != checkPwd) throw new BizError(ERRORENUM.密码错误);

    /**考虑到如果前端把新密码也md5 就不可以验证是不是符合规则 所以前端传的是明文
     * 我们初始化密码的时候 有两次加密 第一次是密码md5 然后再和uscc 进行md5
     * 于是这里也要如此操作
     */
     partyMemberInfo.pwd = sysTools.getPwdMd5(loginId, sysTools.md5PwdStr(newPwd));
    await partyMemberInfo.save();

    return {isSuccess:true};
}