/**
 * 小程序
 * 组织生活
 */

import moment = require("moment");
import * as organizationalLifeData from "../../../data/organizationalLife";
import * as configEnum from "../../../config/enum";
import { eccFormParam } from "../../../util/verificationParam";
import * as eccFormParamConfig from "../../../config/eccFormParamConfig";
import * as verificationEnumTools from "../../../util/verificationEnum";
import { checkChange, extractData } from "../../../util/piecemeal";
import * as splitResultConfig from "../../../config/splitResultConfig";
import * as partyMemberData from "../../../data/partyMember";
import { BizError } from "../../../util/bizError";
import { ERRORENUM } from "../../../config/errorEnum";

moment.suppressDeprecationWarnings = true;

/**
 * 组织生活列表
 * todo  接口协议不对
 * @param status 活动状态 全部、正在进行、即将开展、已经开展
 * @return data
 * 返回说明 根据传入的活动状态返回不同的列表
 */
export async function getOrganizationalLifeList(type:number, name:number) {
    let allmoveAbout = []; //全部活动列表
    let moveAboutList = [];

    let data = [];

    switch (type) {
        case configEnum.TYPE.正在进行:
            moveAboutList = await organizationalLifeData.findAfootActivityList(name);
        break;
        case configEnum.TYPE.即将开展:
            moveAboutList = await organizationalLifeData.findBeginInActivityList(name);
        break;
        case configEnum.TYPE.已经开展:
            moveAboutList = await organizationalLifeData.findStartedActivityList(name);
        break;
        default:
            allmoveAbout = await organizationalLifeData.findOrganizationalLifeList({partyActivityType:name});
            allmoveAbout.forEach(info => {
                let activityStatus = getActivityStatus(info.activityStartTime, info.activityEndTime);
                data.push({
                    id: info.id,
                    partyActivityType: verificationEnumTools.changeEnumValue(configEnum.ACTIVITYTYPE, info.partyActivityType),
                    activityStartTime: moment(info.activityStartTime).format("YYYY.MM.DD HH:mm"),
                    activityEndTime: moment(info.activityEndTime).format("YYYY.MM.DD HH:mm"),
                    activityAdd: info.activityAdd,
                    activityStatus
                });
            })
        break;
    }

    moveAboutList.forEach( info => {
        let changeData:any = extractData(splitResultConfig.moveAboutInfoConfig, info, false);
        changeData.partyActivityType = verificationEnumTools.changeEnumValue(configEnum.ACTIVITYTYPE, info.partyActivityType);
        changeData.activityStartTime = moment(info.activityStartTime).format("YYYY.MM.DD HH:mm"); 
        changeData.activityEndTime = moment(info.activityEndTime).format("YYYY.MM.DD HH:mm");
        changeData.activityStatus = getActivityStatus(info.activityStartTime, info.activityEndTime);
        data.push(changeData);
    })
    
    return data;
}


/**
 * 创建组织生活活动
 * @param param  {党建活动类型, 活动开始时间, 活动结束时间, 活动地点, 活动主持人, 活动记录人, 请假党员, 参与党员, 活动记录}
 *      参数类型  {  number,      number,      number,    string,  [string],  [string],  [number], [number], string}
 * @return {isSuccess: true}
 */
export async function createOrganizationalLife(param) {
    eccFormParam("创建组织生活活动", eccFormParamConfig.CreateActivityConfig, param)
    verificationEnumTools.eccEnumValue('创建组织生活活动的党建活动类型', 'partyActivityType', configEnum.ACTIVITYTYPE, param.partyActivityType);

    await organizationalLifeData.createActivity(param);

    return {isSuccess: true};
}


/**
 * 组织生活详细信息
 * @param oId 编号
 * @returns 
 */
export async function getOrganizationalLifeById(oId:number) {
    let activityInfo = await organizationalLifeData.findEventDetails(oId);
    let changeData:any = {};
    let leavePartyMember = [];
    let participatingPartyMember = [];

    /**截取返回字段 */
    changeData = extractData(splitResultConfig.activityInfoConfig, activityInfo, false);
    changeData.partyActivityType = verificationEnumTools.changeEnumValue(configEnum.ACTIVITYTYPE, activityInfo.partyActivityType);
    changeData.activityStartTime = moment(activityInfo.activityStartTime).format("YYYY.MM.DD HH:mm"); 
    changeData.activityEndTime = moment(activityInfo.activityEndTime).format("YYYY.MM.DD HH:mm"); 

    //请假党员
    for (let i = 0; i < activityInfo.leavePartyMember.length; i++) {
        let nameData = await partyMemberData.getNameById(activityInfo.leavePartyMember[i]);
        nameData.forEach( nameInfo => {
            leavePartyMember.push(nameInfo.name);
        })
    };

    //参与党员
    for (let i = 0; i < activityInfo.participatingPartyMember.length; i++) {
        let name = await partyMemberData.getNameById(activityInfo.participatingPartyMember[i]);
        name.forEach( nameInfo => {
            participatingPartyMember.push(nameInfo.name);
        })
    };
    changeData.activityStatus = getActivityStatus(activityInfo.activityStartTime, activityInfo.activityEndTime);

    changeData.leavePartyMember = leavePartyMember;
    changeData.participatingPartyMember = participatingPartyMember;

    
    return changeData;
}


/**
 * 修改组织生活活动详情
 * @param oId 组织生活id 修改对应oId的组织生活信息
 * @param param {党建活动类型, 活动开始时间, 活动结束时间, 活动地点, 活动主持人, 活动记录人, 请假党员, 参与党员, 活动记录}
 *      参数类型 {  number,      number,      number,    string,  [string],  [string],  [number], [number], string}
 * @returns 
 */
export async function updateOrganizationalLife(oId:number, param) {
    /**校验表单参数 */
    eccFormParam("组织生活活动表单修改", eccFormParamConfig.UpdateActivityFormConfig, param );
    verificationEnumTools.eccEnumValue('修改组织生活', 'partyActivityType', configEnum.ACTIVITYTYPE, param.partyActivityType);
    
    let activityInfo = await organizationalLifeData.findEventDetails(oId);
    if (!activityInfo) throw new BizError(ERRORENUM.未找到数据, `库中不存在id为${oId}的组织活动`);
    
    let changeList = checkChange(param, activityInfo);
    if ( !changeList.length ) throw new BizError(ERRORENUM.数据无更新, `${oId}数据无更新`);

    changeList.forEach(key => {
        activityInfo[key] = param[key];
    });

    await activityInfo.save();

    return {isSuccess:true};
}


/**
 * 创建/修改组织生活选择党员
 * @param type 1：党员干部、2：入党积极分子、3：预备党员、4：全部党员 用于选项卡切换类型 
 * @param name 党员姓名  用于输入党员姓名查询
 * @returns 根据传入的type返回不同党员信息
 */
export async function getPartyMemberNameList(type:number, name:string) {
    let selectParam:any = {};
    let partyMemberList;

    if (type != 4) {
        selectParam.entryType = type;
        if(name != null) {
            selectParam.name = name;
        }
        partyMemberList = await partyMemberData.getPartyMemberListByEntryType(selectParam);
    } else {
        partyMemberList = await partyMemberData.findPartyMemberAll(selectParam);
    }

    let data = [];
    partyMemberList.forEach( info => {
        let changeData:any = extractData(splitResultConfig.partyMemberConfig, info, false);
        changeData.partyPositions = verificationEnumTools.changeEnumValue(configEnum.PARTYPOSITIONS, info.partyPositions);
        data.push(changeData);
    })
    return data;
}


/**
 * 创建/修改组织生活选择党员
 * @returns 返回所有党员信息
 */
export async function getAllPartyMember() {
    let partyMemberList = await partyMemberData.findPartyMemberAll({});
    let data = [];
    partyMemberList.forEach( info => {
        let changeData:any = extractData(splitResultConfig.partyMemberConfig, info, false);
        changeData.partyPositions = verificationEnumTools.changeEnumValue(configEnum.PARTYPOSITIONS, info.partyPositions);
        switch (changeData.entryType) {
            case 1:
                changeData.color = "#C62A23";
            break;
            case 2:
                changeData.color = "#3549B0";
            break;
            case 3:
                changeData.color = "#E4B386";
            break;
        }
        data.push(changeData);
    })

    return data;
}


/**
 * 删除组织生活活动
 * @param oId 活动id
 * @returns 
 */
export async function deleteOrganizationalLife(oId:number) {
    let activityInfo = await organizationalLifeData.findEventDetails(oId);
    if (activityInfo) throw new BizError(ERRORENUM.未找到数据, `库中不存在id为${oId}的组织活动`);

    await organizationalLifeData.deleteActivity(oId);

    return {isSuccess:true};
}


/**
 * 组织生活活动状态
 * 活动状态 1：正在进行、2：即将开展、3：已经开展
 * @param activityStartTime 活动开始时间
 * @param activityEndTime 活动结束时间
 * @return status --活动状态
 */
function getActivityStatus(activityStartTime:number, activityEndTime:number) {
    let nowTime = new Date().valueOf();

    let startTime = moment(activityStartTime).format("YYYY.MM.DD HH:mm");
    let endTime = moment(activityEndTime).format("YYYY.MM.DD HH:mm");
    let underway = moment().isBetween(startTime, endTime); //true：正在进行

    let status;
    if (underway) {
        status = "正在进行";
    }

    if (nowTime < activityStartTime) {
        status = "即将开展";
    }

    if (nowTime > activityStartTime) {
        status = "已经开展";
    }

    return status;
}