/**
 * 党员基础信息
 */

import * as configEnum from "../config/enum";
import * as clientEnum from "../config/clientEnum";
import * as verificationEnumTools from "../util/verificationEnum";
import { extractData } from "../util/piecemeal";
import * as splitResultConfig from "../config/splitResultConfig";
import moment = require("moment");
import { OPERATIONALDATATYPE, TABLENAME } from "../config/dbEnum";
import { operationalData, selectData, selectManyTableData } from "../middleware/databaseSuccess";
import { eccFormParam } from "../util/verificationParam";
import { CreateFZDXConfig, CreateRDJJFZConfig, CreateRDSQConfig, CreateYBDYConfig, CreateZSDYConfig } from "../config/eccFormParamConfig";
import { getBindId, getRandomId, md5PwdStr } from "../tools/system";

const ColorConfig = {
    1:"#C62A23",
    2:"#E4B386",
    3:"#2aa115",//发展对象
    4:"#3549B0",
    5:"#1e5eff"
};

function changePartyPositionsName(partyPositionsName, etId) {
    let str = "";
    for (let i= 0; i < partyPositionsName.length; i++) {
        let item = partyPositionsName[i];
        if (item < 10) continue;
        str += `,${verificationEnumTools.changeEnumValue(configEnum.PARTYPOSITIONS, item)}`;
    }
    if (!str) {
        let partyPositionsNameStr = "";
        if (etId == configEnum.ENTRYTYPE.正式党员) partyPositionsNameStr = "党员";
        else partyPositionsNameStr = verificationEnumTools.changeEnumValue(configEnum.ENTRYTYPE, etId);
        return {partyPositionsNameStr, color:ColorConfig[etId]};
    }

    return {partyPositionsNameStr : str.replace(",", ""), color:"#C62A23"};
}


/**
 * 获取党员名册
 * @param branchId 支部id
 * @returns 返回格式 {nameList:[返回字段参考splitResultConfig文件中的partyMemberConfig配置], partyMemberCount:{目前党员人数}}
 */
export async function partyMemberNameList(bId:number, name:string) {
    let branchName = verificationEnumTools.changeEnumValue(configEnum.BRANCHNAME, bId);
    
    let partyMemberNum = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.党员基础信息表, {bId:{"%like%":bId}}, []); //党员总数
    let partyMemberCount = {
        key: branchName + "目前党员人数",
        value: partyMemberNum || 0
    };
    let param = {bId:{"%like%":bId}}
    if (name) param["name"] = {"%like%":name};
    let clomun = ["pmId", "name", "etId", "partyPositionsName", "partyJoiningTime", "departmentName"];
    let partyMemberList = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.党员基础信息表, param, clomun);

    let nameList = [];
    /**截取返回字段 */
    partyMemberList.forEach( info => {
        let changeData:any = extractData(splitResultConfig.partyMemberConfig, info);

        let {color, partyPositionsNameStr} = changePartyPositionsName( JSON.parse(info.partyPositionsName), changeData.etId);
        changeData.partyPositionsName = partyPositionsNameStr;
        changeData.color = color;

        let time = moment(new Date()).format("YYYY");
        let partyJoiningTime = time;
        if (info.partyJoiningTime) partyJoiningTime = moment(info.partyJoiningTime).format("YYYY")
        changeData.partyStanding = parseInt(time) - parseInt(partyJoiningTime) + "年";

        let departmentName = JSON.parse(changeData.departmentName);
        changeData.departmentName = verificationEnumTools.changeEnumValue(configEnum.DEDPARTMENT, departmentName);

        nameList.push(changeData);
    })

    return {nameList, partyMemberCount};
}


/**
 * 党员详情
 * @param pmId 党员id
 * @returns changeData:{返回字段参考splitResultConfig文件中的partyMemberDetailsConfig配置} 根据党员id返回党员详细信息
 */
 export async function partyMemberDetails(bId:number, pmId:string) {
    let clomun = ["bId", "pmId", "etId", "name", "sex", "birthDate", "phone", "levelOfEducation", "bId", "partyPositionsName", "partyJoiningTime", "administrativePositionName", 
    "departmentName", "askForTime", "listedAsActivistsTime", "liaison", "talkTime", "partyState"];
    let partyMemberInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.党员基础信息表, {pmId}, clomun);
    let changeData:any =  extractData(splitResultConfig.partyMemberConfig, partyMemberInfo);

    let {color, partyPositionsNameStr} = changePartyPositionsName( JSON.parse(partyMemberInfo.partyPositionsName), changeData.etId);
    changeData.color = color;
    changeData.partyPositionsName = partyPositionsNameStr;
    
    changeData.departmentName = verificationEnumTools.changeEnumValue(configEnum.DEDPARTMENT, JSON.parse(partyMemberInfo.departmentName));
    changeData.sex = verificationEnumTools.changeEnumValue(configEnum.SEX, partyMemberInfo.sex);
    changeData.administrativePositionName = verificationEnumTools.changeEnumValue(configEnum.ADMINISTRATIVEPOSITION, partyMemberInfo.administrativePositionName);
    changeData.partyState = verificationEnumTools.changeEnumValue(configEnum.PARTYSTATE, partyMemberInfo.partyState);
    changeData.levelOfEducation = verificationEnumTools.changeEnumValue(configEnum.DEGREE, partyMemberInfo.levelOfEducation);

    changeData.study = await getStudent(bId, pmId); //学习相关数据

    let time = moment(new Date()).format("YYYY");
    let partyJoiningTime = time;
    if (changeData.partyJoiningTime) partyJoiningTime = moment(partyMemberInfo.partyJoiningTime).format("YYYY")
    changeData.partyStanding = parseInt(time) - parseInt(partyJoiningTime) + "年";

    if (partyMemberInfo.partyJoiningTime) {
        changeData.partyStanding = Math.floor(moment().diff(partyMemberInfo.partyJoiningTime, 'years'))+'年';
        changeData.partyJoiningTime = moment(partyMemberInfo.partyJoiningTime).format("YYYY年MM月DD日");
    } else {
        changeData.partyStanding = "-年";
        changeData.partyJoiningTime = "-";
    }
    
    changeData.position = "-";

    let middleInfo = {
        "payAmountCount":100,//年度在线缴纳党费
        "activityCount":0,//年度参加支部生活
        "offlineCount":0,//集中培训和集体学习
        "online":0//网络自学
    }
    return {dataInfo:changeData, middleInfo};
}


/**
 * 党员学习数据
 * @param bId 支部
 * @param pmId 党员
 * @returns studentData:[{key:"学习排名", value:0},{key:"学习课时", value:0},{key:"学习进度", value:0}]
 */
export async function getStudent(bId:number, pmId:string) {
    let orderParam = {"%orderDesc%": "dataTime", "%limit%": 1};
    let monthData = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.学习强国, orderParam, ["dataTime"]);
    if (!monthData || !monthData.dataTime) return [];

    let dataTime = monthData.dataTime;
    let stTime = moment(dataTime).startOf("month").format("YYYY-MM-DD HH:mm:ss");
    let etTime = moment(dataTime).endOf('month').format("YYYY-MM-DD HH:mm:ss");
    
    let param = {
        dataTime:{"%between%": [stTime, etTime]}
    };
    if (bId) param["bId"] = {"%like%": bId};
   
    let clomun = ["pmId", "bId", "dayIntegral", "dataTime"];
    let includeConf = {"partyMember":["name"]}
    let learningPowerData = await selectManyTableData(OPERATIONALDATATYPE.多表联查, TABLENAME.学习强国, param, clomun, includeConf);

    let partyMemberNameData = {};
    learningPowerData.forEach( info => {
        let {pmId, partyMember, bId, dayIntegral, dataTime} = info;
        if (!partyMemberNameData[partyMember.name]) {
            partyMemberNameData[partyMember.name] = {
                pmId,
                name:partyMember.name,
                bId:verificationEnumTools.changeEnumValue(configEnum.BRANCHNAME, bId), 
                dayIntegral:0,
                dataTime: moment(dataTime).format("YYYY-MM"),
            };
        }

        partyMemberNameData[partyMember.name].dayIntegral += dayIntegral;
    });
    let learningPowerList = Object.values(partyMemberNameData);

    learningPowerList.sort( (a:any, b:any) => {
        return b.dayIntegral - a.dayIntegral;
    });

    for ( let i = 0; i < learningPowerList.length; i++) {
        learningPowerList[i]["ranking"] = i + 1;
    }
    
    let studentData:any = [{key: "学习排名", value: 0}];

    let rateLearningData = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.学习进度表, {pmId}, ["mlId", "pmId", "rateOfLearning", "learningCompleted"]);

    let memberLearningData = {rateOfLearning:0};
    for (let i = 0; i < rateLearningData.length; i++) {
        let { rateOfLearning } = rateLearningData[i];
        memberLearningData.rateOfLearning += rateOfLearning;
    }

    let learningHours = rateLearningData.length || 0;
    let rateOfLearning = memberLearningData.rateOfLearning / rateLearningData.length || 0;
    studentData.push(
        {key:"学习课时", value: learningHours},
        {key:"学习进度", value: Math.round(rateOfLearning)}
    );

    return studentData;
}


/**
 * 添加党员信息
 * @param memberType
 * @param param 
 * @returns 
 */
export async function addMember(bId, memberType:number, param) {
    let addInfo:any = {
        pmId:getRandomId("member"),
        bId,
        etId:memberType,
        isAdmin:0,
        loginId:param.phone,
    };//添加参数

    for (let key in param) {
        addInfo[key] = param[key];
    }

    const FuncName = "添加党员";
    switch (memberType) {
        case configEnum.ENTRYTYPE.正式党员:
            eccFormParam(FuncName, CreateZSDYConfig, param);
            addInfo.partyJoiningTime =moment(param.partyJoiningTime).format("YYYY-MM-DD");
            param.partyPositionsName.forEach(itemId => {
                verificationEnumTools.eccEnumValue(FuncName, "partyPositionsName", configEnum.PARTYPOSITIONS, itemId);
            });
            addInfo.partyPositionsName = JSON.stringify(param.partyPositionsName);
        break;
        case configEnum.ENTRYTYPE.入党申请:
            eccFormParam(FuncName, CreateRDSQConfig, param);
            addInfo.askForTime = moment(param.askForTime).format("YYYY-MM-DD");
            addInfo.talkTime = moment(param.talkTime).format("YYYY-MM-DD");
        break;
        case configEnum.ENTRYTYPE.入党积极分子:
            eccFormParam(FuncName, CreateRDJJFZConfig, param);
            addInfo.askForTime = moment(param.askForTime).format("YYYY-MM-DD");
            addInfo.listedAsActivistsTime = moment(param.listedAsActivistsTime).format("YYYY-MM-DD");
        break;
        case configEnum.ENTRYTYPE.发展对象:
            eccFormParam(FuncName, CreateFZDXConfig, param);
            addInfo.askForTime = moment(param.askForTime).format("YYYY-MM-DD");
            addInfo.fzdxTime = moment(param.fzdxTime).format("YYYY-MM-DD");
        break;
        case configEnum.ENTRYTYPE.预备党员:
            eccFormParam(FuncName, CreateYBDYConfig, param);
            addInfo.askForTime = moment(param.askForTime).format("YYYY-MM-DD");
            addInfo.yubeiTime = moment(param.yubeiTime).format("YYYY-MM-DD");
        break;
    }

    verificationEnumTools.eccEnumValue(FuncName, "sex", configEnum.SEX, param.sex);
    verificationEnumTools.eccEnumValue(FuncName, "levelOfEducation", configEnum.DEGREE, param.levelOfEducation);
    verificationEnumTools.eccEnumValue(FuncName, "administrativePositionName", configEnum.ADMINISTRATIVEPOSITION, param.administrativePositionName);

    addInfo.birthDate = moment(param.birthDate).format("YYYY-MM-DD");

    addInfo.pwd = md5PwdStr(param.phone.slice(param.phone.length-6));

    await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.党员基础信息表, addInfo, {});

    return {isSuccess:true};
}


