/**
 * 组织生活
 */

import moment = require("moment");
import { OPERATIONALDATATYPE, TABLENAME } from "../config/dbEnum";
import { ERRORENUM } from "../config/errorEnum";
import { operationalData, selectData } from "../middleware/databaseSuccess";
import { BizError } from "../util/bizError";
import { extractData } from "../util/piecemeal";
import * as splitResultConfig from "../config/splitResultConfig";
import * as verificationEnumTools from "../util/verificationEnum";
import * as configEnum from "../config/enum";
import { getFileUrl } from "../middleware/getUrl";
import { eccFormParam } from "../util/verificationParam";
import { CreateActivityConfig, UpdateActivityConfig } from "../config/eccFormParamConfig";


/**
 * 获取组织生活列表
 * @param bId 支部id
 * @param theme 活动名称 模糊查询
 * @param themeType 活动类型
 * @param dataMonth 活动月份
 * @returns organizationalLifeList[{"oId":1, "theme":"活动主题", "themeType":"活动类型", "dataMonth":"活动月份"}]
 */
export async function organizationalLifeList(bId:number, theme:string, themeType:number) {
    let param = { bId };
    if (theme) param["theme"] = {"%like%": theme};
    if (themeType) param["partyActivityType"] = themeType;

    let clomun = ["oId", "activityStartTime", "activityEndTime", "activityAdd", "partyActivityType"];
    let organizationalLifeDataList = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.组织生活表, param, clomun);

    const Now = new Date().valueOf();
    let dataList = [];
    organizationalLifeDataList.forEach(info => {
        let st = new Date(info.activityStartTime).valueOf();
        let et = new Date(info.activityEndTime).valueOf();
        let state = "";
        if (st < Now && et > Now) state = "正在进行";
        else if (Now > et) state = "已结束";
        else state = "即将开展";

        let sTimeStr = moment(info.activityStartTime).format("YYYY-MM-DD HH:mm");
        let time = `${sTimeStr}-${moment(info.activityEndTime).format("HH:mm")}`;

        dataList.push({
            oId:info.oId,
            address:info.activityAdd,
            time,
            state,
            theme:verificationEnumTools.changeEnumValue(configEnum.THEMETYPE, info.partyActivityType)
        });
    });

    return {dataList};
}




/**
 * 获取组织生活详情
 * @param oId 组织生活id
 * @returns orgLifeInfo:{"oId":1, "theme":"活动主题", "themeType":"活动类型", "fileName":["活动文件名称", ......]}
 */
export async function organlzationalLifeListDetsils(oId:number, token:string, userid:string, bId) {

    let clomun = ["oId", "partyActivityType", "activityStartTime", "activityEndTime", "activityAdd", "activityHost", "activityNoteTaker", "leavePartyMember", "participatingPartyMember", "activityLog", "signedInMember"];
    let orgLifeInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.组织生活表, {oId}, clomun);
    if (!orgLifeInfo || !orgLifeInfo.oId) throw new BizError(ERRORENUM.目标数据不存在);

    let leavePartyMember = JSON.parse(orgLifeInfo.leavePartyMember);
    let signedInMember = JSON.parse(orgLifeInfo.signedInMember);//签到人员列表
    let participatingPartyMemberIdList = JSON.parse(orgLifeInfo.participatingPartyMember);
   
    let thisBIdAllUser = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.党员基础信息表, {bId}, ["pmId", "name"]);
    let checkNameMap = {};
    thisBIdAllUser.forEach(info => {
        checkNameMap[info.pmId] = info.name;
    })

    let notSignedInMember = [];
    let statis = { signedIn:0, notSignedIn:0 };
    let participatingPartyMemberStr = "";
    participatingPartyMemberIdList.forEach(item => {
        participatingPartyMemberStr += `,${checkNameMap[item]}`;
         if (signedInMember.indexOf(item) > -1) {
            statis.signedIn += 1;
        } else {
            statis.notSignedIn += 1;
            notSignedInMember.push(item);
        }
    });
    participatingPartyMemberStr = participatingPartyMemberStr.replace(",", "");

    //  let sTimeStr = moment(info.activityStartTime).format("YYYY-MM-DD HH:mm");
    //     let time = `${sTimeStr}-${moment(info.activityEndTime).format("HH:mm")}`


    let dataInfo = {
        partyActivityType:orgLifeInfo.partyActivityType,//党建活动类型
        activityStartTime:new Date(orgLifeInfo.activityStartTime).valueOf(),//活动开始时间
        activityEndTime:new Date(orgLifeInfo.activityEndTime).valueOf(),//活动结束时间
        // activityStartTime:moment(orgLifeInfo.activityStartTime).format("YYYY-MM-DD"),//活动开始时间
        // activityEndTime:moment(orgLifeInfo.activityEndTime).format("YYYY-MM-DD"),//活动结束时间
        // activityTime:``
        activityAdd:orgLifeInfo.activityAdd,//活动地点
        activityHost:orgLifeInfo.activityHost,//活动主持人
        activityNoteTaker:orgLifeInfo.activityNoteTaker,//活动记录人
        leavePartyMember,//请假党员
        participatingPartyMember:participatingPartyMemberIdList,//参与党员 
        participatingPartyMemberStr:participatingPartyMemberStr,//解析好的字符串
        activityLog:orgLifeInfo.activityLog,
        statis,
        signedInMember,//签到人员id列表
        notSignedInMember,//未签到人员id列表
    };

    return {dataInfo};
}

export async function addOrg(bId, param) {
    const FuncName = "添加支部活动";
    eccFormParam(FuncName, CreateActivityConfig, param);
    verificationEnumTools.eccEnumValue(FuncName, "partyActivityType", configEnum.THEMETYPE, param.partyActivityType );

    let leavePartyMember = [];
    if (param.leavePartyMember) leavePartyMember = [param.leavePartyMember];

    let addInfo = {
        bId,
        partyActivityType:param.partyActivityType,
        activityStartTime:moment(param.activityStartTime).format("YYYY-MM-DD HH:mm:SS"),
        activityEndTime:moment(param.activityEndTime).format("YYYY-MM-DD HH:mm:SS"),
        activityAdd:param.activityAdd,
        activityHost:param.activityHost,
        activityNoteTaker:param.activityNoteTaker,
        leavePartyMember:JSON.stringify(leavePartyMember),
        participatingPartyMember:JSON.stringify(param.participatingPartyMember || []),
    };
    await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.组织生活表, addInfo, {});

    return {isSuccess:true}
}

export async function updateOrg(oId, param) {
    let clomun = ["oId"];
    let orgLifeInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.组织生活表, {oId}, clomun);
    if (!orgLifeInfo || !orgLifeInfo.oId) throw new BizError(ERRORENUM.目标数据不存在);

    eccFormParam("修改支部活动", UpdateActivityConfig, param);

     let activityNoteTaker = [];
    if (param.activityNoteTaker) activityNoteTaker = [param.activityNoteTaker];

    let updateInfo = {
        partyActivityType:param.partyActivityType,
        activityStartTime:moment(param.activityStartTime).format("YYYY-MM-DD HH:mm:SS"),
        activityEndTime:moment(param.activityEndTime).format("YYYY-MM-DD HH:mm:SS"),
        activityAdd:param.activityAdd,
        activityHost:param.activityHost,
        activityNoteTaker:param.activityNoteTaker,
        leavePartyMember:JSON.stringify(param.leavePartyMember || []),
        participatingPartyMember:JSON.stringify(param.participatingPartyMember || []),
        signedInMember:JSON.stringify(param.signedInMember || []),//签到人员
        activityLog:param.log,//活动记录
    };

    await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.组织生活表, updateInfo, {oId});

    return {isSuccess:true}
}

export async function delOrg(oId) {
    let clomun = ["oId"];
    let orgLifeInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.组织生活表, {oId}, clomun);
    if (!orgLifeInfo || !orgLifeInfo.oId) throw new BizError(ERRORENUM.目标数据不存在);


    await operationalData(OPERATIONALDATATYPE.删除, TABLENAME.组织生活表, {}, {oId});

    return {isSuccess:true}
}

const ColorConfig = {
    1:"#C62A23",
    2:"#E4B386",
    3:"#2aa115",//发展对象
    4:"#3549B0",
    5:"#1e5eff"
};

function changePartyPositionsName(partyPositionsName, etId) {
    let str = "";
    for (let i= 0; i < partyPositionsName.length; i++) {
        let item = partyPositionsName[i];
        if (item < 10) continue;
        str += `,${verificationEnumTools.changeEnumValue(configEnum.PARTYPOSITIONS, item)}`;
    }
    if (!str) {
        let partyPositionsNameStr = "";
        if (etId == configEnum.ENTRYTYPE.正式党员) partyPositionsNameStr = "党员";
        else partyPositionsNameStr = verificationEnumTools.changeEnumValue(configEnum.ENTRYTYPE, etId);
        return {partyPositionsNameStr, color:ColorConfig[etId]};
    }

    return {partyPositionsNameStr : str.replace(",", ""), color:"#C62A23"};
}


export async function memberSelect(bId:number) {

    let clomun = ["pmId", "name", "etId", "partyPositionsName", "partyJoiningTime", "departmentName"];
    let partyMemberList = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.党员基础信息表, {bId}, clomun);

    let dataList = [];

    partyMemberList.forEach(info => {
        let {partyPositionsNameStr} = changePartyPositionsName(JSON.parse(info.partyPositionsName), info.etId)
        dataList.push({
            name:info.name,
            userId:info.pmId,
            type:info.etId,
            label:partyPositionsNameStr
        });
    });
    return {dataList}
}














