/**
 * 党员学习 memberLearning
 * 学习进度 rateLearning
 * 学习排行 learningPower
 */

import { OPERATIONALDATATYPE, TABLENAME } from "../config/dbEnum";
import { operationalData, selectData, selectManyTableData } from "../middleware/databaseSuccess";
import moment = require("moment");
import * as verificationEnumTools from "../util/verificationEnum";
import * as configEnum from "../config/enum";
import { systemConfig } from "../config/systemConfig";
import { getFileUrl } from "../middleware/getUrl";
import { BizError } from "../util/bizError";
import { ERRORENUM } from "../config/errorEnum";


/**
 * 获取党员课程学习列表
 * @param pmId 登录的党员id
 * @param courseEntryType 学习类型 视频/文字
 * @param courseTypeName 课程类型
 * @param contentTitle 课程标题
 * @returns rateLearningList:[{"contentTitle":"学习课题名称", "fileName":["视频地址", ......], "rateOfLearning":0, "learningCompleted":0}]
 */
export async function memberLearningList(pmId:string, courseEntryType:number, courseTypeName:number, contentTitle:string, token:string, userid:string) {
    let rateColumn =  ["rlId", "mlId", "rateOfLearning", "learningCompleted", "textSchedule"];
    let rateLearningInfo = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.学习进度表, {pmId}, rateColumn);
    let rateLearningList = [];
    let rateLearningMap = {};
    rateLearningInfo.forEach(info => {
        let {rlId, mlId, rateOfLearning, learningCompleted, textSchedule} = info;
        rateLearningMap[mlId] = {rlId, rateOfLearning, learningCompleted, textSchedule};
    });

    let param = {courseEntryType};
    if(courseTypeName) param["courseTypeName"] = courseTypeName;
    if(contentTitle) param["contentTitle"] = {"%like%": contentTitle};

    let clomun = ["mlId", "contentTitle", "fileName", "coverImage", "videoDuration", "courseEntryType", "desc"];
    let memberLearningInfo = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.党员学习表, param, clomun); 
    
    let courseList = [];
    for(let i = 0; i < memberLearningInfo.length; i++) {
        let memberInfo = memberLearningInfo[i];
        let {mlId, contentTitle, fileName, coverImage, videoDuration, desc} = memberInfo;


        let learningInfo =  rateLearningMap[mlId];
        if (!learningInfo) { //如果对应学习课程没有进度数据，创建一条对应进度为0的数据
            let addInfo = {
                pmId,
                mlId,
                rateOfLearning:0,
                learningCompleted: 1,
                textSchedule:'[]'
            }
            await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.学习进度表, addInfo, {});
            learningInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.学习进度表, {pmId, mlId}, rateColumn); 
            console.log(learningInfo);
        }
        
        let alanysisFileName;
        let successFileList = [];//已完成
        if (courseEntryType == configEnum.COURSEENTRYTYPE.文字) {
            alanysisFileName = getFileUrl(fileName, token, configEnum.UPTYPE.党员学习 , userid);
            successFileList = learningInfo.textSchedule ? JSON.parse(learningInfo.textSchedule) : [];
        } else {
            alanysisFileName = JSON.parse(fileName);
        }
       
         courseList.push({
            videoDuration,
            coverImage,
            contentTitle,
            desc,
            fileName:alanysisFileName
        });

        //  fileName不再是一个字符串 而是一个对象{}
        rateLearningList.push({
            rlId: learningInfo.rlId,
            contentTitle,
            fileName: alanysisFileName,
            rateOfLearning: learningInfo.rateOfLearning,
            learningCompleted: learningInfo.learningCompleted,
            coverImage,
            videoDuration,
            successFileList
        })
    }

    return {rateLearningList, courseList};
}

/**
 * 学习排行
 * 该用户支部的排名
 * 展示数据最新月份的每日积分总和
 * @param bId 支部id 参数存在返回数据最新月份登录账号支部的学习排行，参数不存在返回数据最新月份所有支部的学习排行
 * @returns learningPowerList:[{"pmId":"党员id", "name":"党员名称", "bId":"所属支部", "dayIntegral":得分, "dataTime":"数据月份", "ranking":排名}]
 * @returns oneselfRanking:{"pmId":"党员id", "name":"党员名称", "bId":"所属支部", "dayIntegral":得分, "dataTime":"数据月份", "ranking":排名}
 */
export async function branchRanking(pmId, bId?) {
    let orderParam = {"%orderDesc%": "dataTime", "%limit%": 1};
    let monthData = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.学习强国, orderParam, ["dataTime"]);
    if (!monthData || !monthData.dataTime) return [];

    let dataTime = monthData.dataTime;
    let stTime = moment(dataTime).startOf("month").format("YYYY-MM-DD HH:mm:ss");
    let etTime = moment(dataTime).endOf('month').format("YYYY-MM-DD HH:mm:ss");
    
    let param = {
        dataTime:{"%between%": [stTime, etTime]}
    };
    if (bId) param["bId"] = {"%like%": bId};
   
    let clomun = ["pmId", "bId", "dayIntegral", "dataTime"];
    let includeConf = {"partyMember":["name"]}
    let learningPowerData = await selectManyTableData(OPERATIONALDATATYPE.多表联查, TABLENAME.学习强国, param, clomun, includeConf);

    let partyMemberNameData = {}; //{成员名称:{pmId, bId, score, dataTime} }
    learningPowerData.forEach( info => {
        let {pmId, partyMember, bId, dayIntegral, dataTime} = info;
        if (!partyMemberNameData[partyMember.name]) {
            partyMemberNameData[partyMember.name] = {
                pmId,
                name:partyMember.name,
                bId:verificationEnumTools.changeEnumValue(configEnum.BRANCHNAME, bId), 
                dayIntegral:0,
                dataTime: moment(dataTime).format("YYYY-MM"),
            };
        }

        partyMemberNameData[partyMember.name].dayIntegral += dayIntegral;
    });
    let learningPowerList = Object.values(partyMemberNameData);

    learningPowerList.sort( (a:any, b:any) => {
        return b.dayIntegral - a.dayIntegral;
    });

    let oneselfRanking = {};
    for ( let i = 0; i < learningPowerList.length; i++) {
        learningPowerList[i]["ranking"] = i + 1; //排名
        if(learningPowerList[i]["pmId"] == pmId) {
            oneselfRanking = learningPowerList[i];
        }
    }

    return {learningPowerList, oneselfRanking};
}


/**
 * 修改学习进度
 * @param rlId 学习进度id
 * @param rateOfLearning 学习进度
 */
export async function updateRateOfLearning(rlId:number, rateOfLearning:number) {

    let learningInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.学习进度表, {rlId}, ["rateOfLearning", "mlId"]);

    let mlInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.党员学习表, {mlId:learningInfo.mlId}, ["videoDuration"]);

    let rate = 100;
    if (mlInfo.videoDuration) {
        rate = Math.ceil(rateOfLearning/mlInfo.videoDuration*100);
    }
    if (rate> 100) rate = 100;
    let learningCompleted = 1;
    if (rate == 100) learningCompleted = 2

    if (learningInfo.rateOfLearning < rate) {
        let row = await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.学习进度表, {rateOfLearning:rate, learningCompleted}, {rlId});
    }

    return {isSuccess:true};
}


export async function updateDocRateOfLearning(rlId:number, fileName:string) {
    let rateLearningInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.学习进度表, {rlId}, ["mlId", "rlId", "textSchedule"]);
    if (!rateLearningInfo || !rateLearningInfo.rlId) throw new BizError(ERRORENUM.目标数据不存在, `更新学习进度:rlid:${rlId}`);
    // let {pmId} = rateLearningInfo;
    // let learningInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.学习进度表, {pmId}, ["pmId", "mlId"]);
    // if (!learningInfo || !learningInfo.pmId) throw new BizError(ERRORENUM.目标数据不存在, `更新学习进度:pmId:${pmId}`);
    let mlInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.党员学习表, {mlId:rateLearningInfo.mlId}, ["courseEntryType", "fileName"]);
    if (mlInfo.courseEntryType != configEnum.COURSEENTRYTYPE.文字) throw new BizError(ERRORENUM.该进度不是文字学习, `${rateLearningInfo.mlId}`);

    let textSchedule = !rateLearningInfo.textSchedule ? [] : JSON.parse( rateLearningInfo.textSchedule);

    let check = false;
    textSchedule.forEach(item => {
        if (fileName.indexOf(item) > -1 ) {
            check = true;
        }
    })

    if (!check) {
        textSchedule.push(fileName);
        let fileNameList = JSON.parse(mlInfo.fileName);
         let rate = 0;
        if (fileNameList.length) {
            rate =  Math.ceil(textSchedule.length/fileNameList.length*100);
        }                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           
        await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.学习进度表, {textSchedule:JSON.stringify(textSchedule), rateOfLearning:rate}, {rlId});
    }

    return {isSuccess:true};
}











