/**
 * ABI_智慧党建管理分析平台——党建分析
 */

import moment = require("moment");
import { OPERATIONALDATATYPE, TABLENAME } from "../../config/dbEnum";
import * as enums from "../../config/enum";
import { selectData } from "../../middleware/databaseSuccess";

/**
 * 党建分析——左屏
 * @returns {baseData:基本情况, ratio:男女占比, qualifications:学历情况, ageDistribution:年龄分布, partyExpenses:党费管理}
 */
export async function getLeftScreen() {
    let baseData = {title:"基本情况", list:[]};
    baseData.list.push(
        {key:"支部", value:await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.支部表, {}, null)},
        {key:"正式党员", value:await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.党员基础信息表, {etId:enums.ENTRYTYPE.正式党员}, null)},
        {key:"预备党员", value:await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.党员基础信息表, {etId:enums.ENTRYTYPE.预备党员}, null)}
    );

    let male = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.党员基础信息表, {sex:enums.SEX.男}, null);
    let female = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.党员基础信息表, {sex:enums.SEX.女}, null);
    let ratio = {title:"男女占比", list:[{key:"男", value:male},{key:"女", value:female}]};

    let qualifications = {title:"学历情况", list:[]};
    for(let key in enums.DEGREE) {
        let anyKey:any = key;
        if (isNaN(anyKey)) {
            let keyStr = key;
            let levelOfEducation = enums.DEGREE[key];
            let count = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.党员基础信息表, {levelOfEducation}, null);
            qualifications.list.push({key:keyStr, value:count});
        }
    }

    let ageDistribution = {title:"年龄分布", list:[]};
    let ageDistributionMap = {
        小于29: 0,
        大于30小于39: 0,
        大于40小于49: 0,
        大于50: 0,
    }
    let 小于29 = {"birthDate": {"%gt%": moment().add(-29, 'y').format("YYYY-MM-DD HH:mm:ss")}};
    ageDistributionMap.小于29 = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.党员基础信息表, 小于29, ["birthDate"]);
    let 大于30小于39 = {"birthDate": {"%between%": [moment().add(-39, 'y').format("YYYY-MM-DD HH:mm:ss"), moment().add(-30, 'y').format("YYYY-MM-DD HH:mm:ss")]}};
    ageDistributionMap.大于30小于39 = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.党员基础信息表, 大于30小于39, ["birthDate"]);
    let 大于40小于49 = {"birthDate": {"%between%": [moment().add(-49, 'y').format("YYYY-MM-DD HH:mm:ss"), moment().add(-40, 'y').format("YYYY-MM-DD HH:mm:ss")]}};
    ageDistributionMap.大于40小于49 = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.党员基础信息表, 大于40小于49, ["birthDate"]);
    let 大于50 = {"birthDate": {"%lt%": moment().add(-50, 'y').format("YYYY-MM-DD HH:mm:ss")}};
    ageDistributionMap.大于50 = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.党员基础信息表, 大于50, ["birthDate"]);

    for (let key in ageDistributionMap) {
        ageDistribution.list.push({key, value: ageDistributionMap[key]});
    }

    let partyExpenses = {title:"党费管理", list: []};

    let stTime = moment(new Date().valueOf()).startOf("month").format("YYYY-MM-DD HH:mm:ss");
    let etTime = moment(new Date().valueOf()).endOf('month').format("YYYY-MM-DD HH:mm:ss");
    let monthParam = {payType:enums.PAYTYPE.已缴纳, payTime:{"%between%":[stTime, etTime]}};
    let monthPayment = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.党费管理表, monthParam, ["payAmount"]);
    let payAmount = 0;
    monthPayment.forEach( item => {
        payAmount += item.payAmount;
    })
    partyExpenses["monthPayment"] = payAmount;
    let paymentCount = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.党员基础信息表, {}, null);
    let paidCount = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.党费管理表, {payType:enums.PAYTYPE.已缴纳}, null);
    partyExpenses["paymentRatio"] = Math.round((paidCount / paymentCount)*100)/100;

    for(let key in enums.BRANCHNAME) {
        let anyKey:any = key;
        if (isNaN(anyKey)) {
            let keyStr = key;
            let bId = enums.BRANCHNAME[key];
            let param = {
                bId,
                payType: enums.PAYTYPE.已缴纳,
                payTime: {"%between%":[stTime, etTime]}
            }
            let branchPayment = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.党费管理表, param, ["payAmount"]);
            let branchPayAmount = 0;
            branchPayment.forEach( item => {
                branchPayAmount += item.payAmount;
            })
            partyExpenses.list.push({key:keyStr, value:branchPayAmount});
        }
    }

    return {baseData, ratio, qualifications, ageDistribution, partyExpenses};
}


/**
 * 党建分析——中屏
 * @param year 
 * @returns {partyQualityList:党建质量雷达图, branchPartyQuality:各支部党建质量评分, buildingDynamic:党建动态}
 */
export async function getMidScreen(year:string) {
    let clomun = ["primaryIndicators", "completionProgress"];
    let partyQuality = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.党建质量三级指标, {dataYear:year}, clomun);

    let indicatorsMap = {};
    partyQuality.forEach( info => {
        let {primaryIndicators, completionProgress} = info; //一级指标、完成进度
        if (!indicatorsMap[primaryIndicators]) indicatorsMap[primaryIndicators] = {count:0, completionProgress:0};
        indicatorsMap[primaryIndicators].completionProgress += completionProgress;
        indicatorsMap[primaryIndicators].count += 1;
    });
    let partyQualityList = {title:"党建质量雷达图", list:[]};//list:[{key:一级指标名称,value:一级指标数量:0}]
    for (let key in indicatorsMap) {
        let {count, completionProgress} = indicatorsMap[key];
        let keyStr = enums.PRIMARYINDICATORS[key];
        partyQualityList.list.push({ 
                key:keyStr, 
                value:Math.round((completionProgress / count)*100)/100
            }
        );
    }

    let branchPartyQuality = {title:"各支部党建质量评分", list:[]};
    for(let key in enums.BRANCHNAME) {
        let anyKey:any = key;
        if (isNaN(anyKey)) {
            let keyStr = key;
            let bId = enums.BRANCHNAME[key];
            let branchPartyQualityInfo = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.党建质量三级指标, {bId}, ["currentScore"]);
            let branchScore = 0;
            branchPartyQualityInfo.forEach( item => {
            branchScore += item.currentScore;
            })
            branchPartyQuality.list.push({key:keyStr, value:branchScore});
        }
    }

    let buildingDynamic = {title:"党建动态", list:[]};
    buildingDynamic.list = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.党建动态表, {}, ["theme"])

    return {partyQualityList, branchPartyQuality, buildingDynamic};
}


/**
 * 党建分析——右屏
 * @returns {activity:组织生活, basicSystem:, totalIntegral, partyVanguard}
 */
export async function getRightScreen() {
    let activity = {title:"组织生活", activityCount:0, list:[]}
    activity.activityCount = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.组织生活表, {}, null); //组织生活总数
    
    for(let key in enums.BRANCHNAME) {
        let anyKey:any = key;
        if (isNaN(anyKey)) {
            let keyStr = key;
            let bId = enums.BRANCHNAME[key];
            let count = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.组织生活表, {bId}, null);
            activity.list.push({key:keyStr, value:count});
        }
    }

    let basicSystem = {title:"基本制度", list:[]};
    basicSystem.list = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.支部制度表, {}, ["systemTitle"]);

    let totalIntegral = {title:"学习强国", monthTotalScore:0, list:[]};
    let totalScore = 0;
    for(let key in enums.BRANCHNAME) {
        let anyKey:any = key;
        if (isNaN(anyKey)) {
            let keyStr = key;
            let bId = enums.BRANCHNAME[key];
            let totalIntegralInfo = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.学习强国, {bId}, ["totalIntegral"]);
            let changeScore = 0;
            totalIntegralInfo.forEach(item => {
                changeScore += item.totalIntegral;
                totalScore += item.totalIntegral;
            })
            totalIntegral.list.push({key:keyStr, value:changeScore});
        }
    }
    totalIntegral.monthTotalScore = totalScore;

    let partyVanguard = {title:"党建先锋", list:[]};
    let partyVanguardInfo = await selectData(OPERATIONALDATATYPE.查询多个, TABLENAME.党建先锋表, {}, ["honorName", "grantTime"]);

    partyVanguardInfo.forEach( item => {
        let grantTime = moment(item.grantTime).format("YYYY-MM-DD");
        partyVanguard.list.push({honorName:item.honorName, grantTime});
    })

    return {activity, basicSystem, totalIntegral, partyVanguard};
}

