import { CreatePartyActivistsMemberConfig, CreatePartyCandidateMemberConfig, CreatePartyFormalMemberConfig, UpdatePartyActivistsMemberConfig, UpdatePartyCandidateMemberConfig, UpdatePartyFormalMemberConfig } from "../config/eccParam/eccParamConfig";
import { ADMINISTRATIVEPOSITION, BRANCHNAME, DEDPARTMENT, DEGREE, ENTRYTYPE, OPERATIONALDATATYPE, PARTYPOSITIONS, SEX, TABLENAME } from "../config/enum/enum";
import { selectData, operationalData } from "../data/operationalData";
import { changeEnumValue, eccEnumValue } from "../tools/eccEnum";
import { eccFormParam } from "../tools/eccParam";
import { datechangeToStr } from "../util/piecemeal";
import { getClientMs, getDefPwd, getMySqlMs, getPartyMemberId } from "../tools/systemTools";
import { BizError } from "../util/bizError";
import { ERRORENUM } from "../config/enum/errorEnum";


/**
 * 正式党员列表
 * @param branch 
 * @param name 
 * @param pageNumber 
 * @param entryType 类型，参考枚举 ENTRYTYPE
 * @returns 
 */
export async function formalList(branch:number, name:string, pageNumber:number) {
    let param:any = {etId:ENTRYTYPE.正式党员};
    if (branch) param.bId = branch;
    if (name) param.name = {"%like%":name};
    
    let column = ["pmId", "name", "sex", "birthDate", "phone", "levelOfEducation", "partyJoiningTime", "administrativePositionName", "partyPositionsName", "departmentName", "isAdmin"];

    let dataServerList = await selectData(OPERATIONALDATATYPE.分页查询, TABLENAME.党员基础信息表, param, column, pageNumber, 10 );
    let dataList = [];
    dataServerList.forEach(info => {
        let itemInfo = JSON.parse(JSON.stringify(info));
        itemInfo.administrativePositionName = changeEnumValue(ADMINISTRATIVEPOSITION, info.administrativePositionName);
        itemInfo.partyPositionsName = changeEnumValue(PARTYPOSITIONS, info.partyPositionsName);
        itemInfo.departmentName = changeEnumValue(DEDPARTMENT, info.departmentName);
        itemInfo.branchName = changeEnumValue(BRANCHNAME, itemInfo.bId);
        itemInfo.sex = changeEnumValue(SEX, info.sex);
        itemInfo.levelOfEducation = changeEnumValue(DEGREE, info.levelOfEducation);
        
        itemInfo.partyJoiningTime = datechangeToStr(itemInfo.partyJoiningTime);
        itemInfo.birthDate = datechangeToStr(itemInfo.birthDate);

        dataList.push(itemInfo);
    });
    let dataCount = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.党员基础信息表, param, null);
    
    return {dataList, dataCount};
}


/**
 * 积极分子列表
 * @param branch 
 * @param name 
 * @param pageNumber 
 * @param entryType 类型，参考枚举 ENTRYTYPE
 * @returns 
 */
export async function activistsList(branch:number, name:string, pageNumber:number) {
    let param:any = {etId:ENTRYTYPE.入党积极分子};
    if (branch) param.bId = branch;
    if (name) param.name = {"%like%":name};
    
    let column = ["pmId", "name", "birthDate", "phone", "sex", "levelOfEducation", "bId", "askForTime", "listedAsActivistsTime", "administrativePositionName", "liaison", "departmentName" ];
    
    let dataServerList = await selectData(OPERATIONALDATATYPE.分页查询, TABLENAME.党员基础信息表, param, column, pageNumber, 10 );
    let dataList = [];
    dataServerList.forEach(info => {
        let itemInfo = JSON.parse(JSON.stringify(info));
        itemInfo.administrativePositionName = changeEnumValue(ADMINISTRATIVEPOSITION, info.administrativePositionName);
        itemInfo.departmentName = changeEnumValue(DEDPARTMENT, info.departmentName);
        itemInfo.branchName = changeEnumValue(BRANCHNAME, itemInfo.bId);
        itemInfo.levelOfEducation = changeEnumValue(DEGREE, info.levelOfEducation);
        itemInfo.sex = changeEnumValue(SEX, info.sex);

        itemInfo.askForTime = datechangeToStr(info.askForTime);
        itemInfo.listedAsActivistsTime = datechangeToStr(info.listedAsActivistsTime);
        itemInfo.birthDate = datechangeToStr(info.birthDate);

        dataList.push(itemInfo);
    });
    let dataCount = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.党员基础信息表, param, null);

    return {dataList, dataCount};
}


/**
 * 预备党员列表
 * @param branch 
 * @param name 
 * @param pageNumber 
 * @param entryType 类型，参考枚举 ENTRYTYPE
 * @returns 
 */
export async function candidateList(branch:number, name:string, pageNumber:number) {
    let param:any = {etId:ENTRYTYPE.预备党员};
    if (branch) param.bId = branch;
    if (name) param.name = {"%like%":name};
    
    let column = ["pmId", "name", "sex", "birthDate", "phone", "levelOfEducation", "bId", "askForTime", "administrativePositionName", 
    "talkTime", "departmentName"];

    let dataServerList = await selectData(OPERATIONALDATATYPE.分页查询, TABLENAME.党员基础信息表, param, column, pageNumber, 10 );
    let dataList = [];
    dataServerList.forEach(info => {
        let itemInfo = JSON.parse(JSON.stringify(info));
        itemInfo.administrativePositionName = changeEnumValue(ADMINISTRATIVEPOSITION, info.administrativePositionName);
        itemInfo.departmentName = changeEnumValue(DEDPARTMENT, info.departmentName);
        itemInfo.sex = changeEnumValue(SEX, info.sex);
        itemInfo.branchName = changeEnumValue(BRANCHNAME, itemInfo.bId);
        itemInfo.levelOfEducation = changeEnumValue(DEGREE, info.levelOfEducation);
        
        itemInfo.askForTime = datechangeToStr(itemInfo.askForTime);
        itemInfo.talkTime = datechangeToStr(itemInfo.talkTime);
        itemInfo.birthDate = datechangeToStr(itemInfo.birthDate);

        dataList.push(itemInfo);
    });
    let dataCount = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.党员基础信息表, param, null);
    
    return {dataList, dataCount};
}


/**
 * 入党积极分子----回显
 * @param pmId 
 */
export async function partyActivistsMemberInfomation(pmId:string) {
    let param:any = {pmId};
    
    let column = ["pmId","name", "sex", "birthDate", "phone", "levelOfEducation", "bId", "askForTime", "listedAsActivistsTime", "departmentName", "administrativePositionName",  "liaison"];

    let dataInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.党员基础信息表, param, column );

    if (dataInfo.pmId) {
        dataInfo.departmentName = JSON.parse(dataInfo.departmentName);
        
        dataInfo.birthDate = getClientMs(dataInfo.birthDate);
        dataInfo.askForTime = getClientMs(dataInfo.askForTime);
        dataInfo.listedAsActivistsTime = getClientMs(dataInfo.listedAsActivistsTime);
    }

    return dataInfo;
}



/**
 * 正式党员----回显
 * @param pmId 
 */
export async function partyFormalMemberInfomation(pmId:string) {
    let param:any = {pmId};
    
    let column = ["pmId", "name", "sex", "birthDate", "phone", "levelOfEducation", "bId", "partyJoiningTime", "administrativePositionName", "departmentName", "partyPositionsName"];

    let dataInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.党员基础信息表, param, column );

    if (dataInfo.pmId) {
        dataInfo.departmentName = JSON.parse(dataInfo.departmentName);
        dataInfo.partyPositionsName = JSON.parse(dataInfo.partyPositionsName);

        dataInfo.birthDate = getClientMs(dataInfo.birthDate);
        dataInfo.partyJoiningTime = getClientMs(dataInfo.partyJoiningTime);
    }

    return dataInfo;
}


/**
 * 预备党员----回显
 * @param pmId 
 */
export async function partyCandidateMemberInfomation(pmId:string) {
    let param:any = {pmId};
    
    let column = ["pmId","name", "sex", "birthDate", "phone", "levelOfEducation", "bId", "askForTime", "talkTime", "administrativePositionName", "departmentName"];

    let dataInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.党员基础信息表, param, column );

    if (dataInfo.pmId) {
        dataInfo.departmentName = JSON.parse(dataInfo.departmentName);

        dataInfo.birthDate = getClientMs(dataInfo.birthDate);
        dataInfo.askForTime = getClientMs(dataInfo.askForTime);
        dataInfo.talkTime = getClientMs(dataInfo.talkTime);
    }

    return dataInfo;
}


/**
 * 修改积极分子信息
 * @param pmId 
 * @param param 
 */
export async function updatePartyActivistsMemberData(pmId:string, param) {
    eccFormParam('修改积极分子信息', UpdatePartyActivistsMemberConfig, param);
    /**校验参数 */
    eccEnumValue("修改积极分子信息", "sex", SEX, param.sex);
    eccEnumValue("修改积极分子信息", "bId", BRANCHNAME, param.bId);
    eccEnumValue("修改积极分子信息", "departmentName", DEDPARTMENT, param.departmentName);
    eccEnumValue("修改积极分子信息", "administrativePositionName", ADMINISTRATIVEPOSITION, param.administrativePositionName);
    eccEnumValue("修改党员基础信息", "levelOfEducation", DEGREE, param.levelOfEducation);
    /**多个枚举转换成json */
    param.departmentName = JSON.stringify(param.departmentName);
    /**时间戳转换成Date格式 */
    param.birthDate = getMySqlMs(param.birthDate);
    param.askForTime = getMySqlMs(param.askForTime);
    param.listedAsActivistsTime = getMySqlMs(param.listedAsActivistsTime);
    await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.党员基础信息表, param, {pmId});
    
    return {isSuccess:true};
}


/**
 * 修改党员基础信息
 * @param pmId 
 * @param param 
 */
export async function updatePartyFormalMemberData(pmId:string, param) {
    eccFormParam(`修改党员基础信息`, UpdatePartyFormalMemberConfig, param);
    /**校验枚举 */
    eccEnumValue("修改党员基础信息", "sex", SEX, param.sex);
    eccEnumValue("修改党员基础信息", "bId", BRANCHNAME, param.bId);
    eccEnumValue("修改党员基础信息", "administrativePositionName", ADMINISTRATIVEPOSITION, param.administrativePositionName);
    eccEnumValue("修改党员基础信息", "departmentName", DEDPARTMENT, param.departmentName);
    eccEnumValue("修改党员基础信息", "partyPositionsName", PARTYPOSITIONS, param.partyPositionsName);
    eccEnumValue("修改党员基础信息", "levelOfEducation", DEGREE, param.levelOfEducation);
    /**多个枚举转换成json */
    param.partyPositionsName = JSON.stringify(param.partyPositionsName);
    param.departmentName = JSON.stringify(param.departmentName);
    /**时间戳转换成Date格式 */
    param.birthDate = getMySqlMs(param.birthDate);
    param.partyJoiningTime = getMySqlMs(param.partyJoiningTime);
    await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.党员基础信息表, param, {pmId});
    
    return {isSuccess:true};
}


/**
 * 修改预备党员基础信息
 * @param pmId 
 * @param param 
 */
export async function updatePartyCandidateMemberData(pmId:string, param) {
    eccFormParam(`修改预备党员基础信息`, UpdatePartyCandidateMemberConfig, param);
    /**校验枚举 */
    eccEnumValue("修改预备党员基础信息", "sex", SEX, param.sex);
    eccEnumValue("修改预备党员基础信息", "bId", BRANCHNAME, param.bId);
    eccEnumValue("修改预备党员基础信息", "administrativePositionName", ADMINISTRATIVEPOSITION, param.administrativePositionName);
    eccEnumValue("修改预备党员基础信息", "departmentName", DEDPARTMENT, param.departmentName);
    eccEnumValue("修改预备党员基础信息", "levelOfEducation", DEGREE, param.levelOfEducation);
    /**多个枚举转换成json */
    param.departmentName = JSON.stringify(param.departmentName);
    /**时间戳转换成Date格式 */
    param.birthDate = getMySqlMs(param.birthDate);
    param.askForTime = getMySqlMs(param.askForTime);
    param.talkTime = getMySqlMs(param.talkTime);
    await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.党员基础信息表, param, {pmId});
    
    return {isSuccess:true};
}


/**
 * 创建预备党员
 * @param userInfo 
 * @param param 
 * @returns 
 */
export async function createCandidateMember(userInfo, param) {
    eccFormParam(`创建预备党员`, CreatePartyCandidateMemberConfig, param);
    /**校验枚举 */
    eccEnumValue("创建预备党员", "sex", SEX, param.sex);
    eccEnumValue("创建预备党员", "bId", BRANCHNAME, param.bId);
    eccEnumValue("创建预备党员", "administrativePositionName", ADMINISTRATIVEPOSITION, param.administrativePositionName);
    eccEnumValue("创建预备党员", "departmentName", DEDPARTMENT, param.departmentName);
    eccEnumValue("创建预备党员", "levelOfEducation", DEGREE, param.levelOfEducation);
    /**多个枚举转换成json */
    param.departmentName = JSON.stringify(param.departmentName);
    /**时间戳转换成Date格式 */
    param.birthDate = getMySqlMs(param.birthDate);
    param.askForTime = getMySqlMs(param.askForTime);
    param.talkTime = getMySqlMs(param.talkTime);
    param.etId = ENTRYTYPE.预备党员;
    param.pmId = getPartyMemberId(userInfo.userId);

    param.loginId = param.phone;
    param.pwd = getDefPwd(param.phone);

    await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.党员基础信息表, param, null);
    
    return {isSuccess:true};
}


/**
 * 创建党员信息
 * @param userInfo 
 * @param param 
 * @returns 
 */
export async function createFormalMember(userInfo, param) {
    eccFormParam(`创建党员信息`, CreatePartyFormalMemberConfig, param);
    /**校验枚举 */
    eccEnumValue("修改党员基础信息", "sex", SEX, param.sex);
    eccEnumValue("修改党员基础信息", "bId", BRANCHNAME, param.bId);
    eccEnumValue("修改党员基础信息", "administrativePositionName", ADMINISTRATIVEPOSITION, param.administrativePositionName);
    eccEnumValue("修改党员基础信息", "departmentName", DEDPARTMENT, param.departmentName);
    eccEnumValue("修改党员基础信息", "partyPositionsName", PARTYPOSITIONS, param.partyPositionsName);
    eccEnumValue("修改党员基础信息", "levelOfEducation", DEGREE, param.levelOfEducation);
    /**多个枚举转换成json */
    param.partyPositionsName = JSON.stringify(param.partyPositionsName);
    param.departmentName = JSON.stringify(param.departmentName);
    /**时间戳转换成Date格式 */
    param.birthDate = getMySqlMs(param.birthDate);
    param.partyJoiningTime = getMySqlMs(param.partyJoiningTime);
    param.etId = ENTRYTYPE.正式党员;
    param.pmId = getPartyMemberId(userInfo.userId);

    param.loginId = param.phone;
    param.pwd = getDefPwd(param.phone);

    await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.党员基础信息表, param, null);
    
    return {isSuccess:true};
}


/**
 * 创建积极分子信息
 * @param userInfo 
 * @param param 
 * @returns 
 */
export async function createActivistsMember(userInfo, param) {
    let eccTitle = '创建积极分子信息';
    eccFormParam(eccTitle, CreatePartyActivistsMemberConfig, param);
    /**校验参数 */
    eccEnumValue("修改积极分子信息", "sex", SEX, param.sex);
    eccEnumValue("修改积极分子信息", "bId", BRANCHNAME, param.bId);
    eccEnumValue("修改积极分子信息", "departmentName", DEDPARTMENT, param.departmentName);
    eccEnumValue("修改积极分子信息", "administrativePositionName", ADMINISTRATIVEPOSITION, param.administrativePositionName);
    eccEnumValue("修改党员基础信息", "levelOfEducation", DEGREE, param.levelOfEducation);
    /**多个枚举转换成json */
    param.departmentName = JSON.stringify(param.departmentName);
    /**时间戳转换成Date格式 */
    param.birthDate = getMySqlMs(param.birthDate);
    param.askForTime = getMySqlMs(param.askForTime);
    param.listedAsActivistsTime = getMySqlMs(param.listedAsActivistsTime);
    param.etId = ENTRYTYPE.入党积极分子;
    param.pmId = getPartyMemberId(userInfo.userId);

    param.loginId = param.phone;
    param.pwd = getDefPwd(param.phone);

    await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.党员基础信息表, param, null);
    
    return {isSuccess:true};
}


/**
 * 给正式党员打开管理员权限
 * @param pmId
 * @returns 
 */
export async function openformalAdminState(userInfo, pmId) {
    if (!userInfo.isSuperAdmin) throw new BizError(ERRORENUM.权限不足, `${userInfo.userId}尝试修改管理员权限`);
    let formalInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.党员基础信息表, {pmId}, [] );
    if (!formalInfo) throw new BizError(ERRORENUM.用户不存在, `${userInfo.userId}尝试将${pmId}设为管理员`);
    if (formalInfo.etId != ENTRYTYPE.正式党员 ) throw new BizError(ERRORENUM.目标用户不是正式党员, `${userInfo.userId}尝试将${pmId}设为管理员`);

    await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.党员基础信息表, {isAdmin:true}, {pmId});

    return {isSuccess:true};
}


/**
 * 关闭管理员权限
 * @param pmId
 * @returns 
 */
export async function closeformalAdminState(userInfo, pmId:string) {
    if (!userInfo.isSuperAdmin) throw new BizError(ERRORENUM.权限不足, `${userInfo.userId}尝试修改管理员权限`);
    let formalInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.党员基础信息表, {pmId}, [] );
    if (!formalInfo) throw new BizError(ERRORENUM.用户不存在, `${userInfo.userId}尝试将${pmId}设为管理员`);
    if (formalInfo.etId != ENTRYTYPE.正式党员 ) throw new BizError(ERRORENUM.目标用户不是正式党员, `${userInfo.userId}尝试将${pmId}设为管理员`);

    await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.党员基础信息表, {isAdmin:false}, {pmId});

    return {isSuccess:true};
}

export async function deleteMember(userInfo, pmId:string) {
    let formalInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.党员基础信息表, {pmId}, [] );
    if (!formalInfo) throw new BizError(ERRORENUM.用户不存在, `${userInfo.userId}尝试将${pmId}设删除`);
    if (!userInfo.isSuperAdmin && formalInfo.isAdmin) {
        throw new BizError(ERRORENUM.权限不足, `${userInfo.userId}尝试将${pmId}删除`);
    }

    await operationalData(OPERATIONALDATATYPE.删除, TABLENAME.党员基础信息表, null, {pmId});

    return {isSuccess:true};
}