/**
 * 支部制度
 */

import { BRANCHNAME, OPERATIONALDATATYPE, TABLENAME } from "../config/enum/enum";
import { ERRORENUM } from "../config/enum/errorEnum";
import { operationalData, selectData } from "../data/operationalData";
import { changeEnumValue, eccEnumValue } from "../tools/eccEnum";
import { getMySqlMs } from "../tools/systemTools";
import { BizError } from "../util/bizError";
import { dataListItemTimeChangeToTimeStr, datechangeToStr } from "../util/piecemeal";


/**
 * 支部制度列表
 * @param userInfo 
 * @param title 
 * @param pageNumber 
 * @returns 
 */
export async function getBranchSystemList(userInfo, title:string, pageNumber:number) {
    let selectParam:any = {};
    if (userInfo.branch) {
        selectParam.bId = {"%like%":userInfo.branch};
    }
    if (title) {
        selectParam.systemTitle = {"%like%":title};
    }

    let column = ["systemTitle", "bsId", "releaseTime", "isUse", "bId"];
    let dbList = await selectData(OPERATIONALDATATYPE.分页查询, TABLENAME.支部制度表, selectParam, column, pageNumber, 10 );
    let dataCount = await selectData(OPERATIONALDATATYPE.查询数据量, TABLENAME.支部制度表, selectParam, null );
    
    let dataList = [];
    dbList.forEach(info => {
        let {systemTitle, bsId, releaseTime, isUse, bId} = info;
        let releaseTimeStr = datechangeToStr(releaseTime);
        let branch = changeEnumValue(BRANCHNAME, bId);
        dataList.push({
            title:systemTitle, bsId, releaseTime:releaseTimeStr, isUse, branch
        });
    });

    return {dataList, dataCount};
}


/** 
 * 添加支部制度
 * @param userInfo 
 * @param title 
 * @param branchList 
 * @param filesNameList 
 */
export async function addBranchSystemInfo(userInfo, title:string, branchList, filesNameList) {
    let funName = "添加支部制度";
    eccEnumValue(funName, 'branchList', BRANCHNAME, branchList);
    if (userInfo.branch && branchList.indexOf(userInfo.branch) == -1) {
        throw new BizError(ERRORENUM.权限不足, `${userInfo.userId}${funName}但是制度范围不包含他所在的支部`);
    }

    let addParam = {
        systemTitle:title, 
        bId:JSON.stringify(branchList),
        isUse:true, 
        releaseTime:getMySqlMs(),
        fileName:filesNameList,
        fileType:'pdf',
        uploadTime:getMySqlMs()
    };

    await operationalData(OPERATIONALDATATYPE.增加, TABLENAME.支部制度表, addParam, {});

    return {isSuccess:true};
}


/**
 * 修改
 * @param userInfo 
 * @param bsId 
 * @param title 
 * @param branchList 
 * @param filesNameList 
 */
export async function updateBranchSystemInfo(userInfo, bsId:number, title:string, branchList, filesNameList) {
    let funName = "修改支部制度";
    eccEnumValue(funName, 'branchList', BRANCHNAME, branchList);
    if (userInfo.branch && branchList.indexOf(userInfo.branch) == -1) {
        throw new BizError(ERRORENUM.权限不足, `${userInfo.userId}${funName}但是制度范围不包含他所在的支部`);
    }

    //todo 这里有争议 修改时不知道怎么处理文件  修改先没处理文件

    let updateInfo = {
        systemTitle:title, 
        bId:JSON.stringify(branchList)
    };

    await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.支部制度表, updateInfo, {bsId});

    return {isSuccess:true};
}


/**
 * 回显
 * @param userInfo 
 * @param bsId 
 */
export async function getBranchSystemInfo(userInfo, bsId) {
    let funName = "回显支部制度";

    let column = ["systemTitle", "bsId", "bId"];
    let dataInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.支部制度表, {bsId}, column);
    
    let branch = JSON.parse(dataInfo.bId);
    if (userInfo.branch && branch.indexOf(userInfo.branch) == -1 ) {
        throw new BizError(ERRORENUM.权限不足, `${userInfo.userId}${funName}但是制度范围不包含他所在的支部`);
    }

    return {title:dataInfo.systemTitle, branch };
}


/**
 * 修改制度状态
 * @param userInfo 
 * @param bsId 
 * @param state 
 * @returns 
 */
export async function changeBranchSystemState(userInfo, bsId:number, state:boolean) {
    let funName = "修改支部制度状态";
    let dataInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.支部制度表, {bsId}, []);
    
    if (!dataInfo || !dataInfo.bsId) throw new BizError(ERRORENUM.目标数据不存在, `${userInfo.userId}${funName}但是${bsId}已不存在`);
    if (userInfo.branch && JSON.parse(dataInfo.bId).indexOf(userInfo.branch) == -1 ) {
        throw new BizError(ERRORENUM.权限不足, `${userInfo.userId}${funName}但是制度范围不包含他所在的支部`);
    }

    await operationalData(OPERATIONALDATATYPE.修改, TABLENAME.支部制度表, {isUse:state}, {bsId});

    return {isSuccess:true};
}


export async function delBranchSystemData(userInfo, bsId:number) {
    let funName = "删除支部制度";
    let dataInfo = await selectData(OPERATIONALDATATYPE.查询单个, TABLENAME.支部制度表, {bsId}, []);
    if (!dataInfo || !dataInfo.bsId) throw new BizError(ERRORENUM.目标数据不存在, `${userInfo.userId}${funName}但是${bsId}已不存在`);
    if (userInfo.branch && JSON.parse(dataInfo.bId).indexOf(userInfo.branch) == -1 ) {
        throw new BizError(ERRORENUM.权限不足, `${userInfo.userId}${funName}但是制度范围不包含他所在的支部`);
    }

    //todo  这里还要先删除相关文件

    await operationalData(OPERATIONALDATATYPE.删除, TABLENAME.支部制度表, {}, {bsId});
    
    return {isSuccess:true};
}