/**
 * 标签系统
 */

import moment = require("moment");
import { TABLEENUM } from "../../data/models/model";
import { find, findCount, findOnce, findToPage, findToSortToPage } from "../../data/select";
import { extractData } from "../../util/piecemeal";
import { BizError } from "../../util/bizError";
import { generateSystemId, successResult } from "../../tools/system";
import { addOneData } from "../../data/add";
import { ERRORENUM } from "../../config/errorEnum";
import { updateOneData } from "../../data/update";
import { deleteOneData } from "../../data/delete";


/**
 * 标签系统列表
 * @param lableName 标签名称
 * @param createTime 创建时间
 * @param pageNumber 当前页
 */
export async function lableList({lableName, startCreateTime, endCreateTime, pageNumber}) {
    let selectParam:any = {};
    if (lableName) selectParam.lableName = lableName;
    if (startCreateTime) selectParam.createTime = {"$gt":startCreateTime};
    if (endCreateTime) {
        if (!selectParam["createTime"]) selectParam["createTime"] = {};
        selectParam["createTime"]["$lt"] = endCreateTime;
    }

    const SelectFiles = ["lableId", "lableName", "createTime"];
    let dbList = await findToSortToPage(TABLEENUM.标签表, selectParam, SelectFiles, {createTime:-1}, pageNumber);
    let dataCount = await findCount(TABLEENUM.标签表, selectParam);

    let dataList = [];
    dbList.forEach( info => {
        let item:any = extractData(info, SelectFiles);
        item.createTime = moment(item.createTime).format("YYYY-MM-DD");
        dataList.push(item);
    });

    return {dataList, dataCount};
}


/**
 * 新建标签
 * @param id
 * @param lableName 标签名称
 * @returns 
 */
export async function createLable({userId, lableName}) {
    let lableInfo = await findOnce(TABLEENUM.标签表, {lableName}, ["lableId"]);
    if (lableInfo) throw new BizError(ERRORENUM.标签重复);
    /**标签Id */
    const lableId = generateSystemId(TABLEENUM.标签表, userId);
    
    let addInfo = {
        lableId,
        lableName,
        createTime:new Date().valueOf()
    };

    await addOneData(TABLEENUM.标签表, addInfo);

    return successResult();
}


/**
 * 修改标签名称
 * @param lableId 标签id
 * @param lableName 标签名称
 */
export async function updateLable({lableId, lableName}) {
    let lableInfo = await findOnce(TABLEENUM.标签表, {lableId}, ["lableId"]);
    if (!lableInfo || !lableInfo.lableId) throw new BizError(ERRORENUM.标签数据不存在);

    await updateOneData(TABLEENUM.标签表, {lableId}, {lableName});

    return successResult();
}


/**
 * 删除标签
 * @param lableId 标签Id 
 */
export async function delLable({lableId}) {
    let lableInfo = await findOnce(TABLEENUM.标签表, {lableId}, ["lableId"]);
    if (!lableInfo || !lableInfo.lableId) throw new BizError(ERRORENUM.标签数据不存在);
    
    await deleteOneData(TABLEENUM.标签表, {lableId});
    return successResult();
}


/**
 * 添加标签只显示未选中标签数据
 * @param id 选中需要添加标签的会员id 
 */
export async function unCheckedLable({id}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId:id}, ["userId", "lableId"]);
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    
    let lableInfo = await find(TABLEENUM.标签表, {lableId:{"$nin":userInfo.lableId}}, ["lableId", "lableName"]);

    return lableInfo;
}


/**
 * 个人会员-分配标签
 * @param id 会员id
 * @param lableId 选中新增的标签id 多选 ["", ""]
 */
export async function addMemberLable({id, lableId}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId:id}, ["userId", "lableId"]);
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    
    let lableInfo = userInfo.lableId || [];
    for (let i = 0; i < lableId.length; i++) {
        lableInfo.push(lableId[i]);
        console.log();
    }
    await updateOneData(TABLEENUM.用户表, {userId:id}, {lableId:lableInfo});

    return successResult();
}


/**
 * 删除标签只显示已选中标签数据
 * @param id 选中需要添加标签的会员id 
 */
export async function checkedLable({id}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId:id}, ["userId", "lableId"]);
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    
    let lableInfo = await find(TABLEENUM.标签表, {lableId:{"$in":userInfo.lableId}}, ["lableId", "lableName"]);

    return lableInfo;
}


/**
 * 个人会员-删除标签
 * @param id 会员id 
 * @param lableId 需要删除的标签Id 多选 ["", ""]
 */
export async function delMemberLable({id, lableId}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId:id}, ["userId", "lableId"]);
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    let lableInfo = userInfo.lableId;
    let newLable = lableInfo.filter(item => !lableId.includes(item));

    await updateOneData(TABLEENUM.用户表, {userId:id}, {lableId:newLable});
    
    return successResult();
}







