/**
 * 审批逻辑 success
 */

import moment = require("moment");
import { INDIVIDUALMEMBERTYPE, MEMBERLEVEL, MEMBERTYPE, OPERATIONREHAVIOR, PAYMENTSTATUS, REGISTERFLOW, STATE, UNITMEMBERTYPE, USERREGISTERSTATE } from "../../config/enum";
import { ERRORENUM } from "../../config/errorEnum";
import { addManyData } from "../../data/add";
import { TABLEENUM } from "../../data/models/model";
import { find, findCount, findOnce, findToPage } from "../../data/select";
import { updateOneData } from "../../data/update";
import { generateSystemId, successResult } from "../../tools/system";
import { BizError } from "../../util/bizError";
import { extractData } from "../../util/piecemeal";
import { changeEnumValue } from "../../util/verificationEnum";
import { createOrder } from "./order";

/**
 * 待审核列表 success
 * @param unitName 单位名称
 * @param joinTime 入会时间
 * @param memberType 会员类型 MEMBERTYPE
 * @param sheng 省
 * @param shi 市
 * @param qu 区
 * @param pageNumber 当前页 
 */
export async function pendingReviewList({unitName, joinTime, memberType, sheng, shi, qu, pageNumber}) {
    /**查询条件 */
    let selectParam:any = {userRegisterState:{"$ne": USERREGISTERSTATE.通过}}; //审批列表展示状态除通过外的数据
    if (unitName) selectParam.unitName = unitName;
    if (joinTime) selectParam.joinTime = joinTime;
    if (memberType) selectParam.memberType = memberType;
    if (sheng) selectParam.sheng = sheng;
    if (shi) selectParam.shi = shi;
    if (qu) selectParam.qu = qu;

    const SelectFiles = ["userId", "userRegisterState", "unitName", "loginId", "phone", "askForTime", "memberLevel", "sheng", "shi", "qu"];
    let dbList = await findToPage(TABLEENUM.用户表, selectParam, SelectFiles, pageNumber);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    dbList.forEach(info => {
        let item:any = extractData(info, SelectFiles);
        /**处理枚举值 */
        if (!item.userRegisterState) item.userRegisterState = USERREGISTERSTATE.待审核;
        else item.userRegisterState = changeEnumValue(USERREGISTERSTATE, item.userRegisterState);
        item.memberLevel = changeEnumValue(MEMBERLEVEL, item.memberLevel);
        dataList.push(item);
    });

    return {dataList, dataCount};
}


/**
 * 通过审核 success
 * @param id 待审核列表返回的用户id 
 * @returns 
 */
export async function adopt({id, session}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId:id, isAdmin:STATE.否, session}, ["userId", "registerFlow", "userRegisterState", "memberType", "unitName", "name"]);
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    if (userInfo.memberType == MEMBERTYPE.个人会员 && userInfo.registerFlow != REGISTERFLOW.完成第二步) {
        throw new BizError(ERRORENUM.不具备审批条件, `${userInfo.userId}为个人会员，并没有完成第二步`);
    }
    if (userInfo.memberType == MEMBERTYPE.单位会员 && userInfo.registerFlow != REGISTERFLOW.完成第三步) {
        throw new BizError(ERRORENUM.不具备审批条件, `${userInfo.userId}为单位会员，并没有完成第三步`);
    }
    if (userInfo.userRegisterState == USERREGISTERSTATE.驳回修改) {
        throw new BizError(ERRORENUM.不具备审批条件, `${userInfo.userId}当前状态为驳回修改`);
    }
    if (userInfo.userRegisterState == USERREGISTERSTATE.通过) {
        throw new BizError(ERRORENUM.重复提交, `${userInfo.userId}当前状态为已通过`);
    }

    let thisDate = new Date();
    const NowMs = thisDate.valueOf();
    const thisYear = thisDate.getFullYear();
    const thisMonth = thisDate.getMonth() + 1;
    let addInfo:any = {
        auditTime:NowMs,
        joinTime:NowMs,
        userRegisterState:USERREGISTERSTATE.通过,
        session:getEdition(thisYear, thisMonth),
        paymentStatus:PAYMENTSTATUS.未支付,
        lifespanStartTime:NowMs,
        lifespanEndTime:NowMs + 1000,
        memberLevel:MEMBERLEVEL.普通会员,
    };
    let unitName = userInfo.unitName;
    let subType = userInfo.unitMemberType;
    if (userInfo.memberType == MEMBERTYPE.个人会员) {
        addInfo.individualMemberType = INDIVIDUALMEMBERTYPE.个人会员;
        unitName = userInfo.name;
        subType = userInfo.individualMemberType;
    }

    await updateOneData(TABLEENUM.用户表, {userId:id}, addInfo);

    //todo 创建一条初始订单
    
    await createOrder(id, unitName, NowMs, MEMBERLEVEL.普通会员, userInfo.memberType, subType, true);

    /**提交审批历史信息 */
    let addApprovalHistory = {
        id:generateSystemId(TABLEENUM.审批历史表, id),
        userId:id,
        operationTime:NowMs,
        operationBehavior:OPERATIONREHAVIOR.通过,
        remarks:"通过"
    };
    await addManyData(TABLEENUM.审批历史表, addApprovalHistory);

    return successResult();
}


/**
 * 驳回 success
 * @param userId 
 * @param remarks 驳回理由
 * @returns 
 */
export async function reject({userId, remarks}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId, isAdmin:STATE.否});
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    if (userInfo.memberType == MEMBERTYPE.个人会员 && userInfo.registerFlow != REGISTERFLOW.完成第二步) {
        throw new BizError(ERRORENUM.不具备审批条件, `${userInfo.userId}为个人会员，并没有完成第二步`);
    }
    if (userInfo.memberType == MEMBERTYPE.单位会员 && userInfo.registerFlow != REGISTERFLOW.完成第三步) {
        throw new BizError(ERRORENUM.不具备审批条件, `${userInfo.userId}为单位会员，并没有完成第三步`);
    }
    if (userInfo.userRegisterState == USERREGISTERSTATE.驳回修改) {
        throw new BizError(ERRORENUM.不具备审批条件, `${userInfo.userId}当前状态为驳回修改`);
    }
    if (userInfo.userRegisterState == USERREGISTERSTATE.通过) {
        throw new BizError(ERRORENUM.重复提交, `${userInfo.userId}当前状态为已通过`);
    }

    /**驳回修改，修改该会员注册流程回到第二步 */
    let updateParam = {
        userRegisterState:USERREGISTERSTATE.驳回修改, 
        registerFlow:REGISTERFLOW.完成第一步
    }
    await updateOneData(TABLEENUM.用户表, {userId}, updateParam);
    
    /**提交审批历史信息 */
    const NowMs = new Date().valueOf(); 
    let addApprovalHistory = {
        id:generateSystemId(TABLEENUM.审批历史表, userId),
        userId,
        operationTime:NowMs,
        operationBehavior:OPERATIONREHAVIOR.驳回修改,
        remarks
    };
    await addManyData(TABLEENUM.审批历史表, addApprovalHistory);

    return successResult();
}


/**
 * 审核历史 success
 * @param userId
 * @returns 
 */
export async function approvalHistoryList({userId}) {
    let dbList = await find(TABLEENUM.审批历史表, {userId});

    let dataList = [];
    dbList.forEach(info => {
        let item:any = extractData(info, ["operationTime", "operationBehavior", "remarks"]);
        item.operationBehavior = changeEnumValue(OPERATIONREHAVIOR, item.operationBehavior);
        dataList.push(item);
    });

    return {dataList};
}


/**
 * 获取下拉框选择届次
 * @returns 
 */
export async function getSession() {
    const NowMs = new Date();
    let nowYear = parseInt(moment(NowMs).format("YYYY"));
    let nowMonth = parseInt(moment(NowMs).format("MM"));

    let session = [];
    for (let i = 2008; i <= nowYear; i++) {
        session.push(getEdition(i, 1));
        if (i==nowYear) {
            if(nowMonth>6) session.push(getEdition(i, 7)); 
        }
        else session.push(getEdition(i, 7));
    }

    return {session}
}


/**
 * 获取届次
 * @param year 入会年份 
 * @param month 入会月份
 * 下拉选择起始届次：2008年第四届第一次
 * @returns 
 */
export function getEdition(year, month) {
    //起始年份和月份 第一届第一次
    const startYear = 1993;

    let currentYear = new Date().getFullYear();
    if (year > currentYear) throw new BizError(ERRORENUM.无法获取未来届次);

    //计算从起始时间给定时间的总月份数
    let totalMonth = (year - startYear) * 12 + month;
    //每次是6个月
    let sessions = Math.ceil(totalMonth / 6);
    //每10次一届，从第四届开始计算
    let times = Math.ceil(sessions / 10);
    let sessionNumber = sessions - (times - 1) * 10 //届次是从第四届第一次开始，所以需要减一

    return `第${times}届,第${sessionNumber}次`;
}

