/**
 * 会费管理逻辑
 */

import moment = require("moment");
import { MEMBERLEVEL, MEMBERSTATE, MEMBERTYPE, MEMBERTYPEECCENUM, ORDEREXAMINE, ORDERSTATE, PAYMENTTYPE, RECEIPTCONFIRMATION, STATE, UNITMEMBERTYPE, USERREGISTERSTATE } from "../../config/enum";
import { TABLEENUM } from "../../data/models/model";
import { find, findCount, findOnce, findToPage, findToSort, findToSortToPage } from "../../data/select";
import { extractData } from "../../util/piecemeal";
import { changeEnumValue } from "../../util/verificationEnum";
import { BizError } from "../../util/bizError";
import { ERRORENUM } from "../../config/errorEnum";
import { updateOneData } from "../../data/update";
import { successResult } from "../../tools/system";
import { addOneData } from "../../data/add";



/**
 * 导出会费管理表格
 * @param param0 
 */
export async function outPutDataForExcel({type, exportColumns}) {
    const NowMs = new Date().valueOf(); 
    let selectParam:any = {};
    if (type == 1) {
         //未缴费
         selectParam = {
            "$or":[
                { 
                    isGracePeriod:STATE.否, 
                    lifespanEndTime:{"$lt":NowMs}
                }, 
                {
                    isGracePeriod:STATE.是, 
                    lifespanEndTime:{"$lt":NowMs},
                    gracePeriodEndTime:{"$lt":NowMs}
                }
            ],
            userRegisterState:USERREGISTERSTATE.通过,
            isAdmin:STATE.否,
        };
       
    } else if (type == 2) {
        //已缴费名单
        selectParam.lifespanEndTime = {"$gt":NowMs + (90 * 24 * 3600 * 1000)};
        selectParam.isAdmin = STATE.否;
        selectParam.isGracePeriod = STATE.否;
        selectParam.userRegisterState = USERREGISTERSTATE.通过;
    } else if (type == 3) {
        //缴费期
        selectParam.lifespanEndTime = {"$lt":NowMs + (90 * 24 * 3600 * 1000), "$gt":NowMs};
        selectParam.isAdmin = STATE.否;
        selectParam.isGracePeriod = STATE.否;
        selectParam.userRegisterState = USERREGISTERSTATE.通过;
    } else {
        //宽限期名单
        selectParam.isGracePeriod = STATE.是;
        selectParam.gracePeriodEndTime = {"$gt":NowMs};
        selectParam.isAdmin = STATE.否;
        selectParam.userRegisterState = USERREGISTERSTATE.通过
    }
    let selectFiles =["userId"];
    let titleList = [];
    let valueKeyList = [];
    let isSelectOrder = false;
    exportColumns.forEach(info => {
        let {key, value} = info;
        valueKeyList.push(value);
        selectFiles.push(value);
        if (value == "name") selectFiles.push("unitName");
        if (value == "Money" || value == "duration") isSelectOrder = true;
        if (value == "code") selectFiles.push("documentId", "uscc");
        titleList.push(key);
    });

    let dbList = await find(TABLEENUM.用户表, selectParam, selectFiles);
    let dataList = [titleList];
    for (let i = 0; i < dbList.length; i++) {
        let info = dbList[i];
        let subList = [];
        let orderItemList = [];
        if (isSelectOrder) {
            orderItemList = await find(TABLEENUM.订单表, {userId:info.userId}, ["orderCycleStart", "money", "state"]);
        }
        valueKeyList.forEach(key => {
            let item;
            if (key == "name") {
                item = info.name || info.unitName;
            } else if (key == "Money") {
                let itemCount = 0;
                // 已缴费和缴费期不累计金额
                if (type == 2 && type == 3) {
                    orderItemList.forEach(orderItem => {
                        itemCount = orderItem.money;
                    });
                } else {
                    orderItemList.forEach(orderItem => {
                        itemCount += orderItem.money;
                    });
                }
                item = itemCount;
            } else if (key == "paymentYear") {
                let 缴费Year = 0;
                // 已缴费和缴费期显示缴费年限
                if (type == 2 && type == 3) {
                    orderItemList.forEach(orderItem => {
                        if (orderItem.state == ORDERSTATE.已支付) {
                            缴费Year += 1;
                        }
                    });
                    item = 缴费Year;
                }
            } else if (key == "duration") {
                let itemCount = '';
                orderItemList.forEach(orderItem => {
                    itemCount += ` ${new Date(orderItem.orderCycleStart).getFullYear()}`;
                });
                item = itemCount;
            } else if (key == "durationYear") {
                let 欠费Year = 0;
                // 已缴费和缴费期显示欠费年限
                if (type == 2 && type == 3) {
                    orderItemList.forEach(orderItem => {
                        if (orderItem.state == ORDERSTATE.未支付) {
                            欠费Year += 1;
                        }
                    });
                    item = 欠费Year;
                }
            } else if (key == "code") {
                item = info.documentId || info.uscc;
            }else {
                item = info[key];
            }

            if (key == "memberState") item = changeEnumValue(MEMBERSTATE, item);
            if (key == "memberType") item = changeEnumValue(MEMBERTYPE, item);
            if (key == "memberLevel") item = changeEnumValue(MEMBERLEVEL, item);
            if (key == "joinTime") item = moment(item).format("YYYY-MM-DD");
            subList.push(item);
        });
        dataList.push(subList);
    }

    return {dataList};
}


/**
 * 已缴费列表 success
 * @param name 名称关键字：个人会员匹配真实姓名，单位会员匹配单位名称
 * @param memberType 会员类别 unitMemberType、individualMemberType  多选
 * 会员类别：个人【个人会员、专业会员】单位【院校、机构、其他】返回前端枚举：MEMBERTYPEECCENUM
 * @param documentId 身份证
 * @param phone 登录手机号
 * @param mail 单位/个人邮箱
 * @param joinStartTime 入会开始时间
 * @param joinEndTime 入会结束时间
 * @param memberLevel 会员级别/职务 多选
 * 会员级别/职务：【普通会员、理事会员、常务理事会员】
 * @param pageNumber 当前页
 */
export async function getPaid({name, memberType, documentId, phone, mail, joinStartTime, joinEndTime, memberLevel, pageNumber}) {
    const NowMs = new Date().valueOf();
    /**到期时间距离今天大于90天 */
    let selectParam:any = {
        lifespanEndTime:{"$gt":NowMs + (90 * 24 * 3600 * 1000)},
        isAdmin:STATE.否,
        isGracePeriod:STATE.否,
        userRegisterState:USERREGISTERSTATE.通过
    }

    if (name) {
        selectParam["$or"] = [
            {unitName:{"$regex":`${name}`}, memberType:MEMBERTYPE.单位会员},
            {name:{"$regex":`${name}`}, memberType:MEMBERTYPE.个人会员}
        ]
    }
    // if (memberType.length) {
    //     if (memberType < UNITMEMBERTYPE.院校) {
    //         selectParam.individualMemberType = {"$in":memberType};
    //     } else {
    //         selectParam.unitMemberType = {"$in":memberType};
    //     }
    // }
    if (memberType && memberType.length) selectParam["$or"] = [ {unitMemberType:{"$in":memberType}}, {individualMemberType:{"$in":memberType}} ];
    if (documentId) selectParam.documentId = documentId; 
    if (phone) selectParam.phone = phone;
    if (mail) selectParam.mail = mail;
    if (joinStartTime) {
        selectParam["joinTime"] = {"$gt":joinStartTime};
    }
    if (joinEndTime) {
        if (!selectParam["joinTime"]) selectParam["joinTime"] = {};
        selectParam["joinTime"]["$lt"] = joinEndTime;
    }
    if (memberLevel.length) selectParam.memberLevel = {"$in":memberLevel};

    const SelectFiles = ["userId", "memberState", "unitName", "name", "joinTime", "memberLevel", "individualMemberType", "unitMemberType", 
    "lifespanStartTime", "lifespanEndTime"];
    let dbList = await findToSortToPage(TABLEENUM.用户表, selectParam, SelectFiles, {joinTime:-1}, pageNumber);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    const itemFiles = ["userId", "memberState", "unitName", "name", "joinTime", "memberType", "memberLevel", "lifespanTime", "nextPayment", "money"];
    for (let i = 0; i < dbList.length; i++) {
        let info = dbList[i];
        let item:any = extractData(info, itemFiles);
        item.lifespanTime = `${moment(info.lifespanStartTime).format("YYYY-MM-DD")} 至 ${moment(info.lifespanEndTime).format("YYYY-MM-DD")}`; //会员有效期
        item.nextPayment = moment(info.lifespanEndTime).format("YYYY-MM-DD"); //下次缴费时间
        item.nextLifespanTime = "大于90天";
        item.joinTime = moment(item.joinTime).format("YYYY-MM-DD");
        item.memberState = changeEnumValue(MEMBERSTATE, item.memberState);
        item.memberLevel = changeEnumValue(MEMBERLEVEL, item.memberLevel);
        if(info.individualMemberType) {
            item.memberType = changeEnumValue(MEMBERTYPEECCENUM, info.individualMemberType);
        }
        if(info.unitMemberType) {
            item.memberType = changeEnumValue(MEMBERTYPEECCENUM, info.unitMemberType);
        }

        let orderDbList = await find(TABLEENUM.订单表, {userId:info.userId}, ["money"])
        orderDbList.forEach(orderItem => {
            item.money = orderItem.money;
        });

        dataList.push(item);
    };

    return {dataList, dataCount};
}

/**
 * 已缴费列表中的订单日志 success
 * @param id 会员列表返回的userId  
 */
export async function paidOrderList({id}) {
    let dbList = await findToSort(TABLEENUM.订单表, {userId:id}, {payTime:-1}, ["orderCycleStart", "orderCycleEnd", "money", "paymentMethod", "state"]);

    let dataList = [];
    dbList.forEach(info => {
        let {orderCycleStart, orderCycleEnd, money, paymentMethod, state} = info;
        let paymentMethodStr = '-';
        if ( state == ORDERSTATE.已支付 ) {
            paymentMethodStr = changeEnumValue(PAYMENTTYPE, paymentMethod);
        }
        dataList.push({
            orderCycle:`${moment(orderCycleStart).format("YYYY-MM-DD")}至${moment(orderCycleEnd).format("YYYY-MM-DD")}`,
            money,
            paymentMethod:paymentMethodStr,
            state:changeEnumValue(ORDERSTATE, state)
        });
    });

    return {dataList};
}


/**
 * 待缴费列表 success  缴费期列表
 * @param name 名称关键字：个人会员匹配真实姓名，单位会员匹配单位名称
 * @param memberType 会员类别 unitMemberType、individualMemberType  多选
 * 会员类别：个人【个人会员、专业会员】单位【院校、机构、其他】返回前端枚举：MEMBERTYPEECCENUM
 * @param documentId 身份证
 * @param phone 登录手机号
 * @param mail 单位/个人邮箱
 * @param joinStartTime 入会开始时间
 * @param joinEndTime 入会结束时间
 * @param memberLevel 会员级别/职务 多选
 * 会员级别/职务：【普通会员、理事会员、常务理事会员】
 * @param pageNumber 当前页 
 */
export async function unpaidList({name, memberType, documentId, phone, mail, joinStartTime, joinEndTime, memberLevel, pageNumber}) {
    const NowMs = new Date().valueOf();
    /**到期时间距离今天小于90天 */
    let selectParam:any = {
        lifespanEndTime:{"$lt":NowMs + (90 * 24 * 3600 * 1000), "$gt":NowMs},
        isAdmin:STATE.否,
        isGracePeriod:STATE.否,
        userRegisterState:USERREGISTERSTATE.通过
    }
    if (documentId) selectParam.documentId = documentId;
    if (phone) selectParam.phone = phone;
    if (mail) selectParam.mail = mail;
    if (memberLevel.length) selectParam.memberLevel = {"$in":memberLevel};
    if (name) {
        selectParam["$or"] = [
            {unitName:{"$regex":`${name}`}, memberType:MEMBERTYPE.单位会员},
            {name:{"$regex":`${name}`}, memberType:MEMBERTYPE.个人会员}
        ]
    }
    if (joinStartTime) {
        selectParam["joinTime"] = {"$gt":joinStartTime};
    }
    if (joinEndTime) {
        if (!selectParam["joinTime"]) selectParam["joinTime"] = {};
        selectParam["joinTime"]["$lt"] = joinEndTime;
    }
    // if (memberType.length) {
    //     if (memberType < UNITMEMBERTYPE.院校) {
    //         selectParam.individualMemberType = {"$in":memberType};
    //     } else {
    //         selectParam.unitMemberType = {"$in":memberType};
    //     }
    // }
    if (memberType && memberType.length) selectParam["$or"] = [ {unitMemberType:{"$in":memberType}}, {individualMemberType:{"$in":memberType}} ];

    // const SelectFiles = ["userId", "memberState", "unitName", "loginId", "phone", "joinTime", "memberType", "lifespanStartTime", "lifespanEndTime", "individualMemberType", 
    // "unitMemberType"];
    // let dbList = await findToPage(TABLEENUM.用户表, selectParam, SelectFiles, pageNumber, 10);
    const SelectFiles = ["userId", "memberState", "individualMemberType", "unitMemberType", "unitName", "name", "memberLevel", "joinTime", "lifespanStartTime", "lifespanEndTime"];
    let dbList = await findToSortToPage(TABLEENUM.用户表, selectParam, SelectFiles, {joinTime:-1}, pageNumber);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    // const itemFiles = ["userId", "memberState", "unitName", "loginId", "phone", "joinTime", "individualMemberType", "unitMemberType"];
    const itemFiles = ["userId", "memberState", "unitName", "name", "joinTime", "memberType", "memberLevel", "lifespanTime", "nextPayment", "money", "nextLifespanTime"];
    for (let i = 0; i < dbList.length; i++) {
        let info = dbList[i];
        let item:any = extractData(info, itemFiles);
        item.lifespanTime = `${moment(info.lifespanStartTime).format("YYYY-MM-DD")} 至 ${moment(info.lifespanEndTime).format("YYYY-MM-DD")}`; //会员有效期
        item.nextPayment = moment(info.lifespanEndTime).format("YYYY-MM-DD"); //下次缴费时间
        item.nextLifespanTime = "不足90天";
        if (info.lifespanEndTime < ( NowMs + (30 * 24 * 3600 * 1000) ) ) item.nextLifespanTime = "不足30天";
    
        item.joinTime = moment(item.joinTime).format("YYYY-MM-DD");
        item.memberState = changeEnumValue(MEMBERSTATE, item.memberState);
        item.memberLevel = changeEnumValue(MEMBERLEVEL, item.memberLevel);
        if(info.individualMemberType) {
            item.memberType = changeEnumValue(MEMBERTYPEECCENUM, info.individualMemberType);
        }
        if(info.unitMemberType) {
            item.memberType = changeEnumValue(MEMBERTYPEECCENUM, info.unitMemberType);
        }

        let orderDbList = await find(TABLEENUM.订单表, {userId:info.userId}, ["money"])
        // state 缴费状态
        orderDbList.forEach(orderItem => {
            item.money = orderItem.money;
        });

        dataList.push(item);
    };

    return {dataList, dataCount};
}


/**
 * 会费管理-欠费 success 未缴费
 * @param name 名称关键字：个人会员匹配真实姓名，单位会员匹配单位名称
 * @param memberType 会员类别 unitMemberType、individualMemberType  多选
 * 会员类别：个人【个人会员、专业会员】单位【院校、机构、其他】返回前端枚举：MEMBERTYPEECCENUM
 * @param documentId 身份证
 * @param phone 登录手机号
 * @param mail 单位/个人邮箱
 * @param joinStartTime 入会开始时间
 * @param joinEndTime 入会结束时间
 * @param memberLevel 会员级别/职务 多选
 * 会员级别/职务：【普通会员、理事会员、常务理事会员】
 * @param pageNumber 当前页
 */
export async function getRenewalPeriod({name, memberType, documentId, phone, mail, joinStartTime, joinEndTime, memberLevel, pageNumber}) {
    const NowMs = new Date().valueOf();
    let selectParam:any = {
        "$or":[
            { 
                isGracePeriod:STATE.否, 
                lifespanEndTime:{"$lt":NowMs}
            }, 
            {
                isGracePeriod:STATE.是, 
                lifespanEndTime:{"$lt":NowMs},
                gracePeriodEndTime:{"$lt":NowMs}
            }
        ],
        userRegisterState:USERREGISTERSTATE.通过,
        isAdmin:STATE.否,
    }
    if (name){
        if (!selectParam["$and"]) selectParam["$and"] = [];
        selectParam["$and"].push({
            "$or":[
                {unitName:{"$regex":`${name}`}, memberType:MEMBERTYPE.单位会员},
                {name:{"$regex":`${name}`}, memberType:MEMBERTYPE.个人会员}
            ]
        });
    }
    if (joinStartTime) {
        selectParam["joinTime"] = {"$gt":joinStartTime};
    }
    if (joinEndTime) {
        if (!selectParam["joinTime"]) selectParam["joinTime"] = {};
        selectParam["joinTime"]["$lt"] = joinEndTime;
    }
    // if (memberType.length) {
    //     if (memberType < UNITMEMBERTYPE.院校) {
    //         selectParam.individualMemberType = {"$in":memberType};
    //     } else {
    //         selectParam.unitMemberType = {"$in":memberType};
    //     }
    // }
    if (memberType && memberType.length) selectParam["$or"] = [ {unitMemberType:{"$in":memberType}}, {individualMemberType:{"$in":memberType}} ];
    if (documentId) selectParam.documentId = documentId;
    if (phone) selectParam.phone = phone;
    if (mail) selectParam.mail = mail;
    if (memberLevel.length) selectParam.memberLevel = {"$in":memberLevel};

    // const SelectFiles = ["userId", "memberState", "unitName", "loginId", "phone", "joinTime", "memberType", "lifespanStartTime", "lifespanEndTime", "name", "individualMemberType", "unitMemberType"];
    const SelectFiles = ["userId", "memberState", "individualMemberType", "unitMemberType", "unitName", "name", "memberLevel", "joinTime"];
    let dbList = await findToSortToPage(TABLEENUM.用户表, selectParam, SelectFiles, {joinTime:-1}, pageNumber);
    // let dbList = await findToPage(TABLEENUM.用户表, selectParam, SelectFiles, pageNumber, 10);

    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    const itemFiles = ["userId", "memberState", "memberType", "unitName", "name", "memberLevel", "joinTime", "orderCycleStart", "money", "durationYear"];
    for (let i = 0; i < dbList.length; i++) {
        let info = dbList[i];
        let item:any = extractData(info, itemFiles);
        // item.lifespanTime = `${moment(info.lifespanEndTime).format("YYYY-MM-DD")}`;
        // if (info.lifespanEndTime > ( NowMs - (30 * 24 * 3600 * 1000) ) ) item.nextLifespanTime = "过期30天内";
        // else  {
        //     item.nextLifespanTime = "过期1年及以上";
        // }
    
        item.joinTime = moment(item.joinTime).format("YYYY-MM-DD");
        item.memberState = changeEnumValue(MEMBERSTATE, item.memberState);
        item.memberLevel = changeEnumValue(MEMBERLEVEL, item.memberLevel);
        if(info.individualMemberType) {
            item.memberType = changeEnumValue(MEMBERTYPEECCENUM, info.individualMemberType);
        }
        if(info.unitMemberType) {
            item.memberType = changeEnumValue(MEMBERTYPEECCENUM, info.unitMemberType);
        }
        
        let orderDbList = await find(TABLEENUM.订单表, {userId:info.userId}, ["orderCycleStart", "money", "state"]);
        if (!item.money) item.money = 0;
        if (!item.durationYear) item.durationYear = 0;
        orderDbList.forEach(orderItem => {
            item.money += orderItem.money;
            if (!item.orderCycleStart) item.orderCycleStart = new Date(orderItem.orderCycleStart).getFullYear();
            else item.orderCycleStart += ` ${new Date(orderItem.orderCycleStart).getFullYear()}`;
            
            if (orderItem.state == ORDERSTATE.未支付) {
                item.durationYear += 1; //欠费年限
            }
        });

        dataList.push(item);
    };

    return {dataList, dataCount};
}


/**
 * 线下付款 会费管理 success
 * @param param0 
 */
export async function offlinePayment({orderId, paymentNum, offlinePaymentUrl }) {
    let orderInfo = await findOnce(TABLEENUM.订单表, {id:orderId});
    if (!orderInfo || !orderInfo.id) throw new BizError(ERRORENUM.目标数据不存在);
    if (orderInfo.state == ORDERSTATE.已支付 && orderInfo.confirmReceipt != RECEIPTCONFIRMATION.退回) throw new BizError(ERRORENUM.该订单已支付);

    let updateInfo = {
        paymentNum,
        offlinePaymentUrl,
        state:ORDERSTATE.已支付,
        paymentMethod:PAYMENTTYPE.线下支付,
        payTime:new Date().valueOf(),
        confirmReceipt:RECEIPTCONFIRMATION.待确认
    };

    await updateOneData(TABLEENUM.订单表, {id:orderId}, updateInfo);

    
    //添加审批日志
    let addLogInfo = {
        orderId,
        operationTime:new Date().valueOf(),
        operationBehavior:ORDEREXAMINE.用户提交,
        remarks:""
    };
    await addOneData(TABLEENUM.订单审批历史表, addLogInfo );

    return successResult();
}


/**
 * 多比订单时选择线下付款页面的选择缴费周期页
 * @param param0 
 * @returns 
 */
export async function selectManyOrderForUser({id}) {
    let notPayList = await find(TABLEENUM.订单表, {userId:id, state:ORDERSTATE.未支付},["orderCycleStart", "orderCycleEnd", "id"] );

    let dataList = [];

    notPayList.forEach(info => {
        let {orderCycleStart, orderCycleEnd, id} = info;

        dataList.push({
            key: `${moment(orderCycleStart).format("YYYY-MM-DD")} 至 ${moment(orderCycleEnd).format("YYYY-MM-DD")}`,
            value:id
        });
    });
    return {dataList};
}



/**
 * 设置宽限期 success
 * @param param0 
 */
export async function setGracePeriod({id, gracePeriodEndTime}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId:id, isAdmin:STATE.否}, ["userId"]);
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    let updateInfo = {
        isGracePeriod:STATE.是,
        gracePeriodEndTime
    };

    await updateOneData(TABLEENUM.用户表, {userId:id}, updateInfo);

    return successResult();
}


/**
 * 宽限期列表 success
 * @param name 名称关键字：个人会员匹配真实姓名，单位会员匹配单位名称
 * @param memberType 会员类别 unitMemberType、individualMemberType  多选
 * 会员类别：个人【个人会员、专业会员】单位【院校、机构、其他】返回前端枚举：MEMBERTYPEECCENUM
 * @param documentId 身份证
 * @param phone 登录手机号
 * @param mail 单位/个人邮箱
 * @param joinStartTime 入会开始时间
 * @param joinEndTime 入会结束时间
 * @param memberLevel 会员级别/职务 多选
 * 会员级别/职务：【普通会员、理事会员、常务理事会员】
 * @param pageNumber 当前页
 */
export async function getGracePeriod({name, memberType, documentId, phone, mail, joinStartTime, joinEndTime, memberLevel, duties, pageNumber}) {
    const NowMs = new Date().valueOf();
    /**到期时间距离今天大于90天 */
    let selectParam:any = {
        isGracePeriod:STATE.是, 
        gracePeriodEndTime:{"$gt":NowMs},
        isAdmin:STATE.否,
        userRegisterState:USERREGISTERSTATE.通过
    }
    if (name) {
        selectParam["$or"] = [
            {unitName:{"$regex":`${name}`}, memberType:MEMBERTYPE.单位会员},
            {name:{"$regex":`${name}`}, memberType:MEMBERTYPE.个人会员}
        ]
    }
    if (joinStartTime) {
        selectParam["joinTime"] = {"$gt":joinStartTime};
    }
    if (joinEndTime) {
        if (!selectParam["joinTime"]) selectParam["joinTime"] = {};
        selectParam["joinTime"]["$lt"] = joinEndTime;
    }
    // if (memberType.length) {
    //     if (memberType < UNITMEMBERTYPE.院校) {
    //         selectParam.individualMemberType = {"$in":memberType};
    //     } else {
    //         selectParam.unitMemberType = {"$in":memberType};
    //     }
    // }
    if (memberType && memberType.length) selectParam["$or"] = [ {unitMemberType:{"$in":memberType}}, {individualMemberType:{"$in":memberType}} ];
    if (documentId) selectParam.documentId = documentId;
    if (phone) selectParam.phone = phone;
    if (mail) selectParam.mail = mail;
    if (memberLevel.length) selectParam.memberLevel = {"$in":memberLevel};

    // const SelectFiles = [ "userId", "memberState", "unitName", "loginId", "phone", "joinTime", "memberType", "joinTime", "lifespanEndTime", "individualMemberType", "unitMemberType"];
    const SelectFiles = ["userId", "memberState", "individualMemberType", "unitMemberType", "unitName", "name", "memberLevel", "joinTime", 
    "lifespanEndTime", "lifespanEndTime"];
    let dbList = await findToPage(TABLEENUM.用户表, selectParam, SelectFiles, pageNumber, 10);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    // const itemFiles = ["userId", "memberState", "loginId", "phone", "joinTime", "memberType", "unitName"];
    const itemFiles = ["userId", "memberState", "memberType", "unitName", "name", "memberLevel", "joinTime", "money", "lifespanTime", 
    "nextPayment", "nextLifespanTime"];
    for (let i = 0; i < dbList.length; i++) {
        let info = dbList[i];
        let item:any = extractData(info, itemFiles);
        item.lifespanTime = `${moment(info.lifespanStartTime).format("YYYY-MM-DD")} 至 ${moment(info.lifespanEndTime).format("YYYY-MM-DD")}`; //会员有效期
        item.nextPayment = moment(info.lifespanEndTime).format("YYYY-MM-DD"); //下次缴费时间
        item.nextLifespanTime = "宽限期中";
        item.joinTime = moment(item.joinTime).format("YYYY-MM-DD");
        item.memberState = changeEnumValue(MEMBERSTATE, item.memberState);
        item.memberLevel = changeEnumValue(MEMBERLEVEL, item.memberLevel);
        if(info.individualMemberType) {
            item.memberType = changeEnumValue(MEMBERTYPEECCENUM, info.individualMemberType);
        }
        if(info.unitMemberType) {
            item.memberType = changeEnumValue(MEMBERTYPEECCENUM, info.unitMemberType);
        }

        let orderDbList = await find(TABLEENUM.订单表, {userId:info.userId}, ["money"])
        // state 缴费状态
        orderDbList.forEach(orderItem => {
            item.money = orderItem.money;
        });

        dataList.push(item);
    }

    return {dataList, dataCount};
}


/**
 * 移除宽限期
 * @param id 宽限期列表返回的userId  
 * @returns 
 */
export async function delGracePeriod({id}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId:id, isAdmin:STATE.否}, ["userId"]);
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    if (userInfo.isGracePeriod) throw new BizError(ERRORENUM.该用户已在宽限期内);

    let updateInfo = {
        isGracePeriod:STATE.否,
        gracePeriodEndTime:0
    };

    await updateOneData(TABLEENUM.用户表, {userId:id}, updateInfo);

    return successResult();
}



/**
 * 个人欠费周期列表 
 * @param param0 
 */
export async function memberArrearsList({id}) {
    /**获取用户欠费的周期列表 包含id 周期 金额 */
    
    let dbList = await findToSort(TABLEENUM.订单表, {userId:id}, {orderCycleStart:1}, ["id", "orderCycleStart", "orderCycleEnd", "money"] );

    let dataList = [];
    dbList.forEach(info => {
        let {id, orderCycleStart, orderCycleEnd, money } = info;
        let renewalPeriod = `${moment(orderCycleStart).format("YYYY-MM-DD")}-${moment(orderCycleEnd).format("YYYY-MM-DD")}`;
        dataList.push({
            id, renewalPeriod, money
        })
    });

    return {dataList};
}