/**
 * 订单相关
 */
import moment = require("moment");
import { COSTTYPE, INVOICESTATUS, ISPAYENUM, MEMBERLEVEL, MEMBERTYPE, ORDERSTATE, PAYMENTSTATUS, PAYMENTTYPE, RECEIPTCONFIRMATION, STATE } from "../../config/enum";
import { generateOrderId, successResult } from "../../tools/system";
import { changeEnumValue, eccEnumValue } from "../../util/verificationEnum";
import { addOneData } from "../../data/add";
import { TABLEENUM } from "../../data/models/model";
import { find, findCount, findOnce, findToPage } from "../../data/select";
import { extractData } from "../../util/piecemeal";
import { BizError } from "../../util/bizError";
import { ERRORENUM } from "../../config/errorEnum";
import { updateOneData } from "../../data/update";


/**
 * 对内使用  创建一条订单
 * @param userId 会员id
 * @param unitName 单位名称
 * @param orderStartMs 订单开始时间
 * @param memberLv 会员等级 MEMBERLEVEL
 * @param memberType 会员类型 MEMBERTYPE
 * @param isFirst 是否首次缴费
 * @returns 
 */
export async function createOrder(userId:string, unitName:string, orderStartMs:number, memberLv:number, memberType:number, isFirst:boolean) {
    /**计算订单周期 */
    const EndMs = moment(orderStartMs).add(1, 'years').valueOf();
    /**订单id */
    const OrderId = generateOrderId(userId);
    /**计算会费 */
    let money = 0;
    if (memberLv == MEMBERLEVEL.专家会员) {

    } else {

    }
    let addInfo = {
        id:OrderId,
        orderCycleStart:orderStartMs,
        orderCycleEnd:EndMs,
        unitName,
        money,
        paymentMethod:0,
        memberId:userId,
        memberCategory:memberType,
        paymentNum:0,
        invoiceAdd:'',
        state:ORDERSTATE.未支付,
        isFirst
    };

    // await addOneData(TABLEENUM.订单表, addInfo);
    return true;
}


/**
 * 初始订单 success
 * 会员管理-待支付页
 */
export async function firstOrderList({unitName, joinTime, memberType, orderState, pageNumber}) {
    /** 首次订单 并且未通过待支付页的审核 */
    let findParam:any = { "$or":[{isFirst:true, firstPayExamine:false}] };
    if (unitName) findParam.unitName = {"$regex":unitName};
    if (memberType) findParam.memberCategory = memberType;
    if (orderState) findParam.state = orderState;

    let selectFile = ["unitName", "loginId", "phone", "orderCycleStart", "orderCycleEnd", "memberCategory", "state"];
    let dbList = await findToPage(TABLEENUM.订单表, findParam, selectFile, pageNumber);
    let dataCount = await findCount(TABLEENUM.订单表, findParam);
    let dataList = [];
    let itemFile = ["unitName", "loginId", "phone"];
    dbList.forEach(info =>  {
        let itemData:any = extractData(info, itemFile);
        itemData.joinTime = moment(info.orderCycleStart).format("YYYY-MM-DD");
        itemData.memberType = changeEnumValue(MEMBERTYPE, itemData.memberCategory);
        itemData.paymentMethod = changeEnumValue(MEMBERTYPE, itemData.paymentMethod);
        itemData.cycle = `${moment(info.orderCycleStart).format("YYYY-MM-DD")}至${moment(info.orderCycleEnd).format("YYYY-MM-DD")}`;
        if (info.state == ORDERSTATE.未支付) {
            itemData.state = "未支付";
        } else {
            if (info.paymentMethod == PAYMENTTYPE.银行转账)  itemData.state = "银行转账";
            else itemData.state = "已支付";
        }
        dataList.push(itemData);
    });

    return {dataList, dataCount}
}


/**
 * 会员管理-待支付页-通过审核 success
 * @param param0 
 */
export async function checkOrder({id}) {
    let orderInfo = await findOnce(TABLEENUM.订单表, {id});
    if (!orderInfo || !orderInfo.id ) throw new BizError(ERRORENUM.目标数据不存在);
    if (!orderInfo.isFirst || orderInfo.firstPayExamine)  throw new BizError(ERRORENUM.重复提交);

    await updateOneData(TABLEENUM.订单表, {id}, {firstPayExamine:true, confirmReceipt:RECEIPTCONFIRMATION.待确认});

    return successResult();
}


/**
 * 发票管理-财务核对列表 success
 * @param param0 
 */
export async function paidList({unitName, paymentType, costType, pageNumber }) {
    eccEnumValue("财务核对列表", "支付类型", PAYMENTTYPE, paymentType);
    eccEnumValue("财务核对列表", "会费类别", COSTTYPE, costType );

    let findParam:any = { 
        "$or":[{isFirst:false}, {isFirst:true, firstPayExamine:true }],
        confirmReceipt:RECEIPTCONFIRMATION.待确认,
        state:ORDERSTATE.已支付
    };
    if (unitName) findParam.unitName = {"$regex":unitName};
    if (paymentType) findParam.paymentMethod = paymentType;
    if (costType) findParam.isFirst = costType == COSTTYPE.首次;

    let selectFile = ["id", "unitName", "orderCycleStart", "orderCycleEnd", "money", "paymentMethod", "memberCategory", "isFirst", "paymentNum" ];
    let dbList = await findToPage(TABLEENUM.订单表, findParam, selectFile, pageNumber);
    let dataCount = await findCount(TABLEENUM.订单表, findParam);

    let dataList = [];

    let itemFile = ["unitName", "money", "paymentNum"];
    dbList.forEach(info =>  {
        let itemData:any = extractData(info, itemFile);
        itemData.joinTime = moment(info.orderCycleStart).format("YYYY-MM-DD");
        itemData.memberType = changeEnumValue(MEMBERTYPE, itemData.memberCategory);
        itemData.paymentMethod = changeEnumValue(MEMBERTYPE, itemData.paymentMethod);
        itemData.costType = info.isFirst == COSTTYPE.首次 ? "首次" : "续费";
        itemData.cycle = `${moment(info.orderCycleStart).format("YYYY-MM-DD")}至${moment(info.orderCycleEnd).format("YYYY-MM-DD")}`
        dataList.push(itemData);
    });

    return {dataList, dataCount};
}


/**
 * 发票管理-财务核对页-收款确认 success
 * @param param0 
 */
export async function confirmReceiptPass({id}) {
    let orderInfo = await findOnce(TABLEENUM.订单表, {id});
    if (!orderInfo || !orderInfo.id ) throw new BizError(ERRORENUM.目标数据不存在);
    if (orderInfo.isFirst && !orderInfo.firstPayExamine ) throw new BizError(ERRORENUM.重复提交, '发票管理-财务核对页-收款确认', `提交通过时订单未通过待支付页的校验`);
    if (orderInfo.confirmReceipt != RECEIPTCONFIRMATION.待确认) throw new BizError(ERRORENUM.重复提交, '发票管理-财务核对页-收款确认', `提交通过时订单已经不是待确认 是${orderInfo.confirmReceipt}`);

    await updateOneData(TABLEENUM.订单表, {id}, {confirmReceipt:RECEIPTCONFIRMATION.收款确认, invoiceStatus:INVOICESTATUS.未开发票});

    return successResult();
}


/**
 * 发票管理-财务核对页-收款退回 success
 * @param param0 
 */
export async function confirmReceiptOut({id}) {
    let orderInfo = await findOnce(TABLEENUM.订单表, {id});
    if (!orderInfo || !orderInfo.id ) throw new BizError(ERRORENUM.目标数据不存在);
    if (orderInfo.isFirst && !orderInfo.firstPayExamine ) throw new BizError(ERRORENUM.重复提交, '发票管理-财务核对页-收款确认', `提交通过时订单未通过待支付页的校验`);
    if (orderInfo.confirmReceipt != RECEIPTCONFIRMATION.待确认) throw new BizError(ERRORENUM.重复提交, '发票管理-财务核对页-收款确认', `提交通过时订单已经不是待确认 是${orderInfo.confirmReceipt}`);
    if (orderInfo.paymentMethod == PAYMENTTYPE.微信支付) throw new BizError(ERRORENUM.微信支付无法被退回);

    let updateInfo:any = {confirmReceipt:RECEIPTCONFIRMATION.退回};
    if (orderInfo.isFirst == true) {
        updateInfo.firstPayExamine = false;
    }

    await updateOneData(TABLEENUM.订单表, {id}, {confirmReceipt:RECEIPTCONFIRMATION.退回});

    return successResult();
}


/**
 * 发票管理-待开发票 success
 * @param param0 
 */
export async function billStateList({unitName, paymentType, isPay, pageNumber}) {
    eccEnumValue("发票列表", "支付类型", PAYMENTTYPE, paymentType);
    eccEnumValue("发票列表", "是否支付", ISPAYENUM, isPay );

    let findParam:any = { 
        invoiceStatus:INVOICESTATUS.未开发票
    };
    if (unitName) findParam.unitName = {"$regex":unitName};
    if (paymentType) findParam.paymentMethod = paymentType;
    if (isPay) findParam.state = isPay

    let selectFile = ["id", "unitName", "orderCycleStart", "orderCycleEnd", "money", "paymentMethod", "memberCategory", "state", "paymentNum" ];
    let dbList = await findToPage(TABLEENUM.订单表, findParam, selectFile, pageNumber);
    let dataCount = await findCount(TABLEENUM.订单表, findParam);

    let dataList = [];

    let itemFile = ["unitName", "money", "paymentNum"];
    dbList.forEach(info =>  {
        let itemData:any = extractData(info, itemFile);
        itemData.joinTime = moment(info.orderCycleStart).format("YYYY-MM-DD");
        itemData.memberType = changeEnumValue(MEMBERTYPE, itemData.memberCategory);
        itemData.paymentMethod = changeEnumValue(MEMBERTYPE, itemData.paymentMethod);
        itemData.cycle = `${moment(info.orderCycleStart).format("YYYY-MM-DD")}至${moment(info.orderCycleEnd).format("YYYY-MM-DD")}`
        dataList.push(itemData);
    });

    return {dataList, dataCount};
}


/**
 * 上传发票 success
 * @param param0 
 */
export async function upInvoice({id, invoiceUrl}) {
    let orderInfo = await findOnce(TABLEENUM.订单表, {id});

    if (!orderInfo || !orderInfo.id) throw new BizError(ERRORENUM.目标数据不存在);
    if (orderInfo.invoiceStatus != INVOICESTATUS.未开发票) throw new BizError(ERRORENUM.订单状态无法开具发票, `${orderInfo.id}无法开具发票，因为发票状态为${orderInfo.invoiceStatus}`);
    
    let updateInfo = {
        invoiceStatus : INVOICESTATUS.已开发票,
        invoiceAdd:invoiceUrl
    };
    await updateOneData(TABLEENUM.订单表, {id}, updateInfo);

    return successResult();
}


/**
 * 退回发票 success
 * @param param0 
 */
export async function backInvoice({id}) {
    let orderInfo = await findOnce(TABLEENUM.订单表, {id});

    if (!orderInfo || !orderInfo.id) throw new BizError(ERRORENUM.目标数据不存在);
    if (orderInfo.invoiceStatus != INVOICESTATUS.未开发票) throw new BizError(ERRORENUM.订单状态无法开具发票, `${orderInfo.id}无法开具发票，因为发票状态为${orderInfo.invoiceStatus}`);

    if (orderInfo.paymentMethod == PAYMENTTYPE.微信支付) throw new BizError(ERRORENUM.微信支付无法退回发票请求);

    let updateInfo:any = {
        invoiceStatus : INVOICESTATUS.退回,
    };
    if (orderInfo.state == ORDERSTATE.已支付 ) updateInfo.confirmReceipt = RECEIPTCONFIRMATION.待确认
    await updateOneData(TABLEENUM.订单表, {id}, updateInfo);
    
    return successResult();
}


/**
 * 发票管理-已开发票 success
 * @param param0 
 */
export async function InvoicedList({unitName, paymentType, isPay, pageNumber}) {
    eccEnumValue("发票列表", "支付类型", PAYMENTTYPE, paymentType);
    eccEnumValue("发票列表", "是否支付", ISPAYENUM, isPay );

    let findParam:any = { 
        invoiceStatus:INVOICESTATUS.已开发票
    };
    if (unitName) findParam.unitName = {"$regex":unitName};
    if (paymentType) findParam.paymentMethod = paymentType;
    if (isPay) findParam.state = isPay

    let selectFile = ["id", "unitName", "orderCycleStart", "orderCycleEnd", "money", "paymentMethod", "memberCategory", "state", "paymentNum" ];
    let dbList = await findToPage(TABLEENUM.订单表, findParam, selectFile, pageNumber);
    let dataCount = await findCount(TABLEENUM.订单表, findParam);

    let dataList = [];

    let itemFile = ["unitName", "money", "paymentNum"];
    dbList.forEach(info =>  {
        let itemData:any = extractData(info, itemFile);
        itemData.joinTime = moment(info.orderCycleStart).format("YYYY-MM-DD");
        itemData.memberType = changeEnumValue(MEMBERTYPE, itemData.memberCategory);
        itemData.paymentMethod = changeEnumValue(MEMBERTYPE, itemData.paymentMethod);
        itemData.cycle = `${moment(info.orderCycleStart).format("YYYY-MM-DD")}至${moment(info.orderCycleEnd).format("YYYY-MM-DD")}`
        dataList.push(itemData);
    });

    return {dataList, dataCount};
}


/**
 * 个人页待订单列表  success
 * @param param0 
 */
export async function orderList({userId, type}) {
    eccEnumValue("个人页待订单列表", "类型", ISPAYENUM, type);
    let findParam:any = {userId};
    if (type == ORDERSTATE.已支付) findParam.state = ORDERSTATE.已支付;
    else if (type == ORDERSTATE.未支付) findParam.state = ORDERSTATE.未支付;

    const SelectFile = ["orderCycleStart", "orderCycleEnd", "money", "state", "id"];
    let dbList = await find(TABLEENUM.订单表, findParam, SelectFile);

    let dataList = [];
    dbList.forEach(info => {
        let {orderCycleStart, orderCycleEnd, money, state, id} = info;
        let moneyStr = state == ORDERSTATE.已支付 ? "已缴金额" : "待缴金额"; 
        dataList.push({
            cycle:`${moment(orderCycleStart).format("YYYY-MM-DD")}至${moment(orderCycleEnd).format("YYYY-MM-DD")}`,
            money:`${moneyStr}:${money}`,
            id
        });
    });

    return {dataList}
}