/**
 * 审批逻辑
 */

import { MEMBERLEVEL, MEMBERTYPE, OPERATIONREHAVIOR, PAYMENTSTATUS, REGISTERFLOW, USERREGISTERSTATE } from "../../config/enum";
import { ERRORENUM } from "../../config/errorEnum";
import { addManyData } from "../../data/add";
import { TABLEENUM } from "../../data/models/model";
import { find, findCount, findOnce, findToPage } from "../../data/select";
import { updateOneData } from "../../data/update";
import { generateSystemId, successResult } from "../../tools/system";
import { BizError } from "../../util/bizError";
import { extractData } from "../../util/piecemeal";
import { changeEnumValue, eccEnumValue } from "../../util/verificationEnum";

/**
 * 待审核列表 success
 * @param unitName 单位名称
 * @param joinTime 入会时间
 * @param memberType 会员类型 MEMBERTYPE
 * @param sheng 省
 * @param shi 市
 * @param qu 区
 * @param pageNumber 当前页 
 */
export async function pendingReviewList({unitName, joinTime, memberType, sheng, shi, qu, pageNumber}) {
    /**校验枚举 */
    const EccFuncName = '待审核列表';
    eccEnumValue(EccFuncName, '会员类型', MEMBERTYPE, memberType);

    /**查询条件 */
    let selectParam:any = {userRegisterState:{"$not": USERREGISTERSTATE.通过}}; //审批列表展示状态除通过外的数据
    if (unitName) selectParam.unitName = unitName;
    if (joinTime) selectParam.joinTime = joinTime;
    if (memberType) selectParam.memberType = memberType;
    if (sheng) selectParam.sheng = sheng;
    if (shi) selectParam.shi = shi;
    if (qu) selectParam.qu = qu;

    const SelectConf = ["userId", "userRegisterState", "unitName", "loginId", "phone", "askForTime", "memberLevel", "sheng", "shi", "qu"];
    let dbList = await findToPage(TABLEENUM.用户表, selectParam, SelectConf, pageNumber);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    dbList.forEach(info => {
        let item:any = extractData(info, SelectConf);
        /**处理枚举值 */
        if (!item.userRegisterState) item.userRegisterState = USERREGISTERSTATE.待审核;
        else item.userRegisterState = changeEnumValue(USERREGISTERSTATE, item.userRegisterState);
        item.memberLevel = changeEnumValue(MEMBERLEVEL, item.memberLevel);
        dataList.push(item);
    });

    return {dataList, dataCount};
}


/**
 * 通过审核 success
 * @param userId 
 * @returns 
 */
export async function adopt({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    await updateOneData(TABLEENUM.用户表, {userId}, {userRegisterState:USERREGISTERSTATE.通过});

    /**提交审批历史信息 */
    const NowMs = new Date().valueOf(); 
    let addApprovalHistory = {
        id:generateSystemId(TABLEENUM.审批历史表, userId),
        userId,
        operationTime:NowMs,
        operationBehavior:OPERATIONREHAVIOR.通过,
        remarks:""
    };
    await addManyData(TABLEENUM.审批历史表, addApprovalHistory);

    return successResult();
}


/**
 * 驳回 success
 * @param userId 
 * @param remarks 驳回理由
 * @returns 
 */
export async function reject({userId, remarks}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    /**驳回修改，修改该会员注册流程回到第二步 */
    let updateParam = {
        userRegisterState:USERREGISTERSTATE.驳回修改, 
        registerFlow:REGISTERFLOW.完成第二步
    }
    await updateOneData(TABLEENUM.用户表, {userId}, updateParam);
    
    /**提交审批历史信息 */
    const NowMs = new Date().valueOf(); 
    let addApprovalHistory = {
        id:generateSystemId(TABLEENUM.审批历史表, userId),
        userId,
        operationTime:NowMs,
        operationBehavior:OPERATIONREHAVIOR.驳回修改,
        remarks
    };
    await addManyData(TABLEENUM.审批历史表, addApprovalHistory);

    return successResult();
}


/**
 * 审核历史 success
 * @param userId
 * @returns 
 */
export async function approvalHistoryList({userId}) {
    let oldInfo = await find(TABLEENUM.审批历史表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    let dataList:any = extractData(oldInfo, ["operationTime", "operationBehavior", "remarks"]);
    dataList.operationBehavior = changeEnumValue(OPERATIONREHAVIOR, dataList.operationBehavior);

    return dataList;
}


/**
 * 待支付列表
 * @param unitName 单位名称
 * @param joinTime 入会时间
 * @param memberType 会员类型 MEMBERTYPE
 * @param paymentStatus 支付状态
 * @param pageNumber 当前页
 * @returns 
 */
export async function toBePaidList({unitName, joinTime, memberType, paymentStatus, pageNumber}) {
    /**校验枚举 */
    const EccFuncName = '待审核列表';
    eccEnumValue(EccFuncName, '会员类型', MEMBERTYPE, memberType);
    eccEnumValue(EccFuncName, '支付状态', PAYMENTSTATUS, paymentStatus);

    let selectParam:any = {};
    if (unitName) selectParam.unitName = unitName;
    if (joinTime) selectParam.joinTime = joinTime;
    if (memberType) selectParam.memberType = memberType;
    if (paymentStatus) selectParam.paymentStatus = paymentStatus;

    const SelectConf = ["userId", "状态", "unitName", "loginId", "phone", "joinTime", "memberType", "lifespanStartTime", "lifespanEndTime", "paymentStatus"];
    let dbList = await findToPage(TABLEENUM.用户表, selectParam, SelectConf, pageNumber);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    dbList.forEach(info => {
        let item:any = extractData(info, SelectConf);
        item.memberType = changeEnumValue(MEMBERTYPE, item.memberType);
        item.paymentStatus = changeEnumValue(PAYMENTSTATUS, item.paymentStatus);
        dataList.push(item);
    });

    return {dataList, dataCount};







    
}


/**
 * 待支付通过
 * @param param0 
 * @returns 
 */
export async function passThroughPaid({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    await updateOneData(TABLEENUM.用户表, {userId}, {paymentStatus:PAYMENTSTATUS.已支付});

    return successResult();
}


/**
 * 获取届次
 * @param year 入会年份 
 * @param month 入会月份
 * @returns 
 */
export async function getEdition(year, month) {
    //起始年份和月份
    const startYear = 2018;
    const startMonth = 1;

    let currentYear = new Date().getFullYear();
    if (year > currentYear) throw new BizError(ERRORENUM.无法获取未来届次);

    //计算从起始时间给定时间的总月份数
    let totalMonth = (year - startYear) * 12 + month;
    //每届是6个月
    let sessions = Math.ceil(totalMonth / 6);
    //每10届一次
    let times = Math.ceil(sessions / 10);
    let sessionNumber = sessions - (times - 1) * 10 //届次是从第一次第一届开始，所以需要减一

    return {frequency:"第"+times+"次", session:"第"+sessionNumber+"届"};
}




