/**
 * 会员逻辑
 */

import { MEMBERSTATE, MEMBERLEVEL, DOCUMENTTYPE, NATION, PROFCATEGORY, CODETYPE, UNITINFOTYPE, MEMBERTYPE, REGISTERFLOW, SEX, USERREGISTERSTATE, OPERATIONREHAVIOR, STATE, BANXUELEIXING } from "../../config/enum";
import { ERRORENUM } from "../../config/errorEnum";
import { TABLEENUM } from "../../data/models/model";
import { find, findCount, findOnce, findOnceToSort, findToPage, findToSort } from "../../data/select";
import { updateOneData } from "../../data/update";
import { generateSystemId, successResult } from "../../tools/system";
import { BizError } from "../../util/bizError";
import { extractData } from "../../util/piecemeal";
import { changeEnumValue, eccEnumValue } from "../../util/verificationEnum";
import * as sysTools from "../../tools/system";
import { addManyData, addOneData } from "../../data/add";
import { deleteManyData, deleteOneData } from "../../data/delete";


/**
 * 单位会员列表
 * @param unitName 单位名称 
 * @param joinTime 入会时间
 * @param unitMemberType 单位会员所属类型 UNITMEMBERTYPE
 * @param memberLevel 会员等级 职务 MEMBERLEVEL
 * @param session 届次 
 * @param sheng 省
 * @param shi 市
 * @param qu 区
 * @param pageNumber 当前页
 */
export async function unitMemberList({unitName, joinTime, unitMemberType, memberLevel, session, sheng, shi, qu, pageNumber}) {
    let selectParam:any = {};
    if (unitName) selectParam.unitName = unitName;
    if (joinTime) selectParam.joinTime = joinTime;
    if (unitMemberType) selectParam.unitMemberType = unitMemberType;
    if (memberLevel) selectParam.memberLevel = memberLevel;
    if (session) selectParam.session = session;
    if (sheng) selectParam.sheng = sheng;
    if (shi) selectParam.shi = shi;
    if (qu) selectParam.qu = qu;
    selectParam.memberType = MEMBERTYPE.单位会员;
    selectParam.userRegisterState = USERREGISTERSTATE.通过;

    let selectConf = ["userId", "memberState", "unitName", "loginId", "phone", "joinTime", "unitMemberLevel", "sheng", "shi", "qu"];
    let dbList = await findToPage(TABLEENUM.用户表, selectParam, selectConf, pageNumber);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    dbList.forEach(info => {
        let item:any = extractData(info, selectConf);
        item.memberState = changeEnumValue(MEMBERSTATE, item.memberState);
        if (!item.memberLevel) item.memberLevel = "未选择";
        else item.memberLevel = changeEnumValue(MEMBERLEVEL, item.memberLevel);
        dataList.push(item);
    });

    return {dataList, dataCount};
}


/**
 * 单位会员详情
 * @param param0 
 * @returns {顶部数据, 基本信息, 单位信息, 单位所获得科研成果}
 */

const UnitBasicConfig = ["userId", "unitName", "addres", "uscc", "legalPerson", "legalPersonPhone", "contactPerson", "contactPersonDuties", "sheng", "shi", "qu", 
"uusinessLicenseUrl", "education"];
const UnitConfig = ["unitInfoType", "yuanXiaoBanXueLeiXing", "yuanXiaoZhuGuanBuMen", "yuanXiaoFuZeRen", "yuanXiaoFuZeRenZhiWu", "yuanXiaoFuZeRenDianHua", 
"yuanXiaoBanGongFuZeRenZhiWu", "yuanXiaoBanGongFuZeRenDianHua", "yuanXiaoXueShengZongRenShu", "yuanXiaoJiaoZhiGongZongRenShu", "yuanXiaoJianZhuMianJi",
"yuanXiaoKeYanFuZeRen", "yuanXiaoKeYanFuZeRenDianHua", "jiaoXueFuZeRenXinMing", "jiaoXueFuZeRenDianHua", "ZhuYaoFuZeRenYouXiang"];

export async function unitMemberDetails({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    let topInfo:any = extractData(oldInfo, ["session", "memberLevel"]);
    topInfo.memberLevel = changeEnumValue(MEMBERLEVEL, topInfo.memberLevel);
    let basicInfo = extractData(oldInfo, UnitBasicConfig);
    let unitInfo = extractData(oldInfo, UnitConfig);
    
    let oldKeYanInfo = await findOnce(TABLEENUM.单位所获得科研成果表, {userId});
    let scientificResearch = extractData(oldKeYanInfo, ["id", "startTime", "endTime", "describe"]);
    
    return {topInfo, basicInfo, unitInfo, scientificResearch};
}


/**
 * 单位会员 修改
 * @param userId
 * @param form
 * @returns 
 */
export async function unitMemberUpdate({form, keyanForm, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});

    /**校验枚举 */
    const EccFuncName = '单位会员修改';
    eccEnumValue(EccFuncName, '单位信息类型', UNITINFOTYPE, form.unitInfoType);

    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    await updateOneData(TABLEENUM.用户表, {userId}, form);

    /**删除对应userId科研成果 */
    let oldKeYanInfo = await find(TABLEENUM.单位所获得科研成果表, {userId});
    if (!oldKeYanInfo || !oldKeYanInfo.id) throw new BizError(ERRORENUM.目标数据不存在);
    await deleteManyData(TABLEENUM.单位所获得科研成果表, {userId});

    let addList = [];
    for (let i = 0; i < keyanForm.length; i++) {
        keyanForm[i].id = generateSystemId(TABLEENUM.单位所获得科研成果表, userId);
        keyanForm[i].userId = userId;
        addList.push(keyanForm);
    }
    await addManyData(TABLEENUM.单位所获得科研成果表, addList);

    return successResult();
}


/**
 * 删除单位所获得科研成果
 * @param param0 
 * @returns 
 */
export async function delResearch({id}) {
    let dataInfo = await findOnce(TABLEENUM.单位所获得科研成果表, {id});
    if (!dataInfo || !dataInfo.id) throw new BizError(ERRORENUM.目标数据不存在);
    await deleteOneData(TABLEENUM.单位所获得科研成果表, {id});
    return successResult();
}


/**
 * 会员注册【短信验证码】 success
 * @param userId 
 * @param phone 
 * @returns 
 */
export async function memberRegisterSendCode({phone}) {
    const NowMs = new Date().valueOf();
    let codeSelectParam = { phone, isUse:false,  type:CODETYPE.用户注册, sendMs:{"$gt":sysTools.getTodayMs()} }
    let lastCodeInfo = await findOnceToSort(TABLEENUM.验证码表, codeSelectParam, {sendMs:-1});
    if (lastCodeInfo.code) {
        if ( (lastCodeInfo.sendMs + (60*1000)) > NowMs ) throw new BizError(ERRORENUM.频繁操作请稍后再试);
        /**校验今日同类型验证码次数 */
        let todayNotUseCount = await findCount(TABLEENUM.验证码表, codeSelectParam);
        if (todayNotUseCount >= 4) throw new BizError(ERRORENUM.发送验证码次数超限制, `${phone} 注册验证码超过限制3`);
    }

    const Code = sysTools.generateSMSCode();//生成短信验证码
    let addInfo = {code:Code, phone, sendMs:NowMs, type:CODETYPE.用户注册, isUse:false};
    await addOneData(TABLEENUM.验证码表, addInfo);

    return {code:Code};
}


/**
 * 个人会员列表
 * @param param0 
 */
export async function individualMemberList({name, documentId, phone, individualMemberType, memberLevel, session, joinTime, auditTime, memberState, pageNumber}) {
    let selectParam:any = {};
    if (name) selectParam.name = name;
    if (documentId) selectParam.documentId = documentId;
    if (phone) selectParam.phone = phone;
    if (individualMemberType) selectParam.individualMemberType = individualMemberType;
    if (memberLevel) selectParam.memberLevel = memberLevel;
    if (session) selectParam.session = session;
    if (joinTime) selectParam.joinTime = joinTime;
    if (auditTime) selectParam.auditTime = auditTime;
    if (memberState) selectParam.memberState = memberState;
    selectParam.memberType = MEMBERTYPE.个人会员;
    selectParam.userRegisterState = USERREGISTERSTATE.通过;

    let selectConf = ["userId", "memberState", "unitName", "name", "sex", "nation", "loginId", "phone", "joinTime", "memberLevel", "sheng", "shi", "qu"];
    let dbList = await findToPage(TABLEENUM.用户表, selectParam, selectConf, pageNumber);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    dbList.forEach(info => {
        let item:any = extractData(info, selectConf);
        item.memberState = changeEnumValue(MEMBERSTATE, item.memberState);
        if (!item.memberLevel) item.memberLevel = "未选择";
        else item.memberLevel = changeEnumValue(MEMBERLEVEL, item.memberLevel);
        dataList.push(item);
    });

    return {dataList, dataCount};
}


/**
 * 个人会员详情
 * @param param0 
 * @returns {顶部数据, 基本信息, 专业信息, 其他附件}
 */

const IndividualBasicConfig = ["name", "sex", "nation", "birth", "documentType", "documentId", "sheng", "shi", "qu", "addres", "photoUrl"];
const SpecialityConfig = ["unitName", "memberLevel", "workTitle", "profCategory", "studyResume", "workResume", "profAchievement", "otherEMP", "otherPROF"];

export async function individualMemberDetails({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    let topInfo:any = extractData(oldInfo, ["session", "memberLevel"]);
    topInfo.memberLevel = changeEnumValue(MEMBERLEVEL, topInfo.memberLevel);

    let basicInfo:any = extractData(oldInfo, IndividualBasicConfig);
    basicInfo.documentType = changeEnumValue(DOCUMENTTYPE, basicInfo.documentType);
    basicInfo.nation = changeEnumValue(NATION, basicInfo.nation);

    let specialityInfo:any = extractData(oldInfo, SpecialityConfig);
    basicInfo.profCategory = changeEnumValue(PROFCATEGORY, basicInfo.profCategory);
    
    let otherAttachment:any = extractData(oldInfo, ["cardUrl", "academicCERTUrl", "professionalCERTUrl", "auxiliaryMaterial"]);
    
    return {topInfo, basicInfo, specialityInfo, otherAttachment};
}


/**
 * 个人会员 修改
 * @param userId
 * @param form
 * @returns 
 */
 export async function individualMemberUpdate({form, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    await updateOneData(TABLEENUM.用户表, {userId}, form);

    return successResult();
}


/**
 * 会员注册【个人会员】【流程一】success
 * @param form 
 */
 export async function memberRegister1({form}) {
    let {loginId, mail, pwd, confirmation, phone, codeNum} = form;
    const NowMs = new Date().valueOf();
    /**校验密码 */
    if (pwd.search(/^[A-Za-z0-9]{6,18}$/) < 0) throw new BizError(ERRORENUM.密码只能由6至18位字符和数字组成);
    if (pwd != confirmation)  throw new BizError(ERRORENUM.两次密码不一致);
    
    /**校验去重 */
    let oldData = await findOnce(TABLEENUM.用户表, {"$or":[{loginId:loginId}, {phone:phone}, {mail:mail}] });
    if (oldData && oldData.userId) {
        if (oldData.loginId == loginId) throw new BizError(ERRORENUM.用户名重复, loginId );
        if (oldData.phone == phone) throw new BizError(ERRORENUM.联系人手机号重复, phone );
        if (oldData.mail == mail) throw new BizError(ERRORENUM.邮箱重复, loginId );
    }

    /**比对验证码 */
    let codeInfo = await findOnceToSort(TABLEENUM.验证码表, {phone, isUse:false, type:CODETYPE.用户注册, sendMs:{"$gt":sysTools.getTodayMs() }}, {sendMs:-1});
    if (!codeInfo) throw new BizError(ERRORENUM.code无效, `${phone}在注册时，code表中不存在该手机号的验证码`);
    if (codeInfo != codeNum) throw new BizError(ERRORENUM.验证码错误, `${phone}不存在${codeNum}`);
    if (( codeInfo.sendMs + (30 * 60 *1000) ) < NowMs ) throw new BizError(ERRORENUM.验证码过期);

    /**初始化账号信息 */
    let userId = sysTools.generateUserId();//系统生成唯一id
    let addInfo = {
        userId,
        memberType:MEMBERTYPE.个人会员, 
        isAdmin:STATE.否,
        registerFlow:REGISTERFLOW.完成第一步,//会员注册流程状态
        askForTime:NowMs,//申请时间
        loginId, mail, pwd, phone,
    };
    await addOneData(TABLEENUM.用户表, addInfo);

    await updateOneData(TABLEENUM.验证码表, {phone, code:codeNum, isUse:false }, {isUse:true});

    return {userId}
}


/**
 * 会员注册【个人会员】【流程2】success
 * @param form
 * @param userId  
 * @returns 
 */
export async function memberRegister2({form, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    const NowMs = new Date().valueOf();
    
    /**校验枚举 */
    const EccFuncName = '注册个人会员流程2';
    eccEnumValue(EccFuncName, '性别', SEX, form.sex);
    eccEnumValue(EccFuncName, '证件类型', DOCUMENTTYPE, form.documentType);
    eccEnumValue(EccFuncName, '民族', NATION, form.nation);
    eccEnumValue(EccFuncName, '专业类别', PROFCATEGORY, form.profCategory);
    eccEnumValue(EccFuncName, '会员等级职务', MEMBERLEVEL, form.memberLevel);
    
    form.registerFlow = REGISTERFLOW.完成第二步;
    await updateOneData(TABLEENUM.用户表, {userId}, form);
    
    /**用户提交注册申请，等待审核 */
    let approvalHistoryInfo = await find(TABLEENUM.审批历史表, {userId});
    let operationBehavior;
    /**判断是否第一次提交审核 */
    if(approvalHistoryInfo.length>1) operationBehavior = OPERATIONREHAVIOR.重新提交;
    else operationBehavior = OPERATIONREHAVIOR.用户提交;

    /**提交审批历史信息 */
    let addApprovalHistory = {
        id:generateSystemId(TABLEENUM.审批历史表, userId),
        userId,
        operationTime:NowMs,
        operationBehavior:operationBehavior,
        remarks:""
    };
    await addManyData(TABLEENUM.审批历史表, addApprovalHistory);

    return successResult();
}


/**
 * 会员注册【单位会员】【流程一】success
 * @param form 
 * @param codeId 
 */
 export async function unitMemberRegister1({form}) {
    let {loginId, mail, pwd, confirmation, phone, codeNum} = form;
    const NowMs = new Date().valueOf();
    /**校验密码 */
    if (pwd.search(/^[A-Za-z0-9]{6,18}$/) < 0) throw new BizError(ERRORENUM.密码只能由6至18位字符和数字组成);
    if (pwd != confirmation)  throw new BizError(ERRORENUM.两次密码不一致);

    /**校验去重 */
    let oldData = await findOnce(TABLEENUM.用户表, {"$or":[{loginId:loginId}, {phone:phone}, {mail:mail}] });
    if (oldData && oldData.userId) {
        if (oldData.loginId == loginId) throw new BizError(ERRORENUM.用户名重复, loginId );
        if (oldData.phone == phone) throw new BizError(ERRORENUM.联系人手机号重复, phone );
        if (oldData.mail == mail) throw new BizError(ERRORENUM.邮箱重复, loginId );
    }
    
    /**比对验证码 */
    let codeInfo = await findOnceToSort(TABLEENUM.验证码表, {phone, isUse:false, type:CODETYPE.用户注册, sendMs:{"$gt":sysTools.getTodayMs() }}, {sendMs:-1});
    if (!codeInfo) throw new BizError(ERRORENUM.code无效, `${phone}在注册时，code表中不存在该手机号的验证码`);
    if (codeInfo != codeNum) throw new BizError(ERRORENUM.验证码错误, `${phone}不存在${codeNum}`);
    if (( codeInfo.sendMs + (30 * 60 *1000) ) < NowMs ) throw new BizError(ERRORENUM.验证码过期);

    /**初始化账号信息 */
    let userId = sysTools.generateUserId();
    let addInfo = {
        userId,
        memberType:MEMBERTYPE.单位会员,
        isAdmin:STATE.否, //是否管理员
        registerFlow:REGISTERFLOW.完成第一步, //会员注册流程状态
        askForTime:NowMs, //申请时间
        loginId,
        mail,
        pwd,
        phone
    };
    await addOneData(TABLEENUM.用户表, addInfo);

    /**修改验证码状态为已使用 */
    await updateOneData(TABLEENUM.验证码表, {phone, code:codeNum, isUse:false }, {isUse:true});

    return {userId};
}


/**
 * 单位入会流程2
 * @param form 表单
 * @param keyanForm 科研表单
 * @param userId 
 * @returns 
 */
export async function unitMemberRegister2({form, keyanForm, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    /**校验枚举 */
    const EccFuncName = '注册单位会员流程2';
    eccEnumValue(EccFuncName, '专业类别', PROFCATEGORY, form.profCategory);
    eccEnumValue(EccFuncName, '院校办学类型', BANXUELEIXING, form.yuanXiaoBanXueLeiXing);
    eccEnumValue(EccFuncName, '单位信息类型', UNITINFOTYPE, form.unitInfoType);

    form.registerFlow = REGISTERFLOW.完成第二步;
    await updateOneData(TABLEENUM.用户表, {userId}, form);

    /**注册提交单位科研成果信息 */
    let addList = [];
    for (let i = 0; i < keyanForm.length; i++) {
        keyanForm[i].id = generateSystemId(TABLEENUM.单位所获得科研成果表, userId);
        keyanForm[i].userId = userId;
        addList.push(keyanForm[i]);
    }
    await addManyData(TABLEENUM.单位所获得科研成果表, addList);
    
    return successResult();
}


/**
 * 会员注册【单位会员】【流程3】success
 * @param application 入会申请表
 * @param userId 
 */
export async function unitMemberRegister3({application, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    const NowMs = new Date().valueOf();

    if (oldInfo.memberType != MEMBERTYPE.单位会员) throw new BizError(ERRORENUM.流程3不支持会员身份非单位会员);
    await updateOneData(TABLEENUM.用户表, {userId}, {application, registerFlow:REGISTERFLOW.完成第三步});

    /**用户提交注册申请，等待审核 */
    let approvalHistoryInfo = await find(TABLEENUM.审批历史表, {userId});
    let operationBehavior;
    /**判断是否第一次提交审核 */
    if(approvalHistoryInfo.length>1) operationBehavior = OPERATIONREHAVIOR.重新提交;
    else operationBehavior = OPERATIONREHAVIOR.用户提交;

    /**提交审批历史信息 */
    let addApprovalHistory = {
        id:generateSystemId(TABLEENUM.审批历史表, userId),
        userId,
        operationTime:NowMs,
        operationBehavior:operationBehavior,
        remarks:""
    };
    await addManyData(TABLEENUM.审批历史表, addApprovalHistory);

    return successResult();
}


/**
 * 下载会员注册表 todo
 */
export async function downloadApplication() {
    
}


/**
 * 获取当前会员注册流程
 * registerFlow：注册流程状态 REGISTERFLOW 会员注册流程步骤
 * userRegisterState：用户注册状态 USERREGISTERSTATE 会员注册是否已审批通过
 * 前端通过该方法获取注册流程状态，流程1通过返回状态【完成第一步】，流程2通过返回状态【完成第二步】，流程3通过返回状态【完成第三步】
 */
 export async function registerFlowType({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    if (!oldInfo.registerFlow) {
        await updateOneData(TABLEENUM.用户表, {userId}, {registerFlow:REGISTERFLOW.未注册});
    }

    let dataInfo = await findOnce(TABLEENUM.用户表, {userId});
    let dataList:any = extractData(dataInfo, ["userId", "registerFlow", "userRegisterState"]);

    return dataList;
}












/**
 * 模拟审核通过，修改审核状态
 */
//  export async function examine({userId}) {
//     let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
//     if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
//     let memberList:any = extractData(oldInfo, ["userId", "registerFlow", "memberType"]);

//     let selectParam = {registerFlow:memberList.registerFlow, userRegisterState:USERREGISTERSTATE.待审核};

//     if (memberList.memberType == MEMBERTYPE.个人会员) { //个人会员
//         switch (memberList.registerFlow) {
//             case REGISTERFLOW.个人会员注册1:
//                 selectParam.registerFlow = REGISTERFLOW.个人会员注册2;
//             break;
//             case REGISTERFLOW.个人会员注册2:
//                 selectParam.userRegisterState = USERREGISTERSTATE.通过;
//             break;
//         }
//     } else { //单位会员
//         switch (memberList.registerFlow) {
//             case REGISTERFLOW.单位会员注册1:
//                 selectParam.registerFlow = REGISTERFLOW.单位会员注册2;
//             break;
//             case REGISTERFLOW.单位会员注册2:
//                 selectParam.registerFlow = REGISTERFLOW.单位会员注册3;
//             break;
//             case REGISTERFLOW.单位会员注册3:
//                 selectParam.userRegisterState = USERREGISTERSTATE.通过;
//             break;
//         }
//     }

//     await updateOneData(TABLEENUM.用户表, {userId}, selectParam);

//     return successResult();
// }



