/**
 * 黑名单
 */

import moment = require("moment");
import { MEMBERLEVEL, MEMBERSTATE, MEMBERTYPE, STATE } from "../../config/enum";
import { TABLEENUM } from "../../data/models/model";
import { findCount, findOnce, findToPage } from "../../data/select";
import { extractData } from "../../util/piecemeal";
import { changeEnumValue } from "../../util/verificationEnum";
import { getCityNameByCode } from "../../config/cityConfig";
import { BizError } from "../../util/bizError";
import { ERRORENUM } from "../../config/errorEnum";
import { updateOneData } from "../../data/update";
import { successResult } from "../../tools/system";
import { deleteOneData } from "../../data/delete";


/**
 * 移入黑名单【操作在单位会员页、个人会员页】
 * @param id 会员Id 
 * @returns 
 */
 export async function moveInBlackUser({id}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId:id}, ["userId"]);
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    
    await updateOneData(TABLEENUM.用户表, {userId:id}, {isBlackUser:STATE.是});

    return successResult();
}


/**
 * 黑名单列表
 * @param param0 
 * @returns 
 */
export async function blackList({name, joinStartTime, joinEndTime, memberType, memberLevel, session, sheng, shi, qu, pageNumber}) {
    let selectParam:any = {
        isAdmin: STATE.否,
        isBlackUser: STATE.是
    };
    
    if (name) {
        selectParam["$or"] = [{name:{"$regex":name}}, {unitName:{"$regex":name}}];
    }
    if (joinStartTime) {
        selectParam["joinTime"] = {"$gt":joinStartTime};
    };
    if (joinEndTime) {
        if (!selectParam["joinTime"]) selectParam["joinTime"] = {};
        selectParam["joinTime"]["$lt"] = joinEndTime;
    };
    if (memberType && memberType.length) {
        selectParam["$and"] = [];
        selectParam["$and"].push({["$or"] : [ {unitMemberType:{"$in":memberType}}, {individualMemberType:{"$in":memberType}} ]});
    }
    if (memberLevel && memberLevel.length) selectParam.memberLevel = {"$in":memberLevel};
    if (session) selectParam.session = session;
    if (sheng) selectParam.sheng = sheng;
    if (shi) selectParam.shi = shi;
    if (qu) selectParam.qu = qu;

    const SelectFiles = ["userId", "memberState", "loginId", "phone", "joinTime", "memberLevel", "sheng", "shi", "qu", "unitName", "name", "memberType"];
    let dbList = await findToPage(TABLEENUM.用户表, selectParam, SelectFiles, pageNumber);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    const DataFiles = ["userId", "memberState", "loginId", "phone", "joinTime", "memberLevel", "sheng", "shi", "qu", "name", "memberType"];
    let dataList = [];
    dbList.forEach(info => {
        let item:any = extractData(info, DataFiles);
        if (info.memberType == MEMBERTYPE.单位会员) {
            item.name = info.unitName;
        }
        item.memberState = changeEnumValue(MEMBERSTATE, item.memberState);
        if (!item.memberLevel) item.memberLevel = "未选择";
        else item.memberLevel = changeEnumValue(MEMBERLEVEL, item.memberLevel);
        item.memberType = changeEnumValue(MEMBERTYPE, item.memberType);
        item.joinTime = moment(item.joinTime).format("YYYY-MM-DD");
        item.sheng = getCityNameByCode(item.sheng);
        item.shi = getCityNameByCode(item.shi);
        item.qu = getCityNameByCode(item.qu);
        dataList.push(item);
    });

    return {dataList, dataCount};
}


/**
 * 移除黑名单
 * @param id 会员Id 
 * @returns 
 */
export async function shiftOutBlackUser({id}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId:id}, ["userId"]);
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    
    await updateOneData(TABLEENUM.用户表, {userId:id}, {isBlackUser:STATE.否});

    return successResult();
}


/**
 * 删除信息
 * @param id 会员Id  
 * @returns 
 */
export async function delBlackUser({id}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId:id, isBlackUser:STATE.是}, ["userId"]);
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    await deleteOneData(TABLEENUM.用户表, {userId:id});
    return successResult();
}





