/**
 * 会员首页展板逻辑
 * 【二期内容：申请理事、申请常务理事、申请专家会员、会员积分、活动列表[立即参加]、消息中心暂时展示缴费提醒】
 */

import moment = require("moment");
import { MEMBERTYPE, STATE, UNITMEMBERTYPE, USERREGISTERSTATE } from "../../config/enum";
import { ERRORENUM } from "../../config/errorEnum";
import { addManyData } from "../../data/add";
import { deleteManyData } from "../../data/delete";
import { TABLEENUM } from "../../data/models/model";
import { find, findOnce } from "../../data/select";
import { updateOneData } from "../../data/update";
import { generateSystemId, successResult } from "../../tools/system";
import { BizError } from "../../util/bizError";
import { extractData } from "../../util/piecemeal";
import { eccEnumValue } from "../../util/verificationEnum";


/**
 * 会员信息展示
 * @param userId 登录的用户id
 * @returns 个人会员：{头部数据， 基本信息， 专业信息， 其他附件}
 * @returns 单位会员：{头部数据， 基本信息， 单位信息， 单位所获得科研成果}
 */
export async function getMemberData({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    let topSelectFiles = ["loginId", "memberType", "userRegisterState", "integral", "joinTime", "lifespanStartTime", "lifespanEndTime", "memberLevel"];
    let baseSelectFiles = [];
    let majorSelectFiles = [];
    let otherAttachmentFiles = [];
    /**判断会员类型，个人会员与单位会员返回不同字段 */
    if (oldInfo.memberType == MEMBERTYPE.单位会员) {
        baseSelectFiles = ["userId", "unitName", "uscc", "legalPerson", "legalPersonPhone", "legalPersonMail", "unitMail", "contactPerson", 
        "contactPersonDuties", "contactPersonPhone", "uusinessLicenseUrl", "addres", "sheng", "shi", "qu"];
        majorSelectFiles = ["unitMemberType", "yuanXiaoBanXueLeiXing", "yuanXiaoZhuGuanBuMen", "yuanXiaoFuZeRen", "yuanXiaoFuZeRenZhiWu", 
        "yuanXiaoFuZeRenDianHua", "yuanXiaoBanGongFuZeRen", "yuanXiaoBanGongFuZeRenDianHua", "ZhuYaoFuZeRenYouXiang", "yuanXiaoKeYanFuZeRen", 
        "yuanXiaoKeYanFuZeRenDianHua", "jiaoXueFuZeRenXinMing", "jiaoXueFuZeRenDianHua"];
        /**获取单位科研成果数据 */
        let oldKeYanList = await find(TABLEENUM.单位所获得科研成果表, {userId}, ["id", "startTime", "endTime", "describe"]);
        let otherAttachmentFiles = [];
        oldKeYanList.forEach(info => {
            otherAttachmentFiles.push({
                id:info.id,
                startTime: moment(info.startTime).format("YYYY-MM-DD"),
                endTime: moment(info.endTime).format("YYYY-MM-DD"),
                describe:info.describe
            });
        })

    } else {
        baseSelectFiles = ["userId", "name", "sex", "nation", "birth", "documentType", "documentId", "sheng", "shi", "qu", "phone", "mail", 
        "addres", "photoUrl", "education", "uscc", "unitName"];
        majorSelectFiles = ["unitName", "duties", "workTitle", "profCategory", "studyResume", "workResume", "profAchievement", "otherEMP", "otherPROF"]; 
        otherAttachmentFiles = ["cardUrl", "academicCERTUrl", "professionalCERTUrl", "auxiliaryMaterial"];
    }

    /**返回指定配置字段 */
    let topInfo:any = extractData(oldInfo, topSelectFiles);
    let baseInfo:any = extractData(oldInfo, baseSelectFiles);
    let majorInfo:any = extractData(oldInfo, majorSelectFiles);
    let otherAttachment:any = extractData(oldInfo, otherAttachmentFiles);

    /**处理枚举值 */
    topInfo.integral = 0;
    if(topInfo.joinTime) topInfo.joinTime = moment(topInfo.joinTime).format("YYYY-MM-DD");
    if(topInfo.lifespanEndTime) topInfo.lifespanEndTime = moment(topInfo.lifespanEndTime).format("YYYY-MM-DD");
    if(topInfo.userRegisterState == USERREGISTERSTATE.通过) topInfo.userRegisterState = "已入会";
    else topInfo.userRegisterState = "未入会";
    // if(topInfo.memberLevel) topInfo.memberLevel = changeEnumValue(MEMBERLEVEL, topInfo.memberLevel);
    // if(baseInfo.sex) baseInfo.sex = changeEnumValue(SEX, baseInfo.sex);
    // if(baseInfo.documentType) baseInfo.documentType = changeEnumValue(DOCUMENTTYPE, baseInfo.documentType);
    // if(baseInfo.nation) baseInfo.nation = changeEnumValue(NATION, baseInfo.nation);
    // if(majorInfo.profCategory) majorInfo.profCategory = changeEnumValue(PROFCATEGORY, majorInfo.profCategory);
    // if(majorInfo.unitMemberType) majorInfo.unitMemberType = changeEnumValue(UNITMEMBERTYPE, majorInfo.unitMemberType);
    // if(majorInfo.yuanXiaoBanXueLeiXing) majorInfo.yuanXiaoBanXueLeiXing = changeEnumValue(BANXUELEIXING, majorInfo.yuanXiaoBanXueLeiXing);

    return {topInfo, baseInfo, majorInfo, otherAttachment};
}


/**
 * 单位会员 修改
 * @param form 修改信息表单
 * @param keyanForm 修改科研信息表单
 * @param userId 登录的用户id
 * @returns 
 */
export async function unitMemberUpdate({form, keyanForm, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    /**校验枚举 */
    const EccFuncName = '单位会员修改';
    eccEnumValue(EccFuncName, '单位信息类型', UNITMEMBERTYPE, form.unitMemberType);

    /**修改会员用户信息 */
    await updateOneData(TABLEENUM.用户表, {userId}, form);

    /**删除对应userId科研成果 */
    let oldKeYanList = await find(TABLEENUM.单位所获得科研成果表, {userId});
    if (!oldKeYanList || !oldKeYanList.id) throw new BizError(ERRORENUM.目标数据不存在);
    await deleteManyData(TABLEENUM.单位所获得科研成果表, {userId});

    /**添加新的科研信息 */
    let addList = [];
    for (let i = 0; i < keyanForm.length; i++) {
        keyanForm[i].id = generateSystemId(TABLEENUM.单位所获得科研成果表, userId);
        keyanForm[i].userId = userId;
        addList.push(keyanForm);
    }
    await addManyData(TABLEENUM.单位所获得科研成果表, addList);

    return successResult();
}


/**
 * 个人会员 修改
 * @param userId 登录的用户id
 * @param form 修改信息表单
 * @returns 
 */
 export async function individualMemberUpdate({form, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    await updateOneData(TABLEENUM.用户表, {userId}, form);

    return successResult();
}


/**
 * 消息中心列表
 * 缴费消息提醒
 * @param userId 登录的用户id
 */
export async function getXiaoXiZhongXin ({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId}, ["userId", "lifespanEndTime", "isGracePeriod", "gracePeriodEndTime", "lifespanStartTime"]);
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    let dataInfo:any = extractData(oldInfo, ["lifespanStartTime", "lifespanEndTime"]);

    const NowMs = new Date().valueOf();

    /**判断是否宽限期 */
    if (dataInfo.isGracePeriod == STATE.否) {
        /**
         * 判断会费到期时间
         */
        switch (oldInfo.lifespanEndTime) {
            case oldInfo.lifespanEndTime > NowMs+(90*24*3600*1000):
                dataInfo["membershipFees"] = "大于90天";
            break;
            case oldInfo.lifespanEndTime-(90*24*3600*1000) <= NowMs && oldInfo.lifespanEndTime > NowMs-(24*3600*1000):
                dataInfo["membershipFees"] = "不足90天";
            break;
            case NowMs <= oldInfo.lifespanEndTime+(30*24*3600*1000) && NowMs > oldInfo.lifespanEndTime+(24*3600*1000):
                dataInfo["membershipFees"] = "过期30天内";
            break;
            case NowMs > oldInfo.lifespanEndTime+(30*24*3600*1000) && NowMs <= oldInfo.lifespanEndTime+(90*24*3600*1000):
                dataInfo["membershipFees"] = "过期90天内";
            break;
            case NowMs > oldInfo.lifespanEndTime+(90*24*3600*1000) && NowMs <= oldInfo.lifespanEndTime+(365*24*3600*1000):
                dataInfo["membershipFees"] = "过期1年内";
            break;
            case NowMs > oldInfo.lifespanEndTime+(365*24*3600*1000):
                dataInfo["membershipFees"] = "过期1年外";
            break;
        }
    } else {
        if (oldInfo.gracePeriodEndTime > NowMs) dataInfo["membershipFees"] = "宽限期中";
    }

    return {dataInfo};
}


/**
 * 地图省数量统计
 * @param userId 登录的用户id 
 * @returns 
 */
export async function getSheng ({userId}) {
    let myOldInfo = await findOnce(TABLEENUM.用户表, {userId}, ["sheng"]);
    if (!myOldInfo || !myOldInfo.sheng) throw new BizError(ERRORENUM.目标数据不存在);
    let myDataInfo = extractData(myOldInfo, ["sheng"]);

    let allOldList = await find(TABLEENUM.用户表, {}, ["sheng"]);
    if (!allOldList || !allOldList.length) throw new BizError(ERRORENUM.目标数据不存在);
    
    let allDataList = {};
    allOldList.forEach( info => {
        //初始数据包含没有填省份的，过滤省份为undefined的数据，统计省份个数
        if (info.sheng) {
            if (!allDataList[info.sheng]) allDataList[info.sheng] = 0;
            allDataList[info.sheng] += 1;
        }
    })

    return {myDataInfo, allDataList};
}


/**
 * 活动列表
 */
 export async function getHuoDongTongZhi () {
    let oldList = await find(TABLEENUM.活动通知, {}, ["id", "name"]);
    if (!oldList || oldList.length<1) throw new BizError(ERRORENUM.目标数据不存在);
    
    let dataList = [];
    oldList.forEach( info => {
        let item = extractData(info, ["id", "name"]);
        dataList.push(item);
    })

    return {dataList};
}






