/**
 * 用户逻辑
 */

import { DOCUMENTTYPE, MEMBERTYPE, NATION, PROFCATEGORY, SEX, UNITINFOTYPE } from "../../config/enum";
import { ERRORCODEENUM, ERRORENUM } from "../../config/errorEnum";
import { addOneData } from "../../data/add";
import { TABLEENUM } from "../../data/models/model";
import { findOnce } from "../../data/select";
import { updateOneData } from "../../data/update";
import { generateToken, generateUserId, successResult } from "../../tools/system";
import { BizError } from "../../util/bizError";
import { eccEnumValue } from "../../util/verificationEnum";


/**
 * 注册个人会员
 * @param param0 
 */
export async function signInOfEntity({form}) {
    /**校验枚举 */
    const EccFuncName = '注册个人会员';
    eccEnumValue(EccFuncName, '性别', SEX, form.sex);
    eccEnumValue(EccFuncName, '证件类型', DOCUMENTTYPE, form.documentType);
    eccEnumValue(EccFuncName, '民族', NATION, form.nation);
    eccEnumValue(EccFuncName, '专业类别', PROFCATEGORY, form.profCategory);

    let oldData = await findOnce(TABLEENUM.用户表, {"$or":[{loginId:form.loginId}, {phone:form.phone}, {mail:form.mail}] });
    if (oldData && oldData.userId) {
        if (oldData.loginId == form.loginId) throw new BizError(ERRORENUM.用户名重复, form.loginId );
        if (oldData.phone == form.phone) throw new BizError(ERRORENUM.联系人手机号重复, form.phone );
        if (oldData.mail == form.mail) throw new BizError(ERRORENUM.邮箱重复, form.loginId );
    }

    /**校验密码 */
    if (form.pwd.search(/^[A-Za-z0-9]{6,18}$/) < 0) throw new BizError(ERRORENUM.密码只能由6至18位字符和数字组成);
    if (form.pwd != form.confirmation)  throw new BizError(ERRORENUM.两次密码不一致);

    delete form.confirmation;
    
    form.userId = generateUserId();
    form.memberType = MEMBERTYPE.个人会员;
    form.isAdmin = 0;

    await addOneData(TABLEENUM.用户表, form);

    return successResult();
}

/**
 * 单位会员
 * @param param0 
 */
export async function singInOfUnit({form}) {
    /**校验枚举 */
    const EccFuncName = '注册单位会员';
    eccEnumValue(EccFuncName, '单位信息类型', UNITINFOTYPE, form.unitInfoType);

    let oldData = await findOnce(TABLEENUM.用户表, {"$or":[{loginId:form.loginId}, {phone:form.phone}, {mail:form.mail}] });
    if (oldData && oldData.userId) {
        if (oldData.loginId == form.loginId) throw new BizError(ERRORENUM.用户名重复, form.loginId );
        if (oldData.phone == form.phone) throw new BizError(ERRORENUM.联系人手机号重复, form.phone );
        if (oldData.mail == form.mail) throw new BizError(ERRORENUM.邮箱重复, form.loginId );
    }

    /**校验密码 */
    if (form.pwd.search(/^[A-Za-z0-9]{6,18}$/) < 0) throw new BizError(ERRORENUM.密码只能由6至18位字符和数字组成);
    if (form.pwd != form.confirmation)  throw new BizError(ERRORENUM.两次密码不一致);

    delete form.confirmation;
    
    form.userId = generateUserId();
    form.memberType = MEMBERTYPE.单位会员;
    form.isAdmin = 0;

    await addOneData(TABLEENUM.用户表, form);

    return successResult();
}


/**
 * 登陆
 * @param param0 
 * @returns 
 */
export async function login({loginId, pwd }) {
    let userInfo = await findOnce(TABLEENUM.用户表, {loginId});
    
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.账号或密码错误);
    if (userInfo.pwd != pwd ) throw new BizError(ERRORENUM.账号或密码错误);

    let token = generateToken(userInfo.userId);
    await updateOneData(TABLEENUM.用户表, {userId:userInfo.userId}, {token, tokenMs:new Date().valueOf()});

    return {
        userInfo:{
            isAdmin:userInfo.isAdmin == 1,
            name:userInfo.name,
            userId:userInfo.userId,
            token:token
        }
    }
}


/**
 * 退出登陆
 * @param userId 
 * @returns 
 */
export async function logout({userId}) {
    let token = generateToken(userId);
    await updateOneData(TABLEENUM.用户表, {userId:userId}, {token});

    return successResult();
}



