/**
 * 会员逻辑
 */

import { MEMBERSTATE, MEMBERLEVEL, DOCUMENTTYPE, NATION, PROFCATEGORY, CODETYPE, UNITINFOTYPE, MEMBERTYPE, REGISTERFLOW, SEX } from "../../config/enum";
import { ERRORENUM } from "../../config/errorEnum";
import { TABLEENUM } from "../../data/models/model";
import { find, findCount, findOnce, findToPage } from "../../data/select";
import { updateOneData } from "../../data/update";
import { generateSystemId, successResult } from "../../tools/system";
import { BizError } from "../../util/bizError";
import { extractData } from "../../util/piecemeal";
import { changeEnumValue, eccEnumValue } from "../../util/verificationEnum";
import * as sysTools from "../../tools/system";
import { sendRegisterCode } from "../sms";
import { addManyData, addOneData } from "../../data/add";
import { deleteManyData, deleteOneData } from "../../data/delete";


/**
 * 单位会员列表
 * @param param0 
 */
export async function unitMemberList({unitName, joinTime, unitMemberType, workDuties, session, sheng, shi, qu, pageNumber}) {
    let selectParam:any = {};
    if (unitName) selectParam.unitName = unitName;
    if (joinTime) selectParam.joinTime = joinTime;
    if (unitMemberType) selectParam.unitMemberType = unitMemberType;
    if (workDuties) selectParam.workDuties = workDuties;
    if (session) selectParam.session = session;
    if (sheng) selectParam.sheng = sheng;
    if (shi) selectParam.shi = shi;
    if (qu) selectParam.qu = qu;

    let dbList = await findToPage(TABLEENUM.用户表, selectParam, [], pageNumber);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    dbList.forEach(info => {
        let item:any = extractData(info, ["userId", "memberState", "unitName", "loginId", "phone", "joinTime", "unitMemberLevel", "sheng", "shi", "qu"]);
        changeEnumValue(MEMBERSTATE, item.memberState);
        if (!item.memberLevel) item.memberLevel = "未选择";
        else changeEnumValue(MEMBERLEVEL, item.memberLevel);
        dataList.push(item);
    });

    return {dataList, dataCount};
}


/**
 * 单位会员详情
 * @param param0 
 * @returns {顶部数据, 基本信息, 单位信息, 单位所获得科研成果}
 */

const UnitBasicConfig = ["userId", "unitName", "addres", "officialWebsite", "uscc", "legalPerson", "legalPersonPhone", "gongZhongHao", 
"unitPhone", "contactPerson", "contactPersonDuties", "sheng", "shi", "qu", "uusinessLicenseUrl", "personInChargeDesc"];
const UnitConfig = ["unitInfoType", "yuanXiaoBanXueLeiXing", "yuanXiaoZhuGuanBuMen", "yuanXiaoFuZeRen", "yuanXiaoFuZeRenZhiWu", 
"yuanXiaoFuZeRenDianHua", "yuanXiaoBanGongFuZeRen", "yuanXiaoBanGongFuZeRenZhiWu", "yuanXiaoBanGongFuZeRenDianHua", "yuanXiaoKeYanFuZeRen", 
"yuanXiaoKeYanFuZeRenZhiWu", "yuanXiaoKeYanFuZeRenDianHua", "yuanXiaoXueShengZongRenShu", "yuanXiaoJiaoZhiGongZongRenShu", "yuanXiaoJianZhuMianJi", 
"yuanXiaoGaoJiZhiCheng", "yuanXiaoKaiSheZhuanYe"];

export async function unitMemberDetails({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    let topInfo:any = extractData(oldInfo, ["session", "memberLevel"]);
    changeEnumValue(MEMBERLEVEL, topInfo.memberLevel);
    let basicInfo = extractData(oldInfo, UnitBasicConfig);
    let unitInfo = extractData(oldInfo, UnitConfig);
    
    let oldKeYanInfo = await findOnce(TABLEENUM.单位所获得科研成果表, {userId});
    let scientificResearch = extractData(oldKeYanInfo, ["id", "startTime", "endTime", "describe"]);
    
    return {topInfo, basicInfo, unitInfo, scientificResearch};
}


/**
 * 单位会员 修改
 * @param userId
 * @param form
 * @returns 
 */
export async function unitMemberUpdate({form, keyanForm, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});

    /**校验枚举 */
    changeEnumValue(UNITINFOTYPE, form.unitInfoType);
    changeEnumValue(UNITINFOTYPE, form.unitInfoType);

    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    await updateOneData(TABLEENUM.用户表, {userId}, form);

    /**删除对应userId科研成果 */
    let oldKeYanInfo = await find(TABLEENUM.单位所获得科研成果表, {userId});
    if (!oldKeYanInfo || !oldKeYanInfo.id) throw new BizError(ERRORENUM.目标数据不存在);
    await deleteManyData(TABLEENUM.单位所获得科研成果表, {userId});

    let addList = [];
    for (let i = 0; i < keyanForm.length; i++) {
        keyanForm[i].id = generateSystemId(TABLEENUM.单位所获得科研成果表, userId);
        keyanForm[i].userId = userId;
        addList.push(keyanForm);
    }
    await addManyData(TABLEENUM.单位所获得科研成果表, addList);

    return successResult();
}


/**
 * 删除单位所获得科研成果
 * @param param0 
 * @returns 
 */
export async function delResearch({id}) {
    let dataInfo = await findOnce(TABLEENUM.单位所获得科研成果表, {id});
    if (!dataInfo || !dataInfo.id) throw new BizError(ERRORENUM.目标数据不存在);
    await deleteOneData(TABLEENUM.单位所获得科研成果表, {id});
    return successResult();
}


/**
 * 会员注册的短信验证码
 * @param userId 
 * @param phone 
 * @returns 
 */
export async function memberRegisterSendCode({loginId, phone, codeType}) {
    let todayMs = sysTools.getTodayMs();
    let todayCodeList = await find(TABLEENUM.验证码表, {phone}) || [];
    // if (todayCodeList.length >= 4) throw new BizError(ERRORENUM.发送验证码次数超限制, `${loginId} 修改密码发送验证码次数超限制4`);

    let sendMs = todayMs;
    todayCodeList.forEach(info => {
        sendMs = Math.max(sendMs, info.sendMs);
    });

    let now = new Date().valueOf();
    if ((now - sendMs) <= (60 * 1000) ) throw new BizError(ERRORENUM.发送验证码频率过快, `${loginId}`);

    let codeId = sysTools.getSMSCodeId(loginId, todayCodeList.length||0);
    let code = sysTools.getSMSCode();

    //await sendRegisterCode(phone, code, codeType); //todo 发送短信
    now = new Date().valueOf();

    await addOneData(TABLEENUM.验证码表, {codeId, codeNum:code, sendMs:now, phone, type:codeType, isUse:true});

    let dataList = {codeId}

    return dataList;
}


/**
 * 个人会员列表
 * @param param0 
 */
export async function individualMemberList({name, documentId, phone, memberLevel, workDuties, session, joinTime, auditTime, memberState, pageNumber}) {
    let selectParam:any = {};
    if (name) selectParam.name = name;
    if (documentId) selectParam.documentId = documentId;
    if (phone) selectParam.phone = phone;
    if (memberLevel) selectParam.memberLevel = memberLevel;
    if (workDuties) selectParam.workDuties = workDuties;
    if (session) selectParam.session = session;
    if (joinTime) selectParam.joinTime = joinTime;
    if (auditTime) selectParam.auditTime = auditTime;
    if (memberState) selectParam.memberState = memberState;

    let selectConf = ["userId", "memberState", "unitName", "name", "sex", "nation", "loginId", "phone", "joinTime", "memberLevel", "sheng", "shi", "qu"];
    let dbList = await findToPage(TABLEENUM.用户表, selectParam, selectConf, pageNumber, 10);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    dbList.forEach(info => {
        let item:any = extractData(info, selectConf);
        changeEnumValue(MEMBERSTATE, item.memberState);
        if (!item.memberLevel) item.memberLevel = "未选择";
        else changeEnumValue(MEMBERLEVEL, item.memberLevel);
        dataList.push(item);
    });

    return {dataList, dataCount};
}


/**
 * 个人会员详情
 * @param param0 
 * @returns {顶部数据, 基本信息, 专业信息, 其他附件}
 */

const IndividualBasicConfig = ["name", "sex", "nation", "birth", "documentType", "documentId", "sheng", "shi", "qu", "addres", "photoUrl"];
const SpecialityConfig = ["unitName", "workDuties", "workTitle", "profCategory", "studyResume", "workResume", "profAchievement", "otherEMP", "otherPROF"];

export async function individualMemberDetails({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    let topInfo:any = extractData(oldInfo, ["session", "memberLevel"]);
    changeEnumValue(MEMBERLEVEL, topInfo.memberLevel);

    let basicInfo:any = extractData(oldInfo, IndividualBasicConfig);
    changeEnumValue(DOCUMENTTYPE, basicInfo.documentType);
    changeEnumValue(NATION, basicInfo.nation);

    let specialityInfo:any = extractData(oldInfo, SpecialityConfig);
    changeEnumValue(PROFCATEGORY, basicInfo.profCategory);
    
    let otherAttachment:any = extractData(oldInfo, ["cardUrl", "academicCERTUrl", "professionalCERTUrl", "auxiliaryMaterial"]);
    
    return {topInfo, basicInfo, specialityInfo, otherAttachment};
}


/**
 * 个人会员 修改
 * @param userId
 * @param form
 * @returns 
 */
 export async function individualMemberUpdate({form, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    await updateOneData(TABLEENUM.用户表, {userId}, form);

    return successResult();
}


/**
 * 个人入会流程1
 * @param secureArgs 
 */
 export async function memberRegister1({form, codeId}) {
    let oldData = await findOnce(TABLEENUM.用户表, {"$or":[{loginId:form.loginId}, {phone:form.phone}, {mail:form.mail}] });
    if (oldData && oldData.userId) {
        if (oldData.loginId == form.loginId) throw new BizError(ERRORENUM.用户名重复, form.loginId );
        if (oldData.phone == form.phone) throw new BizError(ERRORENUM.联系人手机号重复, form.phone );
        if (oldData.mail == form.mail) throw new BizError(ERRORENUM.邮箱重复, form.loginId );
    }

    /**校验密码 */
    if (form.pwd.search(/^[A-Za-z0-9]{6,18}$/) < 0) throw new BizError(ERRORENUM.密码只能由6至18位字符和数字组成);
    if (form.pwd != form.confirmation)  throw new BizError(ERRORENUM.两次密码不一致);
    
    form.userId = sysTools.generateUserId();
    form.memberType = MEMBERTYPE.个人会员;
    form.isAdmin = 0;
    form.registerFlow = REGISTERFLOW.个人会员注册1;

    // let todayMs = sysTools.getTodayMs();
    let codeList = await findOnce(TABLEENUM.验证码表, {codeId});

    let now = new Date().valueOf();

    let msg = ERRORENUM.验证码错误;
    if (codeList.codeNum == form.codeNum) {
        if (codeList.isUse) msg = ERRORENUM.验证码失效;
        else if ( (now - codeList.sendMs) > (30 * 60 * 1000) ) msg = ERRORENUM.验证码过期
    }
    
    if (!codeId) throw new BizError(msg, `userId:${form.userId}个人入会注册流程1的code:${form.code}`);
    /**修改验证码状态为已使用 */
    await updateOneData(TABLEENUM.验证码表, {codeId}, {isUse:true});

    await addOneData(TABLEENUM.用户表, form);

    /**注册成功，返回userId和会员注册流程字段 */
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId:form.userId});
    let dataList = extractData(oldInfo, ["userId", "registerFlow"]);

    return dataList;
}


/**
 * 个人入会流程2
 * @param userId 
 * @param secureArgs 
 * @returns 
 */
export async function memberRegister2({form, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    
    /**校验枚举 */
    const EccFuncName = '注册个人会员';
    eccEnumValue(EccFuncName, '性别', SEX, form.sex);
    eccEnumValue(EccFuncName, '证件类型', DOCUMENTTYPE, form.documentType);
    eccEnumValue(EccFuncName, '民族', NATION, form.nation);
    eccEnumValue(EccFuncName, '专业类别', PROFCATEGORY, form.profCategory);
    
    form.registerFlow = REGISTERFLOW.个人会员注册2;
    await updateOneData(TABLEENUM.用户表, {userId}, form);
    
    let unitInfo = await findOnce(TABLEENUM.用户表, {userId});
    let dataList = extractData(unitInfo, ["userId", "registerFlow"]);

    return dataList;
}




/**
 * 单位入会流程1
 * @param secureArgs 
 */
 export async function unitMemberRegister1({form, codeId}) {
    let oldData = await findOnce(TABLEENUM.用户表, {"$or":[{loginId:form.loginId}, {phone:form.phone}, {mail:form.mail}] });
    if (oldData && oldData.userId) {
        if (oldData.loginId == form.loginId) throw new BizError(ERRORENUM.用户名重复, form.loginId );
        if (oldData.phone == form.phone) throw new BizError(ERRORENUM.联系人手机号重复, form.phone );
        if (oldData.mail == form.mail) throw new BizError(ERRORENUM.邮箱重复, form.loginId );
    }

    /**校验密码 */
    if (form.pwd.search(/^[A-Za-z0-9]{6,18}$/) < 0) throw new BizError(ERRORENUM.密码只能由6至18位字符和数字组成);
    if (form.pwd != form.confirmation)  throw new BizError(ERRORENUM.两次密码不一致);
    
    form.userId = sysTools.generateUserId();
    form.memberType = MEMBERTYPE.单位会员;
    form.isAdmin = 0;
    form.registerFlow = REGISTERFLOW.单位会员注册1;

    // let todayMs = sysTools.getTodayMs();

    let codeList = await findOnce(TABLEENUM.验证码表, {codeId});

    let now = new Date().valueOf();

    let msg = ERRORENUM.验证码错误;
    if (codeList.codeNum == form.codeNum) {
        if (codeList.isUse) msg = ERRORENUM.验证码失效;
        else if ( (now - codeList.sendMs) > (30 * 60 * 1000) ) msg = ERRORENUM.验证码过期
    }
    
    if (!codeId) throw new BizError(msg, `userId:${form.userId}单位入会注册流程1的code:${form.code}`);
    /**修改验证码状态为已使用 */
    await updateOneData(TABLEENUM.验证码表, {codeId}, {isUse:true});

    await addOneData(TABLEENUM.用户表, form);

    /**注册成功，返回userId和会员注册流程字段 */
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId:form.userId});
    let dataList = extractData(oldInfo, ["userId", "registerFlow"]);

    return dataList;
}


/**
 * 单位入会流程2
 * @param userId 
 * @param secureArgs 
 * @returns 
 */
export async function unitMemberRegister2({form, keyanForm, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    
    form.registerFlow = REGISTERFLOW.单位会员注册2;
    await updateOneData(TABLEENUM.用户表, {userId}, form);
    
    let unitInfo = await findOnce(TABLEENUM.用户表, {userId});
    let dataList = extractData(unitInfo, ["userId", "registerFlow"]);

    return dataList;
}






