/**
 * 会员管理逻辑
 */

import { MEMBERSTATE, MEMBERLEVEL, DOCUMENTTYPE, NATION, PROFCATEGORY, MEMBERTYPE, USERREGISTERSTATE, STATE, UNITMEMBERTYPE, ADMINLV, ADMINTYPE } from "../../config/enum";
import { ERRORENUM } from "../../config/errorEnum";
import { TABLEENUM } from "../../data/models/model";
import { find, findCount, findOnce, findOnceToSort, findToPage, findToSort } from "../../data/select";
import { updateOneData } from "../../data/update";
import { generateSystemId, getEdition, successResult } from "../../tools/system";
import { BizError } from "../../util/bizError";
import { extractData } from "../../util/piecemeal";
import { changeEnumValue, eccEnumValue } from "../../util/verificationEnum";
import { addManyData, addOneData } from "../../data/add";
import { deleteManyData, deleteOneData } from "../../data/delete";
import moment = require("moment");
import { getCityNameByCode } from "../../config/cityConfig";
import { eccFormParam } from "../../util/verificationParam";


/**
 * 单位会员列表 success
 * @param unitName 单位名称 
 * @param joinTime 入会时间
 * @param unitMemberType 单位会员所属类型 UNITMEMBERTYPE
 * @param memberLevel 会员等级 职务 MEMBERLEVEL
 * @param session 届次 
 * @param sheng 省
 * @param shi 市
 * @param qu 区
 * @param pageNumber 当前页
 */
export async function unitMemberList({unitName, joinTime, unitMemberType, memberLevel, session, sheng, shi, qu, pageNumber}) {
    let selectParam:any = {isAdmin:STATE.否};
    if (unitName) selectParam.unitName = unitName;
    if (joinTime) selectParam.joinTime = joinTime;
    if (unitMemberType) selectParam.unitMemberType = unitMemberType;
    if (memberLevel) selectParam.memberLevel = memberLevel;
    if (session) {
        let thisYear = moment(session).format("YYYY");
        let thisMonth = moment(session).format("MM");
        selectParam.session = getEdition(thisYear, thisMonth);;
    }
    if (sheng) selectParam.sheng = sheng;
    if (shi) selectParam.shi = shi;
    if (qu) selectParam.qu = qu;
    selectParam.memberType = MEMBERTYPE.单位会员;
    selectParam.userRegisterState = USERREGISTERSTATE.通过;

    const SelectFiles = ["userId", "memberState", "unitName", "loginId", "phone", "joinTime", "unitMemberLevel", "sheng", "shi", "qu"];
    let dbList = await findToPage(TABLEENUM.用户表, selectParam, SelectFiles, pageNumber);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    dbList.forEach(info => {
        let item:any = extractData(info, SelectFiles);
        item.memberState = changeEnumValue(MEMBERSTATE, item.memberState);
        if (!item.memberLevel) item.memberLevel = "未选择";
        else item.memberLevel = changeEnumValue(MEMBERLEVEL, item.memberLevel);
        item.joinTime = moment(item.joinTime).format("YYYY-MM-DD");
        dataList.push(item);
    });

    return {dataList, dataCount};
}


/**
 * 单位会员详情 
 * @param id 会员列表返回的userId 
 * @returns {顶部数据, 基本信息, 单位信息, 单位所获得科研成果}
 * 基本信息: "unitName"：单位名称、原社会信用代码, "uscc"：单位信用代码、原发票抬头, "legalPerson"：法人代表, "legalPersonPhone"：法人代表联系电话, "legalPersonMail"：法人邮箱, "unitMail"：单位电子邮箱, "contactPerson"：日常联系人, "contactPersonDuties"：日常联系人职务, "contactPersonPhone"：日常联系人电话, "uusinessLicenseUrl"：营业执照图片地址, "addres"：通信地址, "sheng"：省, "shi"：市, "qu"：区, "newUscc"：新社会信用代码, "newUnitName"：新发票抬头
 * 单位信息: "unitMemberType"：单位类型, "yuanXiaoBanXueLeiXing"：办学类型, "yuanXiaoZhuGuanBuMen"：主管部门, "yuanXiaoFuZeRen"：主要负责人姓名, "yuanXiaoFuZeRenZhiWu"：主要负责人职位, "yuanXiaoFuZeRenDianHua"：主要负责人电话, 
"yuanXiaoBanGongFuZeRen"：办公负责人姓名, "yuanXiaoBanGongFuZeRenDianHua"：办公负责人电话, "ZhuYaoFuZeRenYouXiang"：主要负责人邮箱, "yuanXiaoKeYanFuZeRen"：科研负责人姓名, "yuanXiaoKeYanFuZeRenDianHua"：看科研负责人电话,
"jiaoXueFuZeRenXinMing"：教学负责人姓名, "jiaoXueFuZeRenDianHua"：教学负责人电话
 * 单位所获得科研成果: "id", "startTime"：开始时间, "endTime"：结束时间, "describe"：描述
 */

const UnitBasicConfig = ["userId", "unitName", "uscc", "legalPerson", "legalPersonPhone", "legalPersonMail", "unitMail", "contactPerson", "contactPersonDuties", "contactPersonPhone", 
"uusinessLicenseUrl", "addres", "sheng", "shi", "qu"];
const UnitConfig = ["unitMemberType", "yuanXiaoBanXueLeiXing", "yuanXiaoZhuGuanBuMen", "yuanXiaoFuZeRen", "yuanXiaoFuZeRenZhiWu", "yuanXiaoFuZeRenDianHua", 
"yuanXiaoBanGongFuZeRen", "yuanXiaoBanGongFuZeRenDianHua", "ZhuYaoFuZeRenYouXiang", "yuanXiaoKeYanFuZeRen", "yuanXiaoKeYanFuZeRenDianHua",
"jiaoXueFuZeRenXinMing", "jiaoXueFuZeRenDianHua"];

export async function unitMemberDetails({id}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId:id});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    let topInfo:any = extractData(oldInfo, ["session", "memberLevel"]);
    /**处理枚举值 */
    topInfo.memberLevel = changeEnumValue(MEMBERLEVEL, topInfo.memberLevel);
    let basicInfo:any = extractData(oldInfo, UnitBasicConfig);
    /**获取订单表新社会信用代码和新发票抬头 */
    let orderInfo = await findOnce(TABLEENUM.订单表, {userId:id});
    if (!orderInfo.newUscc || ! orderInfo.newUnitName) {
        basicInfo.newUscc = "-";
        basicInfo.newUnitName = "-";
    } else {
        basicInfo.newUscc = orderInfo.newUscc;
        basicInfo.newUnitName = orderInfo.newUnitName;
    }
    let unitInfo = extractData(oldInfo, UnitConfig);
    
    let oldKeYanList = await find(TABLEENUM.单位所获得科研成果表, {userId:id}, ["id", "startTime", "endTime", "describe"]);
    let scientificResearch = [];
    oldKeYanList.forEach(info => {
        scientificResearch.push({
            id:info.id,
            startTime:info.startTime,
            endTime:info.endTime,
            describe:info.describe
        });
    })
    
    return {topInfo, basicInfo, unitInfo, scientificResearch};
}


/**
 * 单位会员 修改
 * @param form 修改信息表单
 * @param keyanForm 修改科研信息表单
 * @param id 会员列表返回的userId  
 * @returns 
 */
export async function unitMemberUpdate({form, keyanForm, id}) {
    const keyanFormItemConfig = {
        startTime: 'Number',
        endTime: 'Number',
        describe: 'String'
    };
    keyanForm.forEach(item => {
        eccFormParam("单位会员修改", keyanFormItemConfig, item );
    });
    
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId:id});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    /**校验枚举 */
    const EccFuncName = '单位会员修改';
    eccEnumValue(EccFuncName, '单位信息类型', UNITMEMBERTYPE, form.unitMemberType);
    
    /**修改会员用户信息 */
    await updateOneData(TABLEENUM.用户表, {userId:id}, form);

    /**删除对应userId科研成果 */
    let oldKeYanList= await find(TABLEENUM.单位所获得科研成果表, {userId:id});
    if (oldKeYanList && oldKeYanList.length) {
        await deleteManyData(TABLEENUM.单位所获得科研成果表, {userId:id});
    }
    

    /**添加新的科研信息 */
    let addList = [];
    for (let i = 0; i < keyanForm.length; i++) {
        let { startTime, endTime, describe } = keyanForm[i];
        addList.push({
            id:generateSystemId(TABLEENUM.单位所获得科研成果表, id),
            userId:id,
            startTime, endTime, describe
        });
    }
    await addManyData(TABLEENUM.单位所获得科研成果表, addList);

    return successResult();
}


/**
 * 删除单位所获得科研成果
 * @param id 科研成果返回的id 
 * @returns 
 */
export async function delResearch({id}) {
    let dataInfo = await findOnce(TABLEENUM.单位所获得科研成果表, {id});
    if (!dataInfo || !dataInfo.id) throw new BizError(ERRORENUM.目标数据不存在);
    await deleteOneData(TABLEENUM.单位所获得科研成果表, {id});
    return successResult();
}


/**
 * 个人会员列表
 * @param param0 
 */
export async function individualMemberList({name, documentId, phone, individualMemberType, memberLevel, session, joinTime, auditTime, memberState, pageNumber}) {
    let selectParam:any = {};
    if (name) selectParam.name = name;
    if (documentId) selectParam.documentId = documentId;
    if (phone) selectParam.phone = phone;
    if (individualMemberType) selectParam.individualMemberType = individualMemberType;
    if (memberLevel) selectParam.memberLevel = memberLevel;
    if (session) {
        let thisYear = moment(session).format("YYYY");
        let thisMonth = moment(session).format("MM");
        selectParam.session = getEdition(thisYear, thisMonth);;
    }
    if (joinTime) selectParam.joinTime = joinTime;
    if (auditTime) selectParam.auditTime = auditTime;
    if (memberState) selectParam.memberState = memberState;
    selectParam.memberType = MEMBERTYPE.个人会员;
    selectParam.userRegisterState = USERREGISTERSTATE.通过;

    let selectConf = ["userId", "memberState", "unitName", "name", "sex", "nation", "loginId", "phone", "joinTime", "memberLevel", "sheng", "shi", "qu"];
    let dbList = await findToPage(TABLEENUM.用户表, selectParam, selectConf, pageNumber);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    dbList.forEach(info => {
        let item:any = extractData(info, selectConf);
        item.memberState = changeEnumValue(MEMBERSTATE, item.memberState);
        if (!item.memberLevel) item.memberLevel = "未选择";
        else item.memberLevel = changeEnumValue(MEMBERLEVEL, item.memberLevel);

        item.joinTime = moment(item.joinTime).format("YYYY-MM-DD");

        item.shi = getCityNameByCode(item.shi);
        item.sheng = getCityNameByCode(item.sheng);
        item.qu = getCityNameByCode(item.qu);
        dataList.push(item);
    });

    return {dataList, dataCount};
}


/**
 * 个人会员详情
 * @param id 会员列表返回的userId  
 * @returns {顶部数据, 基本信息, 专业信息, 其他附件}
 * 基本信息: "name"：真实姓名, "sex"：性别, "nation"：民族, "birth"：出身年月, "documentType"：证件类型, "documentId"：证件号码, "sheng"：省, "shi"：市, "qu"：区, "phone"：手机号码,"mail"：电子邮箱,"addres"：通信地址, "photoUrl"：蓝底证件照, "education"：学历, "uscc"：原社会信用代码, "unitName"：原发票抬头, "newUscc"：新社会信用代码, "newUnitName"：新发票抬头
 * 专业信息: "unitName"：工作单位, "duties"：职务, "workTitle"：职称, "profCategory"：专业类别, "studyResume"：学习简历, "workResume"：工作简历, "profAchievement"：科研成果, "otherEMP"：其他组织任职情况, "otherPROF"：其他情况
 * 其他附件: "cardUrl"：身份证正反面图片地址, "academicCERTUrl"：学历证明图片地址, "professionalCERTUrl"：工作证明图片地址, "auxiliaryMaterial"：其他辅助材料
 */

const TopConfig = ["session", "memberLevel"];
const IndividualBasicConfig = ["userId", "name", "sex", "nation", "birth", "documentType", "documentId", "sheng", "shi", "qu", "phone", "mail", "addres", "photoUrl", "education", "uscc", "unitName"];
const SpecialityConfig = ["unitName", "duties", "workTitle", "profCategory", "studyResume", "workResume", "profAchievement", "otherEMP", "otherPROF"];

export async function individualMemberDetails({id}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId:id});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    let topInfo:any = extractData(oldInfo, TopConfig);
    topInfo.memberLevel = changeEnumValue(MEMBERLEVEL, topInfo.memberLevel);

    let basicInfo:any = extractData(oldInfo, IndividualBasicConfig);
    // basicInfo.documentType = changeEnumValue(DOCUMENTTYPE, basicInfo.documentType);
    // basicInfo.nation = changeEnumValue(NATION, basicInfo.nation);
    /**获取订单表新社会信用代码和新发票抬头 */
    let orderInfo = await findOnceToSort(TABLEENUM.订单表, {userId:id}, {ct:-1});
    if (!orderInfo || !orderInfo.userId) {
        basicInfo.newUscc = "-";
        basicInfo.newUnitName = "-";
    } else {
        basicInfo.newUscc = orderInfo.newUscc;
        basicInfo.newUnitName = orderInfo.newUnitName;
    }

    let specialityInfo:any = extractData(oldInfo, SpecialityConfig);
    // basicInfo.profCategory = changeEnumValue(PROFCATEGORY, basicInfo.profCategory);
    
    let otherAttachment:any = extractData(oldInfo, ["cardUrl", "academicCERTUrl", "professionalCERTUrl", "auxiliaryMaterial"]);
    
    return {topInfo, basicInfo, specialityInfo, otherAttachment};
}


/**
 * 个人会员 修改
 * @param id 会员列表返回的userId 
 * @param form
 * @returns 
 */
 export async function individualMemberUpdate({form, id}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId:id});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    await updateOneData(TABLEENUM.用户表, {userId:id}, form);

    return successResult();
}

export async function systemBase({userId}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    let superAdmin = userInfo.loginId == "admin";
    let adminLv = [];
    userInfo.adminLv.forEach(item => {
        adminLv.push({
            key:changeEnumValue(ADMINLV, item),
            value:item
        });
    });
    let dataInfo = {
        isAdmin:userInfo.isAdmin == STATE.是,
        name:userInfo.isAdmin ? changeEnumValue(ADMINTYPE, userInfo.adminType ) : userInfo.name,
        userId:userInfo.userId,
        token:userInfo.token,
        adminLv,
        superAdmin,
        timeMs:moment(userInfo.lastLoginMs).format("YYYY-MM-DD HH:mm:SS"),
    }

    return {dataInfo}
}