/**
 * 审批逻辑 success
 */

import moment = require("moment");
import { INDIVIDUALMEMBERTYPE, MEMBERLEVEL, MEMBERTYPE, OPERATIONREHAVIOR, PAYMENTSTATUS, REGISTERFLOW, STATE, UNITMEMBERTYPE, USERREGISTERSTATE } from "../../config/enum";
import { ERRORENUM } from "../../config/errorEnum";
import { addManyData } from "../../data/add";
import { TABLEENUM } from "../../data/models/model";
import { find, findCount, findOnce, findOnceToSort, findToPage } from "../../data/select";
import { updateOneData } from "../../data/update";
import { generateSystemId, getMenberNum, successResult } from "../../tools/system";
import { BizError } from "../../util/bizError";
import { extractData } from "../../util/piecemeal";
import { changeEnumValue } from "../../util/verificationEnum";
import { createOrder } from "./order";
import { getCityNameByCode } from "../../config/cityConfig";

/**
 * 待审核列表 success
 * @param unitName 单位名称
 * @param joinTime 入会时间
 * @param memberType 会员类型 MEMBERTYPE
 * @param sheng 省
 * @param shi 市
 * @param qu 区
 * @param pageNumber 当前页 
 */
export async function pendingReviewList({unitName, joinTime, memberType, sheng, shi, qu, pageNumber}) {
    /**查询条件 */
    let selectParam:any = {
        "$or":[
            { memberType:MEMBERTYPE.个人会员, userRegisterState:USERREGISTERSTATE.待审核, registerFlow:REGISTERFLOW.完成第二步 },
            { memberType:MEMBERTYPE.个人会员, userRegisterState:USERREGISTERSTATE.驳回修改, registerFlow:REGISTERFLOW.完成第一步 },
            { memberType:MEMBERTYPE.个人会员, userRegisterState:USERREGISTERSTATE.重新提交, registerFlow:REGISTERFLOW.完成第二步 },
            { memberType:MEMBERTYPE.单位会员, userRegisterState:USERREGISTERSTATE.待审核, registerFlow:REGISTERFLOW.完成第三步 },
            { memberType:MEMBERTYPE.单位会员, userRegisterState:USERREGISTERSTATE.驳回修改, registerFlow:REGISTERFLOW.完成第一步 },
            { memberType:MEMBERTYPE.单位会员, userRegisterState:USERREGISTERSTATE.重新提交, registerFlow:REGISTERFLOW.完成第三步 },
        ],
        isAdmin:STATE.否
    }; //审批列表展示状态除通过外的数据
    if (unitName) selectParam.unitName = unitName;
    if (joinTime) selectParam.joinTime = joinTime;
    if (memberType) selectParam.memberType = memberType;
    if (sheng) selectParam.sheng = sheng;
    if (shi) selectParam.shi = shi;
    if (qu) selectParam.qu = qu;

    const SelectFiles = ["userId", "userRegisterState", "unitName", "loginId", "phone", "askForTime", "memberLevel", "sheng", "shi", "qu", "memberType"];
    let dbList = await findToPage(TABLEENUM.用户表, selectParam, SelectFiles, pageNumber);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    dbList.forEach(info => {
        let item:any = extractData(info, SelectFiles);
        /**处理枚举值 */
        if (!item.userRegisterState) item.userRegisterState = USERREGISTERSTATE.待审核;
        else item.userRegisterState = changeEnumValue(USERREGISTERSTATE, item.userRegisterState);
        item.memberLevel = changeEnumValue(MEMBERLEVEL, item.memberLevel);
        item.sheng = getCityNameByCode(item.sheng);
        item.shi = getCityNameByCode(item.shi);
        item.qu = getCityNameByCode(item.qu);
        item.memberType = changeEnumValue(MEMBERTYPE, info.memberType);

        dataList.push(item);
    });

    return {dataList, dataCount};
}


/**
 * 通过审核 success
 * @param id 待审核列表返回的用户id 
 * @returns 
 */
export async function adopt({id, session}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId:id, isAdmin:STATE.否}, ["userId", "registerFlow", "loginId","userRegisterState", "memberType", "unitName", "name", "phone", "sheng"]);
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    if (userInfo.memberType == MEMBERTYPE.个人会员 && userInfo.registerFlow != REGISTERFLOW.完成第二步) {
        throw new BizError(ERRORENUM.不具备审批条件, `${userInfo.userId}为个人会员，并没有完成第二步`);
    }
    if (userInfo.memberType == MEMBERTYPE.单位会员 && userInfo.registerFlow != REGISTERFLOW.完成第三步) {
        throw new BizError(ERRORENUM.不具备审批条件, `${userInfo.userId}为单位会员，并没有完成第三步`);
    }
    if (userInfo.userRegisterState == USERREGISTERSTATE.驳回修改) {
        throw new BizError(ERRORENUM.不具备审批条件, `${userInfo.userId}当前状态为驳回修改`);
    }
    if (userInfo.userRegisterState == USERREGISTERSTATE.通过) {
        throw new BizError(ERRORENUM.重复提交, `${userInfo.userId}当前状态为已通过`);
    }

    let thisDate = new Date();
    const NowMs = thisDate.valueOf();
    
    let userCount = await findCount(TABLEENUM.用户表, {isAdmin:0, memberType:userInfo.memberType,  userRegisterState:USERREGISTERSTATE.通过});
    let memberNum = getMenberNum(thisDate, userInfo.sheng, userCount, userInfo.memberType);

    let updateInfo:any = {
        auditTime:NowMs,
        joinTime:NowMs,
        userRegisterState:USERREGISTERSTATE.通过,
        session:session,
        paymentStatus:PAYMENTSTATUS.未支付,
        lifespanStartTime:NowMs,
        lifespanEndTime:NowMs + 1000,
        memberLevel:MEMBERLEVEL.普通会员,
        memberNum
    };
    let unitName = userInfo.unitName;
    let subType = userInfo.unitMemberType;
    if (userInfo.memberType == MEMBERTYPE.个人会员) {
        updateInfo.individualMemberType = INDIVIDUALMEMBERTYPE.普通会员;
        subType = userInfo.individualMemberType;
    }

    //todo 创建一条初始订单
    
    await createOrder(id, userInfo.loginId, userInfo.phone, unitName, NowMs + 1000, MEMBERLEVEL.普通会员, userInfo.memberType, subType, true);

    await updateOneData(TABLEENUM.用户表, {userId:id}, updateInfo);

    /**提交审批历史信息 */
    let addApprovalHistory = {
        id:generateSystemId(TABLEENUM.审批历史表, id),
        userId:id,
        operationTime:NowMs,
        operationBehavior:OPERATIONREHAVIOR.通过,
        remarks:"通过"
    };
    await addManyData(TABLEENUM.审批历史表, addApprovalHistory);

    return successResult();
}


/**
 * 驳回 success
 * @param userId 
 * @param remarks 驳回理由
 * @returns 
 */
export async function reject({id, remarks}) {
    let userId = id;
    let userInfo = await findOnce(TABLEENUM.用户表, {userId, isAdmin:STATE.否});
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    if (userInfo.memberType == MEMBERTYPE.个人会员 && userInfo.registerFlow != REGISTERFLOW.完成第二步) {
        throw new BizError(ERRORENUM.不具备审批条件, `${userInfo.userId}为个人会员，并没有完成第二步`);
    }
    if (userInfo.memberType == MEMBERTYPE.单位会员 && userInfo.registerFlow != REGISTERFLOW.完成第三步) {
        throw new BizError(ERRORENUM.不具备审批条件, `${userInfo.userId}为单位会员，并没有完成第三步`);
    }
    if (userInfo.userRegisterState == USERREGISTERSTATE.驳回修改) {
        throw new BizError(ERRORENUM.不具备审批条件, `${userInfo.userId}当前状态为驳回修改`);
    }
    if (userInfo.userRegisterState == USERREGISTERSTATE.通过) {
        throw new BizError(ERRORENUM.重复提交, `${userInfo.userId}当前状态为已通过`);
    }

    /**驳回修改，修改该会员注册流程回到第二步 */
    let updateParam = {
        userRegisterState:USERREGISTERSTATE.驳回修改, 
        registerFlow:REGISTERFLOW.完成第一步
    }
    await updateOneData(TABLEENUM.用户表, {userId}, updateParam);
    
    /**提交审批历史信息 */
    const NowMs = new Date().valueOf(); 
    let addApprovalHistory = {
        id:generateSystemId(TABLEENUM.审批历史表, userId),
        userId,
        operationTime:NowMs,
        operationBehavior:OPERATIONREHAVIOR.驳回修改,
        remarks
    };
    await addManyData(TABLEENUM.审批历史表, addApprovalHistory);

    return successResult();
}


/**
 * 审核历史 success
 * @param userId
 * @returns 
 */
export async function approvalHistoryList({userId}) {
    let dbList = await find(TABLEENUM.审批历史表, {userId});

    let dataList = [];
    dbList.forEach(info => {
        let item:any = extractData(info, ["operationTime", "operationBehavior", "remarks"]);
        item.operationBehavior = changeEnumValue(OPERATIONREHAVIOR, item.operationBehavior);
        dataList.push(item);
    });

    return {dataList};
}
