/**
 * 会员首页展板逻辑
 * 【二期内容：申请理事、申请常务理事、申请专家会员、会员积分、活动列表[立即参加]、消息中心暂时展示缴费提醒】
 */

import moment = require("moment");
import { BANXUELEIXING, CHANGEAPPLYTYPE, CLIENTMSGLABLE, CODPARTICIPANT, DOCUMENTTYPE, INDIVIDUALMEMBERTYPE, INFOCHANGEAPPLYTYPE, MEMBERLEVEL, MEMBERLEVELCHANGE, MEMBERSTATE, MEMBERTYPE, NATION, PAYMENTSTATUS, PROFCATEGORY, SEX, STATE, UNITMEMBERTYPE, USERREGISTERSTATE } from "../../config/enum";
import { ERRORENUM } from "../../config/errorEnum";
import { addManyData, addOneData } from "../../data/add";
import { deleteManyData } from "../../data/delete";
import { TABLEENUM } from "../../data/models/model";
import { find, findOnce, findOnceToSort, findToSort } from "../../data/select";
import { updateOneData } from "../../data/update";
import { generateSystemId, getEdition, successErrorResult, successResult } from "../../tools/system";
import { BizError } from "../../util/bizError";
import { extractData } from "../../util/piecemeal";
import { changeEnumValue, eccEnumValue } from "../../util/verificationEnum";
import { eccFormParam } from "../../util/verificationParam";
import { getCityNameByCode } from "../../config/cityConfig";


/**
 * 会员信息展示
 * @param userId 登录的用户id
 * @returns 个人会员：{头部数据， 基本信息， 专业信息， 其他附件}
 * @returns 单位会员：{头部数据， 基本信息， 单位信息， 单位所获得科研成果}
 */
export async function getMemberData({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    let topSelectFiles = ["loginId", "memberType", "userRegisterState", "integral", "joinTime", "lifespanStartTime", "lifespanEndTime", "memberLevel", "isFirstPay", "paymentStatus", "memberState"];
    let baseSelectFiles = [];
    let majorSelectFiles = [];
    let otherAttachment:any= {};
    /**判断会员类型，个人会员与单位会员返回不同字段 */
    if (oldInfo.memberType == MEMBERTYPE.单位会员) {
        baseSelectFiles = ["userId", "unitName", "uscc", "legalPerson", "legalPersonPhone", "legalPersonMail", "mail", "contactPerson", 
        "contactPersonDuties", "phone", "uusinessLicenseUrl", "addres", "sheng", "shi", "qu", "kaishezhuanYe", "danWeiJianJie"];
        majorSelectFiles = ["unitMemberType", "yuanXiaoBanXueLeiXing", "yuanXiaoZhuGuanBuMen", "yuanXiaoFuZeRen", "yuanXiaoFuZeRenZhiWu", 
        "yuanXiaoFuZeRenDianHua", "yuanXiaoBanGongFuZeRen", "yuanXiaoBanGongFuZeRenDianHua", "ZhuYaoFuZeRenYouXiang", "yuanXiaoKeYanFuZeRen", 
        "yuanXiaoKeYanFuZeRenDianHua", "jiaoXueFuZeRenXinMing", "jiaoXueFuZeRenDianHua"];
        /**获取单位科研成果数据 */
        let oldKeYanList = await find(TABLEENUM.单位所获得科研成果表, {userId}, ["id", "startTime", "endTime", "describe"]);
        if (!otherAttachment.dataList) otherAttachment.dataList = [];
        oldKeYanList.forEach(info => {
            otherAttachment.dataList.push({
                id:info.id,
                startTime: info.startTime,
                endTime: info.endTime,
                describe:info.describe
            });
        })

    } else {
        baseSelectFiles = ["userId", "name", "sex", "nation", "birth", "documentType", "documentId", "sheng", "shi", "qu", "phone", "mail", 
        "addres", "photoUrl", "education", "uscc", "unitName"];
        majorSelectFiles = ["unitName", "duties", "workTitle", "profCategory", "studyResume", "workResume", "profAchievement", "otherEMP", "otherPROF"]; 
        let otherAttachmentFiles = ["cardUrlJust", "cardUrlBack", "academicCERTUrl", "professionalCERTUrl", "auxiliaryMaterial"];
        otherAttachment = extractData(oldInfo, otherAttachmentFiles);
    }

    /**返回指定配置字段 */
    let topInfo:any = extractData(oldInfo, topSelectFiles);
    // 变更申请职位不为空并且状态不等于通过的时候，代表提交了变更理事申请，并且还未通过审批，需要先使用变更前职位
    // if(!oldInfo.beforeChange && oldInfo.changeApplyType != CHANGEAPPLYTYPE.通过) topInfo.memberLevel = oldInfo.beforeChange;
    let baseInfo:any = extractData(oldInfo, baseSelectFiles);
    let majorInfo:any = extractData(oldInfo, majorSelectFiles);
    
    /**处理枚举值 */
    topInfo.integral = 0;
    if(topInfo.joinTime) topInfo.joinTime = moment(topInfo.joinTime).format("YYYY-MM-DD");
    if(topInfo.lifespanEndTime) {
        if(topInfo.memberState == MEMBERSTATE.提醒) topInfo.lifespanEndTime = "欠费中"; 
        else topInfo.lifespanEndTime = moment(topInfo.lifespanEndTime).format("YYYY-MM-DD");
    }

    if(topInfo.userRegisterState == USERREGISTERSTATE.通过) topInfo.userRegisterState = "已入会";
    else topInfo.userRegisterState = "未入会";

    // if(topInfo.memberLevel) topInfo.memberLevel = changeEnumValue(MEMBERLEVEL, topInfo.memberLevel);
    // if(baseInfo.sex) baseInfo.sex = changeEnumValue(SEX, baseInfo.sex);
    // if(baseInfo.documentType) baseInfo.documentType = changeEnumValue(DOCUMENTTYPE, baseInfo.documentType);
    // if(baseInfo.nation) baseInfo.nation = changeEnumValue(NATION, baseInfo.nation);
    // if(majorInfo.profCategory) majorInfo.profCategory = changeEnumValue(PROFCATEGORY, majorInfo.profCategory);
    // if(majorInfo.unitMemberType) majorInfo.unitMemberType = changeEnumValue(UNITMEMBERTYPE, majorInfo.unitMemberType);
    // if(majorInfo.yuanXiaoBanXueLeiXing) majorInfo.yuanXiaoBanXueLeiXing = changeEnumValue(BANXUELEIXING, majorInfo.yuanXiaoBanXueLeiXing);
    
    if(baseInfo.sheng) baseInfo.shengName = getCityNameByCode(baseInfo.sheng);
    if(baseInfo.shi) baseInfo.shiName = getCityNameByCode(baseInfo.shi);
    if(baseInfo.qu) baseInfo.quName = getCityNameByCode(baseInfo.qu);

    if (baseInfo.sheng  == "-")  baseInfo.sheng = ""; 
    if (baseInfo.shi  == "-") baseInfo.shi = ""; 
    if (baseInfo.qu == "-") baseInfo.qu = ""; 

    return {topInfo, baseInfo, majorInfo, otherAttachment};
}


/**
 * 单位会员 修改
 * @param form 修改信息表单
 * @param keyanForm 修改科研信息表单
 * @param userId 登录的用户id
 * @returns 
 */
export async function unitMemberUpdate({form, keyanForm, userId}) {
    const keyanFormItemConfig = {
        startTime: 'Number',
        endTime: 'Number',
        describe: 'String'
    };
    keyanForm.forEach(item => {
        eccFormParam("单位会员修改", keyanFormItemConfig, item );
    });
    
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    /**校验枚举 */
    const EccFuncName = '单位会员修改';
    eccEnumValue(EccFuncName, '单位信息类型', UNITMEMBERTYPE, form.unitMemberType);

    // /**确保邮箱的唯一性 */
    // let distinctMailInfo = await findOnce(TABLEENUM.用户表, {mail:form.unitMail}, ["userId"]);
    // if (distinctMailInfo && distinctMailInfo.userId && distinctMailInfo.userId != oldInfo.userId) throw new BizError(ERRORENUM.邮箱重复, oldInfo.loginId, oldInfo.mail );
    // form.mail = form.unitMail;
    /**修改会员用户信息 */
    await updateOneData(TABLEENUM.用户表, {userId}, form);

    /**删除对应userId科研成果 */
    let oldKeYanList= await find(TABLEENUM.单位所获得科研成果表, {userId});
    if (oldKeYanList && oldKeYanList.length) {
        await deleteManyData(TABLEENUM.单位所获得科研成果表, {userId});
    }
    

    /**添加新的科研信息 */
    let addList = [];
    for (let i = 0; i < keyanForm.length; i++) {
        let { startTime, endTime, describe } = keyanForm[i];
        addList.push({
            id:generateSystemId(TABLEENUM.单位所获得科研成果表, userId),
            userId,
            startTime, endTime, describe
        });
    }
    await addManyData(TABLEENUM.单位所获得科研成果表, addList);

    return successResult();
}


/**
 * 个人会员 修改
 * @param userId 登录的用户id
 * @param form 修改信息表单
 * @returns 
 */
 export async function individualMemberUpdate({form, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    await updateOneData(TABLEENUM.用户表, {userId}, form);

    return successResult();
}


/**
 * 消息中心列表
 * 缴费消息提醒
 * @param userId 登录的用户id
 */
export async function getXiaoXiZhongXin({userId}) {
    const SelectFiles = ["userId", "lifespanEndTime", "isGracePeriod", "gracePeriodEndTime", "lifespanStartTime"];
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId, isAdmin:STATE.否}, SelectFiles);

    let dataInfo:any = extractData(oldInfo, ["lifespanStartTime", "lifespanEndTime"]);

    const NowMs = new Date().valueOf();

    /**判断是否宽限期 */
    let str = '';
    if (oldInfo.isGracePeriod == STATE.是 && oldInfo.gracePeriodEndTime > NowMs) str = "宽限期中";
    else {
        if (NowMs > oldInfo.lifespanEndTime) {
            str = '过期';
            if (NowMs > oldInfo.lifespanEndTime + (90*24*3600*1000)){
                str += '1年外';
            } else if ( NowMs > oldInfo.lifespanEndTime + (90*24*3600*1000) ) {
                str += '1年内';
            } else if (NowMs > oldInfo.lifespanEndTime + (30*24*3600*1000)) {
                str += '90天内';
            } else {
                str += '30天内'; 
            }
        } else {
            if (NowMs > oldInfo.lifespanEndTime - (90*24*3600*1000) ) {
                str = "不足90天";
            } else {
                str = "大于90天";
            }
        }
    }
    dataInfo.membershipFees = str;

    return {dataInfo};
}


/**
 * 地图省数量统计
 * @param userId 登录的用户id 
 * @returns 
 */
export async function getSheng ({userId}) {
    let myOldInfo = await findOnce(TABLEENUM.用户表, {userId}, ["sheng"]);
    if (!myOldInfo || !myOldInfo.sheng) throw new BizError(ERRORENUM.目标数据不存在);
    let myDataInfo = extractData(myOldInfo, ["sheng"]);

    let allOldList = await find(TABLEENUM.用户表, {}, ["sheng"]);
    if (!allOldList || !allOldList.length) throw new BizError(ERRORENUM.目标数据不存在);
    
    let allDataList = {};
    allOldList.forEach( info => {
        //初始数据包含没有填省份的，过滤省份为undefined的数据，统计省份个数
        if (info.sheng) {
            if (!allDataList[info.sheng]) allDataList[info.sheng] = 0;
            allDataList[info.sheng] += 1;
        }
    })

    return {myDataInfo, allDataList};
}


/**
 * 活动列表
 */
 export async function getHuoDongTongZhi () {
    let oldList = await find(TABLEENUM.活动通知, {}, ["id", "name"]);
    if (!oldList || oldList.length<1) throw new BizError(ERRORENUM.目标数据不存在);
    
    let dataList = [];
    oldList.forEach( info => {
        let item = extractData(info, ["id", "name"]);
        dataList.push(item);
    })

    return {dataList};
}


/**
 * 根据会员类别返回对应可下载的证书类型
 * @param param0 
 * @returns 
 */
 export async function getCertificate({userId}) {
    const Files = ["userId", "memberType", "userRegisterState", "paymentStatus", "individualMemberType", "unitMemberType", 
    "sheng", "documentId", "memberNum", "lifespanStartTime", "lifespanEndTime", "shi","qu","addres", "unitName", "session", "name"];
    let userInfo = await findOnce(TABLEENUM.用户表, {userId}, Files);
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    let certificate = false;//是否可以下载
    let cefaInfo:any = {};
    let msg = '会员注册尚未审核通过，无法下载证书';
    //判断是否审批通过
    if (userInfo.userRegisterState == USERREGISTERSTATE.通过) {
        let addresStr = "";
        
        if (getCityNameByCode(userInfo.sheng)) {
            addresStr += `${getCityNameByCode(userInfo.sheng)}`;
            if (getCityNameByCode(userInfo.shi)) {
                if (getCityNameByCode(userInfo.sheng) != getCityNameByCode(userInfo.shi)) {
                    addresStr += `${getCityNameByCode(userInfo.shi)}`;
                }
                if (getCityNameByCode(userInfo.qu)) {
                    addresStr += `${getCityNameByCode(userInfo.qu)}`;
                }
            }
        }
        // if (getCityNameByCode(userInfo.sheng) == getCityNameByCode(userInfo.shi)) {
        //     addresStr = `${getCityNameByCode(userInfo.shi)}${getCityNameByCode(userInfo.qu)}`;
        // } else {
        //     addresStr = `${getCityNameByCode(userInfo.sheng)}${getCityNameByCode(userInfo.shi)}${getCityNameByCode(userInfo.qu)}`;
        // }

        if(userInfo.memberType == MEMBERTYPE.个人会员) {
            //个人会员已支付可下载
            if (userInfo.individualMemberType == INDIVIDUALMEMBERTYPE.普通会员 && userInfo.paymentStatus != PAYMENTSTATUS.未支付) certificate = true;
            //个人会员专家无需支付可下载
            if (userInfo.individualMemberType == INDIVIDUALMEMBERTYPE.专家会员) certificate = true;
            if (certificate) {
                cefaInfo = {
                    name:userInfo.name,
                    area:addresStr,
                    documentId:userInfo.documentId,
                    memberNum:userInfo.memberNum,
                    session:getEdition(Math.floor(userInfo.session/100), userInfo.session%100 ),
                    lifespanTime:`${moment(userInfo.lifespanStartTime).format("YYYY年MM月DD日")}至${moment(userInfo.lifespanEndTime).format("YYYY年MM月DD日")}`
                }
                msg = ``;
            } else {
                msg = `会员注册尚未缴费，无法下载证书`;
            }
        } else if(userInfo.memberType == MEMBERTYPE.单位会员) {
            //院校无需支付可下载
            if (userInfo.unitMemberType == UNITMEMBERTYPE.院校) certificate = true;
            //机构和其他已支付可下载
            else if (userInfo.paymentStatus != PAYMENTSTATUS.未支付) {
                certificate = true;
            }
            if (certificate) {
                // if (getCityNameByCode(userInfo.sheng) == getCityNameByCode(userInfo.shi)) {
                //     addresStr = `${getCityNameByCode(userInfo.shi)}${getCityNameByCode(userInfo.qu)}${userInfo.addres}`;
                // } else {
                //     addresStr = `${getCityNameByCode(userInfo.sheng)}${getCityNameByCode(userInfo.shi)}${getCityNameByCode(userInfo.qu)}${userInfo.addres}`;
                // }
                cefaInfo = {
                    unitName:userInfo.unitName,
                    addres:addresStr,
                    memberNum:userInfo.memberNum,
                    session:getEdition(Math.floor(userInfo.session/100), userInfo.session%100 ),
                    lifespanTime:`${moment(userInfo.lifespanStartTime).format("YYYY年MM月DD日")}至${moment(userInfo.lifespanEndTime).format("YYYY年MM月DD日")}`
                }
                if (userInfo.unitMemberType == UNITMEMBERTYPE.院校) {
                    let diffYearNum = moment().diff(moment(userInfo.lifespanStartTime), 'years');
                    let startTimeStr = moment(userInfo.lifespanStartTime).add(diffYearNum, 'years').format("YYYY年MM月DD日");
                    let endTimeStr = moment(userInfo.lifespanStartTime).add(diffYearNum+1, 'years').format("YYYY年MM月DD日");
                    cefaInfo.lifespanTime = `${startTimeStr}至${endTimeStr}`;
                }
                msg = ``;
            } else {
                msg = `会员注册尚未缴费，无法下载证书`;
            }
        }
    }
    
    return {certificate, cefaInfo, msg};
}


/**
 * 理事、常务理事证明函下载
 * @param param0 
 * @returns 
 */
export async function getCertificateType({userId}) {
    const Files = ["userId", "memberLevel", "changeApplyType", "applyApplicationTime", "unitName", "name", "beforeChange", "memberType"];
    let userInfo = await findOnce(TABLEENUM.用户表, {userId}, Files);
    if (!userInfo || !userInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    let certificate = false;//是否可以下载
    let cefaInfo:any = {};
    let msg = '会员理事审核未通过，无法下载证明函';
    //判断是否审批通过
    if (userInfo.beforeChange) {
        if (userInfo.changeApplyType == CHANGEAPPLYTYPE.通过) {
            if(userInfo.memberLevel == MEMBERLEVEL.理事会员 || userInfo.memberLevel == MEMBERLEVEL.常务理事会员) certificate = true
            if (certificate) {
                let name = "";
                if (userInfo.memberType == MEMBERTYPE.个人会员) name = userInfo.name;
                else name = userInfo.unitName;
                cefaInfo = {
                    memberLevel:changeEnumValue(MEMBERLEVEL, userInfo.memberLevel),
                    name,
                    applyApplicationTime: moment().format("YYYY年MM月DD日")
                }
                // if (userInfo.applyApplicationTime) cefaInfo.applyApplicationTime = moment(userInfo.applyApplicationTime).format("YYYY年MM月DD日")
                msg = ``;
            }
        } else if (userInfo.changeApplyType == CHANGEAPPLYTYPE.注册信息待补充) {
            msg = `会员注册信息待补充，无法下载证明函`;
        } else if (userInfo.changeApplyType == CHANGEAPPLYTYPE.待审批) {
            msg = `会员理事审核中，无法下载证明函`;
        }
    } else {
        if(userInfo.memberLevel == MEMBERLEVEL.理事会员 || userInfo.memberLevel == MEMBERLEVEL.常务理事会员) certificate = true
        if (certificate) {
            let name = "";
            if (userInfo.memberType == MEMBERTYPE.个人会员) name = userInfo.name;
            else name = userInfo.unitName;
            cefaInfo = {
                memberLevel:changeEnumValue(MEMBERLEVEL, userInfo.memberLevel),
                name,
                applyApplicationTime: moment().format("YYYY年MM月DD日")
            }
            // if (userInfo.applyApplicationTime) cefaInfo.applyApplicationTime = moment(userInfo.applyApplicationTime).format("YYYY年MM月DD日")
            msg = ``;
        }
    }

    return {certificate, cefaInfo, msg};
}


/**
 * 获取可下载缴费通知年份列表
 * 当前年超过3月1日则返回2023年至当前年份列表，否则不包括当前年份
 * @returns 
 */
export async function getYearList() {
    const currentDate = new Date();
    const currentYear = currentDate.getFullYear();
    const currentMonth = currentDate.getMonth() + 1; // JavaScript 月份从 0 开始，需加 1
    const currentDay = currentDate.getDate();

    let endYear;
    if (currentMonth > 3 || (currentMonth === 3 && currentDay >= 1)) {
        endYear = currentYear;
    } else {
        endYear = currentYear - 1;
    }
  
    let existingYears = []; //年份列表
    // 生成年份数组
    if (endYear >= 2023) {
        for (let i = 2023; i <= endYear; i++) {
            existingYears.push(i);
        }
    }
  
    return existingYears;
}


/**
 * 能否申请变更理事
 * @param userId 登录的userid
 * @returns 
 */
export async function getApply({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    const NowMs = new Date().valueOf();
    let isApply = true;

    let time = moment(1747895510765).format("YYYY-MM-dd")

    if (oldInfo.memberType == MEMBERTYPE.个人会员) {
        if(oldInfo.lifespanEndTime) {
            // 会员有效期结束时间小于当前时间的欠费会员
            if(oldInfo.memberState == MEMBERSTATE.提醒 || oldInfo.lifespanEndTime < NowMs) isApply = false; 
        }
    }
    
    return {isApply};
}


/**
 * 职务申请是否审批中
 * @param param0 
 * @returns 
 */
export async function getChangeApplyType({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    let userInfo = await findOnce(TABLEENUM.用户表, {userId}, ["changeApplyType"]);

    let changeApplyType = false;
    if (userInfo.changeApplyType == CHANGEAPPLYTYPE.待审批) changeApplyType = false;
    else changeApplyType = true;

    return {changeApplyType};
}


/**
 * 变更理事申请，职位未发生变更不需要走当前流程
 * @param memberLevel 变更理事 
 * @param userId 登录的userid
 * @returns 
 */
export async function memberlevelUpdate({memberLevel, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    if (oldInfo.memberLevel == memberLevel) throw new BizError(ERRORENUM.当前变更申请未发生变化);
    if (memberLevel > oldInfo.memberLevel +1) throw new BizError(`当前职位为：${oldInfo.memberLevel}，申请${memberLevel}${ERRORENUM.超出可申请范围}`);

    /**添加至变更职位 */
    eccEnumValue("变更申请", '变更申请职位' , MEMBERLEVELCHANGE, memberLevel); //校验枚举
    
    let param = {
        applyMemberLevel:memberLevel, 
        beforeChange:oldInfo.memberLevel,
    }
    await updateOneData(TABLEENUM.用户表, {userId}, param);

    return successResult();
}


/**
 * 是否需要补充用户信息
 * 判断注册资料是否完整，不完整返回false(需要补充用户信息)
 * @param userId 登录的userid
 * @param isChangeLevel 是否职务变更
 */
export async function isNeedSupplement({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});

    const GrRequiredConf = {
        "基本信息": ["name", "sex", "documentId", "documentType", "birth", "nation", "sheng", "shi", "qu", "addres", "education", "photoUrl"],
        "专业信息": ["duties", "workTitle", "studyResume", "workResume", "profAchievement", "otherEMP", "otherPROF", "profCategory"],
        "其他附件": ["cardUrlJust", "cardUrlBack", "academicCERTUrl"]
    };

    /**职务变更 */
    // const ZyRequiredConf = {
    //     "资料变更填写字段":["unitName", "uscc", "legalPerson", "legalPersonPhone", "legalPersonMail", "unitMemberType", "yuanXiaoBanXueLeiXing", "uusinessLicenseUrl", 
    //     "applicationForm"]
    // }
    // const DwRequiredConf = {
    //     "基本信息": ["sheng", "shi", "qu", "addres", "contactPerson", "contactPersonDuties", "phone"],
    //     "单位信息": ["yuanXiaoZhuGuanBuMen", "yuanXiaoFuZeRen", "yuanXiaoFuZeRenZhiWu", 
    //     "yuanXiaoFuZeRenDianHua", "yuanXiaoBanGongFuZeRen", "yuanXiaoBanGongFuZeRenDianHua", "ZhuYaoFuZeRenYouXiang", "yuanXiaoKeYanFuZeRen", 
    //     "yuanXiaoKeYanFuZeRenDianHua", "jiaoXueFuZeRenXinMing", "jiaoXueFuZeRenDianHua"]
    // };

    /**资料变更 */
    // const DwRequiredConf = {
    //     "基本信息": ["unitName", "sheng", "shi", "qu", "addres", "uscc", "legalPerson", "legalPersonPhone", "legalPersonMail", "contactPerson", 
    //     "contactPersonDuties", "phone", "uusinessLicenseUrl", "applicationForm"],
    //     "单位信息": ["unitMemberType", "yuanXiaoBanXueLeiXing", "yuanXiaoZhuGuanBuMen", "yuanXiaoFuZeRen", "yuanXiaoFuZeRenZhiWu", 
    //     "yuanXiaoFuZeRenDianHua", "yuanXiaoBanGongFuZeRen", "yuanXiaoBanGongFuZeRenDianHua", "ZhuYaoFuZeRenYouXiang", "yuanXiaoKeYanFuZeRen", 
    //     "yuanXiaoKeYanFuZeRenDianHua", "jiaoXueFuZeRenXinMing", "jiaoXueFuZeRenDianHua"]
    // };
        // "基本信息": ["sheng", "shi", "qu", "addres", "contactPerson", "contactPersonDuties", "phone"],// "applicationForm"
    const DwRequiredConf = { //, "contactPerson", "contactPersonDuties", "applicationForm"
        "基本信息": ["unitName", "sheng", "shi", "qu", "addres", "uscc", "legalPerson", "legalPersonPhone", "phone", "mail", "uusinessLicenseUrl"],
        "单位信息": ["yuanXiaoZhuGuanBuMen", "yuanXiaoFuZeRen", "yuanXiaoFuZeRenZhiWu", 
        "yuanXiaoFuZeRenDianHua", "yuanXiaoBanGongFuZeRen", "yuanXiaoBanGongFuZeRenDianHua", "ZhuYaoFuZeRenYouXiang", "yuanXiaoKeYanFuZeRen", 
        "yuanXiaoKeYanFuZeRenDianHua", "jiaoXueFuZeRenXinMing", "jiaoXueFuZeRenDianHua"]
    };

    const Dwshzycj = {
        "单位所获主要成绩及科研成果": ["startTime", "endTime", "describe"]
    };
    // const Gzsqb = {
    //     "盖章申请表": ["applicationForm"]
    // };

    let dwshzycjInfo = await findOnce(TABLEENUM.单位所获得科研成果表, {userId});

    let infoCompleteData = {};
    let isZwbgtxzd = {}; //职务变更判断资料变更重要字段是否填写完整，true：资料变更填写字段完整。false：资料变更填写字段不全。
    let changeApplyType = CHANGEAPPLYTYPE.注册信息补充完成;
    if (oldInfo.memberType == MEMBERTYPE.个人会员) {
        for (let key in GrRequiredConf) {
            for (let i = 0; i < GrRequiredConf[key].length; i++) {
                if(!oldInfo[GrRequiredConf[key][i]]) {
                    infoCompleteData[key] = false;
                    changeApplyType = CHANGEAPPLYTYPE.注册信息待补充;
                    break;
                } else {
                    infoCompleteData[key] = true;
                }
            }
        }
        
    } else {
        for (let key in DwRequiredConf) {
            for (let i = 0; i < DwRequiredConf[key].length; i++) {
                if(!oldInfo[DwRequiredConf[key][i]]) {
                    infoCompleteData[key] = false;
                    changeApplyType = CHANGEAPPLYTYPE.注册信息待补充;
                    break;
                } else {
                    infoCompleteData[key] = true;
                }
            }
        }
        
        for (let i = 0; i < Dwshzycj.单位所获主要成绩及科研成果.length; i++) {
            if (!dwshzycjInfo) {
                infoCompleteData["单位所获主要成绩及科研成果"] = false;
                break;
            } else {
                if(!dwshzycjInfo[Dwshzycj.单位所获主要成绩及科研成果[i]]) {
                    infoCompleteData["单位所获主要成绩及科研成果"] = false;
                    changeApplyType = CHANGEAPPLYTYPE.注册信息待补充;
                    break;
                } else {
                    infoCompleteData["单位所获主要成绩及科研成果"] = true;
                }
            }
        }

        // for (let i = 0; i < Gzsqb["盖章申请表"].length; i++) {
        //     if(!oldInfo[Gzsqb["盖章申请表"][i]]) {
        //         infoCompleteData["盖章申请表"] = false;
        //         changeApplyType = CHANGEAPPLYTYPE.注册信息待补充;
        //         break;
        //     } else {
        //         infoCompleteData["盖章申请表"] = true;
        //     }
        // }
    }

    let infoComplete = [];
    for (let key in infoCompleteData) {
        infoComplete.push({key, value:infoCompleteData[key]})
    }

    return infoComplete;
}


/**
 * 职务变更-资料填写是否完整
 * @param param0 
 * @returns 
 */
export async function isZwbgtxzd({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});

    /**职务变更 */
    const ZyRequiredConf = {
        "资料变更填写字段":["unitName", "uscc", "legalPerson", "legalPersonPhone", "legalPersonMail", "unitMemberType", "yuanXiaoBanXueLeiXing", "uusinessLicenseUrl", 
        "applicationForm"]
    }

    let isZwbgtxzd = false; //职务变更判断资料变更重要字段是否填写完整，true：资料变更填写字段完整。false：资料变更填写字段不全。
    let changeApplyType = CHANGEAPPLYTYPE.注册信息补充完成;
    for (let key in ZyRequiredConf) {
        for (let i = 0; i < ZyRequiredConf[key].length; i++) {
            if(!oldInfo[ZyRequiredConf[key][i]]) {
                isZwbgtxzd = false;
                changeApplyType = CHANGEAPPLYTYPE.注册信息待补充
                break;
            } else {
                isZwbgtxzd = true;
            }
        }
    }

    return {isZwbgtxzd};
}


/**
 * 个人会员用户注册信息补充
 * @param userId 登录的userid
 * @param form 补充的信息表单
 * @returns 
 */
export async function infoSupplement({form, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    /**校验枚举 */
    const EccFuncName = '个人会员用户注册信息补充';
    eccEnumValue(EccFuncName, '性别', SEX, form.sex);
    eccEnumValue(EccFuncName, '证件类型', DOCUMENTTYPE, form.documentType);
    eccEnumValue(EccFuncName, '民族', NATION, form.nation);
    eccEnumValue(EccFuncName, '专业类别', PROFCATEGORY, form.profCategory);

    await updateOneData(TABLEENUM.用户表, {userId}, form);

    return successResult();
}


/**
 * 单位会员用户注册信息补充
 * @param userId 登录的用户id
 * @param form 补充的信息表单
 * @returns 
 */
export async function unitInfoSupplement({form, keyanForm, userId}) {
    const keyanFormItemConfig = {
        startTime: 'Number',
        endTime: 'Number',
        describe: 'String'
    };
    keyanForm.forEach(item => {
        eccFormParam("单位会员用户注册信息补充", keyanFormItemConfig, item );
    });
    
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    /**校验枚举 */
    const EccFuncName = '单位会员用户注册信息补充';
    eccEnumValue(EccFuncName, '单位信息类型', UNITMEMBERTYPE, form.unitMemberType);

    /**确保邮箱的唯一性 */
    // let distinctMailInfo = await findOnce(TABLEENUM.用户表, {mail:form.unitMail}, ["userId"]);
    // if (distinctMailInfo && distinctMailInfo.userId && distinctMailInfo.userId != oldInfo.userId) throw new BizError(ERRORENUM.邮箱重复, oldInfo.loginId, oldInfo.mail );
    // form.mail = form.unitMail;
    /**修改会员用户信息 */
    await updateOneData(TABLEENUM.用户表, {userId}, form);

    /**删除对应userId科研成果 */
    let oldKeYanList= await find(TABLEENUM.单位所获得科研成果表, {userId});
    if (oldKeYanList && oldKeYanList.length) {
        await deleteManyData(TABLEENUM.单位所获得科研成果表, {userId});
    }
    
    /**添加新的科研信息 */
    let addList = [];
    for (let i = 0; i < keyanForm.length; i++) {
        let { startTime, endTime, describe } = keyanForm[i];
        addList.push({
            id:generateSystemId(TABLEENUM.单位所获得科研成果表, userId),
            userId,
            startTime, endTime, describe
        });
    }
    await addManyData(TABLEENUM.单位所获得科研成果表, addList);

    return successResult();
}


/**
 * 变更理事申请理由
 * @param applyStr 变更申请理由
 * @param userId 登录的userid
 * @returns 
 */
export async function applyReason({applyStr, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    const Now = new Date().valueOf();
    let applyId = generateSystemId(TABLEENUM.理事变更审批历史表, userId);
    let param = {
        applyStr,
        changeApplyType:CHANGEAPPLYTYPE.待审批,
        applyApplicationTime:Now,
        applyId,
    }
    await updateOneData(TABLEENUM.用户表, {userId}, param);
    
    /**添加审批历史 */
    let applyInfo:any = {
        applyId,
        userId,
        applyMemberLevel:oldInfo.applyMemberLevel,
        beforeChange:oldInfo.beforeChange,
        applyStr,
        changeApplyType:CHANGEAPPLYTYPE.待审批,
        applyApplicationTime:Now,
    };
    await addOneData(TABLEENUM.理事变更审批历史表, applyInfo);

    return successResult();
}


/**
 * 变更理事申请理由-单位会员 2025-04-22 单位会员新增理事申请字段需要与个人会员区分方法
 * @param applyStr 变更申请理由
 * @param userId 登录的userid
 * @returns 
 */
 export async function applyReasonMember({applyName, applySex, applyDuties, applyPhone, applyStr, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    if ( oldInfo.memberState == MEMBERSTATE.提醒 ) throw new BizError(ERRORENUM.请先缴费);

    const Now = new Date().valueOf();
    let applyId = generateSystemId(TABLEENUM.理事变更审批历史表, userId);
    let param = {
        applyStr,
        changeApplyType:CHANGEAPPLYTYPE.待审批,
        applyApplicationTime:Now,
        applyId,

    }
    await updateOneData(TABLEENUM.用户表, {userId}, param);
    
    /**添加审批历史 */
    let applyInfo:any = {
        applyId,
        userId,
        applyMemberLevel:oldInfo.applyMemberLevel,
        beforeChange:oldInfo.beforeChange,
        changeApplyType:CHANGEAPPLYTYPE.待审批,
        applyApplicationTime:Now,
        applyName,
        applySex,
        applyDuties,
        applyPhone,
        applyStr,
    };
    await addOneData(TABLEENUM.理事变更审批历史表, applyInfo);

    return successResult();
}



/**
 * 资料变更
 * @param param0 
 * @returns 
 */
export async function infomationChangeUpdate({userId, unitName, uscc, legalPerson, unitMemberType, uusinessLicenseUrl, applicationForm, desc, yuanXiaoBanXueLeiXing, legalPersonMail, legalPersonPhone}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId}, ["memberType"]);
    if (userInfo.memberType != MEMBERTYPE.单位会员) throw new BizError(ERRORENUM.非单位会员不可操作);

    let applyToSort = await findOnceToSort(TABLEENUM.资料变更审批历史表, {userId}, {createTimeMs:-1});
    let infoChangeOptionType = INFOCHANGEAPPLYTYPE.待审批;
    if (applyToSort) {
        //新增了状态：驳回提交，当上次审批状态为驳回修改时，当前提交状态为驳回提交
        if (applyToSort.infoChangeOptionType == INFOCHANGEAPPLYTYPE.驳回修改) infoChangeOptionType = INFOCHANGEAPPLYTYPE.驳回提交;
    }

    //添加日志
    /**添加审批历史 */
    let applyInfo:any = {
        applyId:generateSystemId(TABLEENUM.资料变更审批历史表, userId),
        infoChangeId:generateSystemId(TABLEENUM.资料变更审批历史表, userId),
        userId,
        unitName,
        uscc, 
        legalPerson, 
        unitMemberType, 
        uusinessLicenseUrl, 
        applicationForm,
        yuanXiaoBanXueLeiXing,
        desc,
        legalPersonMail, 
        legalPersonPhone,
        infoChangeOptionType,
        createTimeMs:new Date().valueOf(),
        isDraft:0
    };

    await updateOneData(TABLEENUM.用户表, {userId}, {infoChangeId:applyInfo.infoChangeId, infoChangeMs:new Date().valueOf(), infoChangeOptionType});
    await addOneData(TABLEENUM.资料变更审批历史表, applyInfo);

    return successResult();
}


/**
 * 保存 资料变更
 * @param param0 
 * @returns 
 */
export async function infomationSave({userId, unitName, uscc, legalPerson, unitMemberType, uusinessLicenseUrl, applicationForm, desc, yuanXiaoBanXueLeiXing, legalPersonMail, legalPersonPhone}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId}, ["memberType"]);
    if (userInfo.memberType != MEMBERTYPE.单位会员) throw new BizError(ERRORENUM.非单位会员不可操作);


    let applyToSort = await findOnceToSort(TABLEENUM.资料变更审批历史表, {userId}, {createTimeMs:-1});

    let infoChangeOptionType = INFOCHANGEAPPLYTYPE.待审批;
    if (applyToSort) {
        //新增了状态：驳回提交，当上次审批状态为驳回修改时，当前提交状态为驳回提交
        if (applyToSort.infoChangeOptionType == INFOCHANGEAPPLYTYPE.驳回修改) infoChangeOptionType = INFOCHANGEAPPLYTYPE.驳回提交;
    }

    if (applyToSort.infoChangeOptionType == INFOCHANGEAPPLYTYPE.驳回提交 || applyToSort.infoChangeOptionType == INFOCHANGEAPPLYTYPE.通过 ) {
        return successResult();
    } 

    //直接修改
    let updateInfo = {
        unitName, uscc, legalPerson, unitMemberType, uusinessLicenseUrl, applicationForm, desc, yuanXiaoBanXueLeiXing, legalPersonMail, legalPersonPhone
    };

    await updateOneData(TABLEENUM.资料变更审批历史表, {applyId:applyToSort.applyId}, updateInfo);

    return successResult();
}


/**
 * 资料变更回显
 * @param param0 
 * @returns 
 */
export async function infomationChangeInfo({userId}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId});

    let dataInfo:any = {};
    let isUpdate = true;
    //这里的条件有点乱，因为加班到很晚了
    if (!userInfo.infoChangeId || (userInfo.infoChangeId && userInfo.infoChangeOptionType == INFOCHANGEAPPLYTYPE.通过)) {
        let userInfo = await findOnce(TABLEENUM.用户表, {userId});
        if (userInfo.memberType != MEMBERTYPE.单位会员) throw new BizError(ERRORENUM.非单位会员不可操作);
        dataInfo = {
            unitName:userInfo.unitName || "",//单位会员名称
            uscc:userInfo.uscc || "",//统一信用代码
            legalPerson:userInfo.legalPerson || "",//法人代表
            unitMemberType:userInfo.unitMemberType || "",//单位会员类型
            uusinessLicenseUrl:userInfo.uusinessLicenseUrl || "",//营业执照
            applicationForm:userInfo.applicationForm || "",//盖章申请表
            desc:"",
            yuanXiaoBanXueLeiXing:userInfo.yuanXiaoBanXueLeiXing || "",
            legalPersonMail:userInfo.legalPersonMail,
            legalPersonPhone:userInfo.legalPersonPhone,
        };
        isUpdate = true;
    } else {
        let infomationChangeHistoryInfo = await findOnceToSort(TABLEENUM.资料变更审批历史表, {userId}, {createTimeMs:-1});
        dataInfo = {
            unitName:infomationChangeHistoryInfo.unitName,//单位会员名称
            uscc:infomationChangeHistoryInfo.uscc,//统一信用代码
            legalPerson:infomationChangeHistoryInfo.legalPerson,//法人代表
            unitMemberType:infomationChangeHistoryInfo.unitMemberType,//单位会员类型
            uusinessLicenseUrl:infomationChangeHistoryInfo.uusinessLicenseUrl,//营业执照
            applicationForm:infomationChangeHistoryInfo.applicationForm,//盖章申请表
            desc:infomationChangeHistoryInfo.desc,
            yuanXiaoBanXueLeiXing:infomationChangeHistoryInfo.yuanXiaoBanXueLeiXing,
            legalPersonMail:infomationChangeHistoryInfo.legalPersonMail,
            legalPersonPhone:infomationChangeHistoryInfo.legalPersonPhone,
        };
        if (infomationChangeHistoryInfo.infoChangeOptionType == INFOCHANGEAPPLYTYPE.待审批 || infomationChangeHistoryInfo.infoChangeOptionType == INFOCHANGEAPPLYTYPE.驳回提交) {
            isUpdate = false; 
        }
        else isUpdate = true;//true是可以编辑
    }
    const UnitConfig = ["unitMemberType", "yuanXiaoBanXueLeiXing", "yuanXiaoZhuGuanBuMen", "yuanXiaoFuZeRen", "yuanXiaoFuZeRenZhiWu", "yuanXiaoBanGongFuZeRenZhiWu", 
        "yuanXiaoFuZeRenDianHua", "yuanXiaoBanGongFuZeRen", "yuanXiaoBanGongFuZeRenDianHua", "ZhuYaoFuZeRenYouXiang", "yuanXiaoKeYanFuZeRen", "yuanXiaoKeYanFuZeRenDianHua", 
        "jiaoXueFuZeRenXinMing", "yuanXiaoXueShengZongRenShu", "yuanXiaoJiaoZhiGongZongRenShu", "jiaoXueFuZeRenDianHua", "kaishezhuanYe", "danWeiJianJie"];
        
    let unitInfo:any = extractData(userInfo, UnitConfig);
    if (unitInfo.unitMemberType) unitInfo.unitMemberType = changeEnumValue(UNITMEMBERTYPE, unitInfo.unitMemberType);
    if (unitInfo.yuanXiaoBanXueLeiXing) unitInfo.yuanXiaoBanXueLeiXing = changeEnumValue(BANXUELEIXING, unitInfo.yuanXiaoBanXueLeiXing);
    
    dataInfo.unitInfo = unitInfo;
    dataInfo.isUpdate = isUpdate;
    return {dataInfo};
}





