/**
 * 权限管理
 */

import { ADMINFIRSTDIR, ADMINLV, ADMINTYPE, MEMBERSTATE, STATE } from "../../config/enum";
import { ERRORENUM } from "../../config/errorEnum";
import { TABLEENUM } from "../../data/models/model";
import { findOnce, findToPage } from "../../data/select";
import { BizError } from "../../util/bizError";
import { changeEnumValue, eccEnumValue } from "../../util/verificationEnum";
import * as sysTools from "../../tools/system";
import { addOneData } from "../../data/add";
import { updateOneData } from "../../data/update";

/**
 * 管理员列表
 * @param param0 
 * @returns 
 */
export async function adminList({pageNumber}) {
    let fileList = ["userId", "memberState", "loginId", "adminLv", "adminType"];
    let dbList = await findToPage(TABLEENUM.用户表, {isAdmin:STATE.是}, fileList, pageNumber);

    let dataList = [];
    dbList.forEach(info => {
        let {userId, memberState, loginId, adminLv, adminType} = info;
        dataList.push({
            userId,
            memberState:changeEnumValue(MEMBERSTATE, memberState),
            loginId,
            adminType,
            adminLv
        });
    });

    return {dataList};
}

/**
 * 添加管理员 success
 */
export async function addAdmin({userId, form}) {
    let {loginId, pwd, confirmation, adminType, adminLv} = form;

    /**校验参数 */
    eccEnumValue("添加管理员", "管理员类别", ADMINTYPE, adminType);
    eccEnumValue("添加管理员", "管理员等级", ADMINLV, adminLv);
    /**校验密码 */
    if (pwd.search(/^[A-Za-z0-9]{6,18}$/) < 0) throw new BizError(ERRORENUM.密码只能由6至18位字符和数字组成);
    if (pwd != confirmation)  throw new BizError(ERRORENUM.两次密码不一致);

    let addInfo = {
        userId:sysTools.generateUserId(),//系统生成唯一id
        loginId,
        pwd,
        adminType,
        adminLv,
        isAdmin:STATE.是
    };

    await addOneData(TABLEENUM.用户表, addInfo);

    return sysTools.successResult();
}

/**
 * 修改密码
 * @param param0 
 * @returns 
 */
export async function changeAdminPwd({userId, form}) {
    let {pwd, confirmation} = form;

    let userInfo = await findOnce(TABLEENUM.用户表, {userId:form.userId}, ["id"]);
    if (!userInfo || !userInfo.id) throw new BizError(ERRORENUM.目标数据不存在);

    /**校验密码 */
    if (pwd.search(/^[A-Za-z0-9]{6,18}$/) < 0) throw new BizError(ERRORENUM.密码只能由6至18位字符和数字组成);
    if (pwd != confirmation)  throw new BizError(ERRORENUM.两次密码不一致);

    await updateOneData(TABLEENUM.用户表, {userId:form.userId}, {pwd:form.pwd});

    return sysTools.successResult();
}


/**
 * 修改权限
 */
export async function updateMGMT({id, adminLv}) {
    eccEnumValue("修改管理员权限权限", "权限列表", ADMINLV, adminLv);
    let userInfo = await findOnce(TABLEENUM.用户表, {userId:id}, ["adminLv", "userId", "loginId"]);
    if (!userInfo || !userInfo.id) throw new BizError(ERRORENUM.目标数据不存在);

    await updateOneData(TABLEENUM.用户表, {userId:id}, {adminLv});    

    return sysTools.successResult();
}


/**
 * 回显
 */
export async function info({id}) {
    let userInfo = await findOnce(TABLEENUM.用户表, {userId:id}, ["adminLv", "userId", "loginId"]);
    if (!userInfo || !userInfo.id) throw new BizError(ERRORENUM.目标数据不存在);

    let dataInfo = {
        adminLv:userInfo.adminLv,
        loginId:userInfo.loginId
    };

    return {dataInfo}
}

/**
 * 权限下拉列表
 */
export async function mgmtSelectList() {
   
    let distinctMap = {};//去重
    for (let key in ADMINFIRSTDIR) {
        let anyKey:any = key;
        if (isNaN(anyKey)) {//字符
            let keyStr = key;
            let value = ADMINFIRSTDIR[key];
            distinctMap[value] = {
                key:keyStr,
                value,
                subList:[]
            }
        }
    }
    
    for (let key in ADMINLV) {
        let anyKey:any = key;
        if (isNaN(anyKey)) {//字符
            let keyStr = key;
            let value:any = ADMINFIRSTDIR[key];
            let firstKey = Math.floor(value/100);
            distinctMap[firstKey].subList.push({
                key:keyStr,
                value
            });
        }
    }
    let dataList = Object.values(distinctMap);
   
    return {dataList};
}