/**
 * 会员管理逻辑
 */

import { MEMBERSTATE, MEMBERLEVEL, DOCUMENTTYPE, NATION, PROFCATEGORY, CODETYPE, MEMBERTYPE, REGISTERFLOW, SEX, USERREGISTERSTATE, OPERATIONREHAVIOR, STATE, BANXUELEIXING, UNITMEMBERTYPE } from "../../config/enum";
import { ERRORENUM } from "../../config/errorEnum";
import { TABLEENUM } from "../../data/models/model";
import { find, findCount, findOnce, findOnceToSort, findToPage, findToSort } from "../../data/select";
import { updateOneData } from "../../data/update";
import { generateSystemId, successResult } from "../../tools/system";
import { BizError } from "../../util/bizError";
import { extractData } from "../../util/piecemeal";
import { changeEnumValue, eccEnumValue } from "../../util/verificationEnum";
import { addManyData, addOneData } from "../../data/add";
import { deleteManyData, deleteOneData } from "../../data/delete";
import moment = require("moment");
import { getEdition } from "./examine";


/**
 * 单位会员列表 success
 * @param unitName 单位名称 
 * @param joinTime 入会时间
 * @param unitMemberType 单位会员所属类型 UNITMEMBERTYPE
 * @param memberLevel 会员等级 职务 MEMBERLEVEL
 * @param session 届次 
 * @param sheng 省
 * @param shi 市
 * @param qu 区
 * @param pageNumber 当前页
 */
export async function unitMemberList({unitName, joinTime, unitMemberType, memberLevel, session, sheng, shi, qu, pageNumber}) {
    let selectParam:any = {isAdmin:STATE.否};
    if (unitName) selectParam.unitName = unitName;
    if (joinTime) selectParam.joinTime = joinTime;
    if (unitMemberType) selectParam.unitMemberType = unitMemberType;
    if (memberLevel) selectParam.memberLevel = memberLevel;
    if (session) {
        let thisYear = moment(session).format("YYYY");
        let thisMonth = moment(session).format("MM");
        selectParam.session = getEdition(thisYear, thisMonth);;
    }
    if (sheng) selectParam.sheng = sheng;
    if (shi) selectParam.shi = shi;
    if (qu) selectParam.qu = qu;
    selectParam.memberType = MEMBERTYPE.单位会员;
    selectParam.userRegisterState = USERREGISTERSTATE.通过;

    const SelectFiles = ["userId", "memberState", "unitName", "loginId", "phone", "joinTime", "unitMemberLevel", "sheng", "shi", "qu"];
    let dbList = await findToPage(TABLEENUM.用户表, selectParam, SelectFiles, pageNumber);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    dbList.forEach(info => {
        let item:any = extractData(info, SelectFiles);
        item.memberState = changeEnumValue(MEMBERSTATE, item.memberState);
        if (!item.memberLevel) item.memberLevel = "未选择";
        else item.memberLevel = changeEnumValue(MEMBERLEVEL, item.memberLevel);
        dataList.push(item);
    });

    return {dataList, dataCount};
}


/**
 * 单位会员详情 
 * @param userId 
 * @returns {顶部数据, 基本信息, 单位信息, 单位所获得科研成果}
 */

const UnitBasicConfig = ["userId", "unitName", "addres", "uscc", "legalPerson", "legalPersonPhone", "contactPerson", "contactPersonDuties", "sheng", "shi", "qu", 
"uusinessLicenseUrl", "education"];
const UnitConfig = ["unitMemberType", "yuanXiaoBanXueLeiXing", "yuanXiaoZhuGuanBuMen", "yuanXiaoFuZeRen", "yuanXiaoFuZeRenZhiWu", "yuanXiaoFuZeRenDianHua", 
"yuanXiaoBanGongFuZeRenZhiWu", "yuanXiaoBanGongFuZeRenDianHua", "yuanXiaoXueShengZongRenShu", "yuanXiaoJiaoZhiGongZongRenShu", "yuanXiaoJianZhuMianJi",
"yuanXiaoKeYanFuZeRen", "yuanXiaoKeYanFuZeRenDianHua", "jiaoXueFuZeRenXinMing", "jiaoXueFuZeRenDianHua", "ZhuYaoFuZeRenYouXiang"];

export async function unitMemberDetails({id}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId:id});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    let topInfo:any = extractData(oldInfo, ["session", "memberLevel"]);
    /**处理枚举值 */
    topInfo.memberLevel = changeEnumValue(MEMBERLEVEL, topInfo.memberLevel);
    let basicInfo = extractData(oldInfo, UnitBasicConfig);
    let unitInfo = extractData(oldInfo, UnitConfig);
    
    let oldKeYanInfo = await findOnce(TABLEENUM.单位所获得科研成果表, {userId:id});
    let scientificResearch = extractData(oldKeYanInfo, ["id", "startTime", "endTime", "describe"]);
    
    return {topInfo, basicInfo, unitInfo, scientificResearch};
}


/**
 * 单位会员 修改
 * @param form 修改信息表单
 * @param keyanForm 修改科研信息表单
 * @param userId 
 * @returns 
 */
export async function unitMemberUpdate({form, keyanForm, id}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId:id});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    /**校验枚举 */
    const EccFuncName = '单位会员修改';
    eccEnumValue(EccFuncName, '单位信息类型', UNITMEMBERTYPE, form.unitMemberType);
    
    /**修改会员用户信息 */
    await updateOneData(TABLEENUM.用户表, {userId:id}, form);

    /**删除对应userId科研成果 */
    let oldKeYanInfo = await find(TABLEENUM.单位所获得科研成果表, {userId:id});
    if (!oldKeYanInfo || !oldKeYanInfo.id) throw new BizError(ERRORENUM.目标数据不存在);
    await deleteManyData(TABLEENUM.单位所获得科研成果表, {userId:id});

    /**添加新的科研信息 */
    let addList = [];
    for (let i = 0; i < keyanForm.length; i++) {
        keyanForm[i].id = generateSystemId(TABLEENUM.单位所获得科研成果表, id);
        keyanForm[i].userId = id;
        addList.push(keyanForm);
    }
    await addManyData(TABLEENUM.单位所获得科研成果表, addList);

    return successResult();
}


/**
 * 删除单位所获得科研成果
 * @param param0 
 * @returns 
 */
export async function delResearch({id}) {
    let dataInfo = await findOnce(TABLEENUM.单位所获得科研成果表, {id});
    if (!dataInfo || !dataInfo.id) throw new BizError(ERRORENUM.目标数据不存在);
    await deleteOneData(TABLEENUM.单位所获得科研成果表, {id});
    return successResult();
}




/**
 * 个人会员列表
 * @param param0 
 */
export async function individualMemberList({name, documentId, phone, individualMemberType, memberLevel, session, joinTime, auditTime, memberState, pageNumber}) {
    let selectParam:any = {};
    if (name) selectParam.name = name;
    if (documentId) selectParam.documentId = documentId;
    if (phone) selectParam.phone = phone;
    if (individualMemberType) selectParam.individualMemberType = individualMemberType;
    if (memberLevel) selectParam.memberLevel = memberLevel;
    if (session) {
        let thisYear = moment(session).format("YYYY");
        let thisMonth = moment(session).format("MM");
        selectParam.session = getEdition(thisYear, thisMonth);;
    }
    if (joinTime) selectParam.joinTime = joinTime;
    if (auditTime) selectParam.auditTime = auditTime;
    if (memberState) selectParam.memberState = memberState;
    selectParam.memberType = MEMBERTYPE.个人会员;
    selectParam.userRegisterState = USERREGISTERSTATE.通过;

    let selectConf = ["userId", "memberState", "unitName", "name", "sex", "nation", "loginId", "phone", "joinTime", "memberLevel", "sheng", "shi", "qu"];
    let dbList = await findToPage(TABLEENUM.用户表, selectParam, selectConf, pageNumber);
    let dataCount = await findCount(TABLEENUM.用户表, selectParam);

    let dataList = [];
    dbList.forEach(info => {
        let item:any = extractData(info, selectConf);
        item.memberState = changeEnumValue(MEMBERSTATE, item.memberState);
        if (!item.memberLevel) item.memberLevel = "未选择";
        else item.memberLevel = changeEnumValue(MEMBERLEVEL, item.memberLevel);
        dataList.push(item);
    });

    return {dataList, dataCount};
}


/**
 * 个人会员详情
 * @param param0 
 * @returns {顶部数据, 基本信息, 专业信息, 其他附件}
 */

const IndividualBasicConfig = ["name", "sex", "nation", "birth", "documentType", "documentId", "sheng", "shi", "qu", "addres", "photoUrl"];
const SpecialityConfig = ["unitName", "memberLevel", "workTitle", "profCategory", "studyResume", "workResume", "profAchievement", "otherEMP", "otherPROF"];

export async function individualMemberDetails({id}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId:id});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    let topInfo:any = extractData(oldInfo, ["session", "memberLevel"]);
    topInfo.memberLevel = changeEnumValue(MEMBERLEVEL, topInfo.memberLevel);

    let basicInfo:any = extractData(oldInfo, IndividualBasicConfig);
    basicInfo.documentType = changeEnumValue(DOCUMENTTYPE, basicInfo.documentType);
    basicInfo.nation = changeEnumValue(NATION, basicInfo.nation);

    let specialityInfo:any = extractData(oldInfo, SpecialityConfig);
    basicInfo.profCategory = changeEnumValue(PROFCATEGORY, basicInfo.profCategory);
    
    let otherAttachment:any = extractData(oldInfo, ["cardUrl", "academicCERTUrl", "professionalCERTUrl", "auxiliaryMaterial"]);
    
    return {topInfo, basicInfo, specialityInfo, otherAttachment};
}


/**
 * 个人会员 修改
 * @param userId
 * @param form
 * @returns 
 */
 export async function individualMemberUpdate({form, id}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId:id});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    await updateOneData(TABLEENUM.用户表, {userId:id}, form);

    return successResult();
}













/**
 * 模拟审核通过，修改审核状态
 */
//  export async function examine({userId}) {
//     let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
//     if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
//     let memberList:any = extractData(oldInfo, ["userId", "registerFlow", "memberType"]);

//     let selectParam = {registerFlow:memberList.registerFlow, userRegisterState:USERREGISTERSTATE.待审核};

//     if (memberList.memberType == MEMBERTYPE.个人会员) { //个人会员
//         switch (memberList.registerFlow) {
//             case REGISTERFLOW.个人会员注册1:
//                 selectParam.registerFlow = REGISTERFLOW.个人会员注册2;
//             break;
//             case REGISTERFLOW.个人会员注册2:
//                 selectParam.userRegisterState = USERREGISTERSTATE.通过;
//             break;
//         }
//     } else { //单位会员
//         switch (memberList.registerFlow) {
//             case REGISTERFLOW.单位会员注册1:
//                 selectParam.registerFlow = REGISTERFLOW.单位会员注册2;
//             break;
//             case REGISTERFLOW.单位会员注册2:
//                 selectParam.registerFlow = REGISTERFLOW.单位会员注册3;
//             break;
//             case REGISTERFLOW.单位会员注册3:
//                 selectParam.userRegisterState = USERREGISTERSTATE.通过;
//             break;
//         }
//     }

//     await updateOneData(TABLEENUM.用户表, {userId}, selectParam);

//     return successResult();
// }



