/**
 * 会员首页展板逻辑
 * 【二期内容：申请理事、申请常务理事、申请专家会员、会员积分、活动列表[立即参加]、消息中心暂时展示缴费提醒】
 */

import moment = require("moment");
import { BANXUELEIXING, DOCUMENTTYPE, INDIVIDUALMEMBERTYPE, INVOICESTATUS, MEMBERLEVEL, MEMBERTYPE, NATION, ORDERSTATE, PROFCATEGORY, SEX, STATE, UNITMEMBERTYPE } from "../../config/enum";
import { ERRORENUM } from "../../config/errorEnum";
import { addManyData } from "../../data/add";
import { deleteManyData } from "../../data/delete";
import { TABLEENUM } from "../../data/models/model";
import { find, findOnce } from "../../data/select";
import { updateOneData } from "../../data/update";
import { generateSystemId, successResult } from "../../tools/system";
import { BizError } from "../../util/bizError";
import { extractData } from "../../util/piecemeal";
import { changeEnumValue, eccEnumValue } from "../../util/verificationEnum";


/**
 * 会员信息展示
 * @param userId 
 * @returns {头部数据， 基本信息， 专业信息}
 */
export async function getMemberData({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    let topSelectFiles = ["loginId", "memberType", "userRegisterState", "integral", "lifespanStartTime", "lifespanEndTime", "memberLevel"];
    let baseSelectFiles = [];
    let majorSelectFiles = [];
    /**判断会员类型，个人会员与单位会员返回不同字段 */
    if (oldInfo.memberType == MEMBERTYPE.个人会员) {
        baseSelectFiles = ["name", "sex", "documentType", "documentId", "sheng", "shi", "qu", "nation", "birth", "addres"];
        majorSelectFiles = ["workUnit", "duties", "workTitle", "studyResume", "workResume", "profAchievement", "otherEMP", "otherPROF", "profCategory"];
    } else if (oldInfo.memberType == MEMBERTYPE.单位会员) {
        baseSelectFiles = ["unitName", "uscc", "sheng", "shi", "qu", "nation", "birth", "addres", "legalPerson", "legalPersonPhone", "legalPersonMail", "unitMail", "contactPerson", "contactPersonDuties", 
        "contactPersonPhone", "uusinessLicenseUrl"];
        majorSelectFiles = ["unitMemberType", "yuanXiaoBanXueLeiXing", "yuanXiaoZhuGuanBuMen", "yuanXiaoFuZeRen", "yuanXiaoFuZeRenZhiWu", "yuanXiaoFuZeRenDianHua", 
        "yuanXiaoBanGongFuZeRen", "yuanXiaoBanGongFuZeRenDianHua", "ZhuYaoFuZeRenYouXiang", "yuanXiaoKeYanFuZeRen", "yuanXiaoKeYanFuZeRenDianHua", "jiaoXueFuZeRenXinMing", 
        "yuanXiaoXueShengZongRenShu", "yuanXiaoJiaoZhiGongZongRenShu", "jiaoXueFuZeRenDianHua"];
    }

    /**返回指定配置字段 */
    let topInfo:any = extractData(oldInfo, topSelectFiles);
    let baseInfo:any = extractData(oldInfo, baseSelectFiles);
    let majorInfo:any = extractData(oldInfo, majorSelectFiles);

    /**处理枚举值 */
    if(topInfo.memberLevel) topInfo.memberLevel = changeEnumValue(MEMBERLEVEL, topInfo.memberLevel);
    if(baseInfo.sex) baseInfo.sex = changeEnumValue(SEX, baseInfo.sex);
    if(baseInfo.documentType) baseInfo.documentType = changeEnumValue(DOCUMENTTYPE, baseInfo.documentType);
    if(baseInfo.nation) baseInfo.nation = changeEnumValue(NATION, baseInfo.nation);
    if(majorInfo.profCategory) majorInfo.profCategory = changeEnumValue(PROFCATEGORY, majorInfo.profCategory);
    if(majorInfo.unitMemberType) majorInfo.unitMemberType = changeEnumValue(UNITMEMBERTYPE, majorInfo.unitMemberType);
    if(majorInfo.yuanXiaoBanXueLeiXing) majorInfo.yuanXiaoBanXueLeiXing = changeEnumValue(BANXUELEIXING, majorInfo.yuanXiaoBanXueLeiXing);

    return {topInfo, baseInfo, majorInfo};
}


/**
 * 单位会员 修改
 * @param form 修改信息表单
 * @param keyanForm 修改科研信息表单
 * @param userId 
 * @returns 
 */
export async function unitMemberUpdate({form, keyanForm, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);

    /**校验枚举 */
    const EccFuncName = '单位会员修改';
    eccEnumValue(EccFuncName, '单位信息类型', UNITMEMBERTYPE, form.unitMemberType);

    /**修改会员用户信息 */
    await updateOneData(TABLEENUM.用户表, {userId}, form);

    /**删除对应userId科研成果 */
    let oldKeYanList = await find(TABLEENUM.单位所获得科研成果表, {userId});
    if (!oldKeYanList || !oldKeYanList.id) throw new BizError(ERRORENUM.目标数据不存在);
    await deleteManyData(TABLEENUM.单位所获得科研成果表, {userId});

    /**添加新的科研信息 */
    let addList = [];
    for (let i = 0; i < keyanForm.length; i++) {
        keyanForm[i].id = generateSystemId(TABLEENUM.单位所获得科研成果表, userId);
        keyanForm[i].userId = userId;
        addList.push(keyanForm);
    }
    await addManyData(TABLEENUM.单位所获得科研成果表, addList);

    return successResult();
}


/**
 * 个人会员 修改
 * @param userId
 * @param form 修改信息表单
 * @returns 
 */
 export async function individualMemberUpdate({form, userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId});
    if (!oldInfo || !oldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    await updateOneData(TABLEENUM.用户表, {userId}, form);

    return successResult();
}


/**
 * 消息中心列表
 * 缴费消息提醒
 */
export async function getXiaoXiZhongXin ({userId}) {
    let oldInfo = await findOnce(TABLEENUM.用户表, {userId}, ["lifespanEndTime", "isGracePeriod", "gracePeriodEndTime"]);
    if (!oldInfo || !oldInfo.lifespanEndTime) throw new BizError(ERRORENUM.目标数据不存在);

    let dataList:any = extractData(oldInfo, ["lifespanStartTime", "lifespanEndTime"]);

    const NowMs = new Date().valueOf();

    /**判断是否宽限期 */
    if (dataList.isGracePeriod == STATE.否) {
        /**
         * 判断会费到期时间
         */
        switch (oldInfo.lifespanEndTime) {
            case oldInfo.lifespanEndTime > NowMs+(90*24*3600*1000):
                dataList["membershipFees"] = "大于90天";
            break;
            case oldInfo.lifespanEndTime-(90*24*3600*1000) <= NowMs && oldInfo.lifespanEndTime > NowMs-(24*3600*1000):
                dataList["membershipFees"] = "不足90天";
            break;
            case NowMs <= oldInfo.lifespanEndTime+(30*24*3600*1000) && NowMs > oldInfo.lifespanEndTime+(24*3600*1000):
                dataList["membershipFees"] = "过期30天内";
            break;
            case NowMs > oldInfo.lifespanEndTime+(30*24*3600*1000) && NowMs <= oldInfo.lifespanEndTime+(90*24*3600*1000):
                dataList["membershipFees"] = "过期90天内";
            break;
            case NowMs > oldInfo.lifespanEndTime+(90*24*3600*1000) && NowMs <= oldInfo.lifespanEndTime+(365*24*3600*1000):
                dataList["membershipFees"] = "过期1年内";
            break;
            case NowMs > oldInfo.lifespanEndTime+(365*24*3600*1000):
                dataList["membershipFees"] = "过期1年外";
            break;
        }
    } else {
        if (oldInfo.gracePeriodEndTime > NowMs) dataList["membershipFees"] = "宽限期中";
    }

    return {dataList};
}


/**
 * 地图省数量统计
 * @param param0 
 * @returns 
 */
export async function getSheng ({userId}) {
    let myOldInfo = await findOnce(TABLEENUM.用户表, {userId}, ["sheng"]);
    if (!myOldInfo || !myOldInfo.sheng) throw new BizError(ERRORENUM.目标数据不存在);
    let myDataInfo = extractData(myOldInfo, ["sheng"]);

    let allOldList = await find(TABLEENUM.用户表, {}, ["sheng"]);
    if (!allOldList || !allOldList.length) throw new BizError(ERRORENUM.目标数据不存在);
    
    let allDataList = {};
    allOldList.forEach( info => {
        //过滤省份为undefined的数据,统计省份个数
        if (info.sheng) {
            if (!allDataList[info.sheng]) allDataList[info.sheng] = 0;
            allDataList[info.sheng] += 1;
        }
    })

    return {myDataInfo, allDataList};
}


/**
 * 活动列表
 */
 export async function getHuoDongTongZhi () {
    let oldList = await find(TABLEENUM.活动通知, {}, ["id", "name"]);
    if (!oldList || oldList.length<1) throw new BizError(ERRORENUM.目标数据不存在);
    
    let dataList = [];
    oldList.forEach( info => {
        let item = extractData(info, ["id", "name"]);
        dataList.push(item);
    })

    return {dataList};
}


/**
 * 会员缴费列表
 * @param userId 
 * @param state 
 * @returns dataList:[{data:列表, proofLetterInfo:证明函}]
 */
export async function memberPayment({userId, state}) {
    let selectParam:any = {userId};
    if(state) selectParam.state = state;
    let oldList = await find(TABLEENUM.订单表, selectParam, ["id", "userId", "unitName", "orderCycleStart", "orderCycleEnd", "money", "state"]);
    if (!oldList || oldList.length<1) throw new BizError(ERRORENUM.目标数据不存在);
    /**查询证明函需要字段会员等级 */
    let myOldInfo = await findOnce(TABLEENUM.用户表, {userId:oldList.userId}, ["memberLevel"]);
    if (!myOldInfo || !myOldInfo.userId) throw new BizError(ERRORENUM.目标数据不存在);
    //证明函返回数据配置
    const proofLetterFiles = ["userId", "unitName", "orderCycleStart", "orderCycleEnd", "money"];

    let dataList = [];
    oldList.forEach( async info => {
        /**处理列表返回数据字段 */
        let item:any = extractData(info, ["id", "orderCycleStart", "orderCycleEnd", "money", "state"]);
        item.state = changeEnumValue(ORDERSTATE, item.state);

        /**处理证明函返回数据字段 */
        let proofLetterInfo:any = extractData(info, proofLetterFiles);
        proofLetterInfo.memberLevel = changeEnumValue(MEMBERLEVEL, myOldInfo.memberLevel);
        proofLetterInfo.orderCycleStart = moment(proofLetterInfo.orderCycleStart).format("YYYY");
        proofLetterInfo.orderCycleEnd = moment(proofLetterInfo.orderCycleEnd).format("YYYY");

        dataList.push({list:item, proofLetterInfo});
    })

    return {dataList};
}


/**
 * 申请发票
 * @param id 会员缴费列表返回的id 
 */
export async function applicationInvoice({id}) {
    let oldInfo = await findOnce(TABLEENUM.订单表, {id, invoiceStatus:INVOICESTATUS.未开发票}, ["invoiceAdd"]);
    if (!oldInfo || !oldInfo.invoiceAdd) throw new BizError(ERRORENUM.目标数据不存在);

    let dataInfo = extractData(oldInfo, ["invoiceAdd"]);

    return {dataInfo};
}


/**
 * 下载发票
 * @param id 会员缴费列表返回的id 
 */
export async function downInvoice({id}) {
    let oldInfo = await findOnce(TABLEENUM.订单表, {id, invoiceStatus:INVOICESTATUS.已开发票}, ["invoiceAdd"]);
    if (!oldInfo || !oldInfo.invoiceAdd) throw new BizError(ERRORENUM.目标数据不存在);

    let dataInfo = extractData(oldInfo, ["invoiceAdd"]);

    return {dataInfo};
}


/**
 * 上传转账截图
 * @param param0 
 */
export async function upTransferImg({id, weChartPR, offlinePaymentUrl}) {
    let oldInfo = await findOnce(TABLEENUM.订单表, {id});
    if (!oldInfo || !oldInfo.id) throw new BizError(ERRORENUM.目标数据不存在);

    await updateOneData(TABLEENUM.订单表, {id}, {weChartPR, offlinePaymentUrl});

    return successResult();
}









