/**
 * 企业服务追踪
 */

import * as verificationEnumTools from "../util/verificationEnum";
import * as configEnum from "../config/enum";
import * as industryEnum from "../config/enum/industryEnum";
import * as trackData from "../data/track";
import { checkChange, extractData } from "../util/piecemeal";
import * as splitResultConfig from '../config/splitResultConfig';
import { eccFormParam } from "../util/verificationParam";
import { TrackAddConfig, TrackDetailsAddConfig, TrackDetailsUpdateConfig, TrackUpdateConfig } from "../config/eccParam/admin";
import { randomId } from "../tools/system";
import { TABLEID } from "../config/enum/dbEnum";
import { BizError } from "../util/bizError";
import { ERRORENUM } from "../config/errorEnum";
import moment from "moment";


/**
 * 服务追踪-列表
 * @param name 企业名称
 * @param demand 需求类型
 * @param trackType 需求跟进状态
 * @param page 页
 * @returns 
 */
export async function trackList(name:string, demand:number, trackType:number, page:number) {
    let selectParam:any = {};
    if (name) {
        selectParam.name = {"$regex":`${name}`};
    }
    if (demand) {
        selectParam.demand = demand;
    }
    if (trackType) {
        selectParam.trackType = trackType;
    }

    /**需要用到的查询数据 */
    let trackDbList = await trackData.findTrackListToPage(selectParam, (page - 1) * 10);
    let count = await trackData.findTrackCount(selectParam);//符合查询条件的数据总数

    /**组合返回结果 */
    let dataList = [];
    trackDbList.forEach( info => {
        /**截取返回数据 */
        let changeData:any = extractData(splitResultConfig.trackListConfig, info, true);

        /**将枚举值转为字符 */
        changeData.demand = verificationEnumTools.changeEnumValue(configEnum.VISITDEMAND, changeData.demand);
        changeData.trackType = verificationEnumTools.changeEnumValue(configEnum.TRACKTYPE, changeData.trackType);

        /**获取最新一条需求追踪详情 */
        if (info.details && info.details.length > 0) {
            let detailsInfo = getLatestFollowUpRecord(info.details);
            changeData.followUpName = detailsInfo.followUpName; //跟进人
            changeData.followUpTime = moment(detailsInfo.followUpTime).format("YYYY-MM-DD"); //跟进时间
            changeData.followUpDsc = detailsInfo.followUpDsc; //跟进详情
        }

        dataList.push(changeData);
    })

    return {count, dataList};
}


/**
 * 服务追踪-新增
 * @param param 
 */
export async function trackCreate(param) {
    /**校验表单参数 */
    eccFormParam("管理后台新增服务追踪", TrackAddConfig, param);

    /**校验枚举 */
    verificationEnumTools.eccEnumValue('管理后台新增服务追踪', 'demand', configEnum.VISITDEMAND, param.demand);

    //服务追踪详情
    let details = [];
    if (param.details && param.details.length > 0) {
        /**校验表单详情参数 */
        param.details.forEach( info => {
            eccFormParam("管理后台新增服务追踪记录详情", TrackDetailsAddConfig, info);
            
            let item = {
                tdid: randomId(TABLEID.需求跟踪详情),
                followUpName: info.followUpName,
                followUpTime: info.followUpTime,
                followUpDsc: info.followUpDsc,
                followUpStatus: info.followUpStatus,
            }
    
            details.push(item)
        })
    }
    
    /**
     * 需求跟进状态需要获取服务追踪详情内最新一条的跟进状态followUpStatus
     */
    let trackType = configEnum.TRACKTYPE.未解决;
    if (param.details && param.details.length > 0) {
        let latestRecord = getLatestFollowUpRecord(details);
        trackType = latestRecord.followUpStatus;
    }

    let trackInfo = {
        id: randomId(TABLEID.企业需求表),
        uscc: param.uscc,
        name: param.name,
        demand: param.demand,
        content: param.content,
        contact: param.contact,
        contactUs: param.contactUs,
        trackType,
        createTime: param.createTime, //new Date().valueOf(),
        details,
    }

    await trackData.createDataByParam(trackInfo);

    return {isSuccess:true};
}


/**
 * 获取服务追踪详情最新记录
 * @param details 
 * @returns 
 */
function getLatestFollowUpRecord(details) {
    if (!details || details.length === 0) return null;
    
    return details.reduce((prev, current) => 
      prev.followUpTime > current.followUpTime ? prev : current
    );
}


/**
 * 服务追踪-回显
 * @param uscc 
 */
export async function trackInfo(id:string) {
    /**校验企业 */
    let trackDbInfo = await trackData.findTrackByParam({id});
    if (!trackDbInfo || !trackDbInfo.id) throw new BizError(ERRORENUM.未找到数据, `库中不存在id=${id}这个服务追踪`);

    //服务追踪详情
    let details = [];
    if (trackDbInfo.details && trackDbInfo.details.length > 0) {
        trackDbInfo.details.forEach( info => {
            let item = {
                tdid: info.tdid,
                followUpName: info.followUpName,
                followUpTime: info.followUpTime,
                followUpDsc: info.followUpDsc,
                followUpStatus: info.followUpStatus,
            }
    
            details.push(item)
        })
    }

    let dataInfo = {
        id:trackDbInfo.id,
        uscc: trackDbInfo.uscc,
        name: trackDbInfo.name,
        demand: trackDbInfo.demand,
        content: trackDbInfo.content,
        contact: trackDbInfo.contact,
        contactUs: trackDbInfo.contactUs,
        trackType: trackDbInfo.trackType,
        createTime: trackDbInfo.createTime,
        details,
    };

    return {dataInfo};
}


/**
 * 服务追踪详情-回显
 * @param uscc 
 */
export async function trackDetailsInfo(tdid:string) {
    /**校验服务追踪详情 */
    let selectParam = {details: {"$elemMatch":{tdid:{"$in":tdid} } } };
    let trackDbInfo = await trackData.findTrackByParam(selectParam);
    if (!trackDbInfo || !trackDbInfo.details || !trackDbInfo.details[0].tdid) throw new BizError(ERRORENUM.未找到数据, `库中不存在tdid=${tdid}这个服务追踪`);

    //服务追踪详情
    let details = trackDbInfo.details[0];

    let dataInfo = {
        tdid: details.tdid,
        followUpName: details.followUpName,
        followUpTime: details.followUpTime,
        followUpDsc: details.followUpDsc,
        followUpStatus: details.followUpStatus,
    }

    return {dataInfo};
}



/**
 * 服务追踪-修改
 * @param uscc 
 * @param param 
 */
export async function trackUpdate(id:string, param) {
    /**校验表单参数 */
    eccFormParam("管理后台修改服务追踪企业", TrackUpdateConfig, param);

    /**校验枚举 */
    verificationEnumTools.eccEnumValue('管理后台修改服务追踪企业', 'demand', configEnum.VISITDEMAND, param.demand);

    //服务追踪详情
    let details = [];
    if (param.details && param.details.length > 0) {
        /**校验表单详情参数 */
        param.details.forEach( info => {
            eccFormParam("管理后台新增服务追踪记录详情", TrackDetailsUpdateConfig, info);
            verificationEnumTools.eccEnumValue('管理后台修改服务追踪企业', 'followUpStatus', configEnum.TRACKTYPE, info.followUpStatus);
            
            let item = {
                tdid: randomId(TABLEID.需求跟踪详情),
                followUpName: info.followUpName,
                followUpTime: info.followUpTime,
                followUpDsc: info.followUpDsc,
                followUpStatus: info.followUpStatus,
            }
    
            details.push(item)
        })
    }

    /**校验企业 */
    let trackDbInfo = await trackData.findTrackByParam({id});
    if (!trackDbInfo || !trackDbInfo.id) throw new BizError(ERRORENUM.未找到数据, `库中不存在id=${id}这个企业`);
    
    /**
     * 需求跟进状态需要获取服务追踪详情内最新一条的跟进状态followUpStatus
     */
    if (param.details && param.details.length > 0) {
        let latestRecord = getLatestFollowUpRecord(details);
        trackDbInfo.trackType = latestRecord.followUpStatus;
    }

    /**修改字段 */
    let changeList = checkChange(param, trackDbInfo);
    if (!changeList.length) throw new BizError(ERRORENUM.数据无更新, `${param.name}数据无更新`);
    changeList.forEach(key => {
        trackDbInfo[key] = param[key];
    });

    await trackDbInfo.save();

    return {isSuccess:true};
}


/**
 * 服务追踪-删除
 * @param id 
 */
export async function trackDelete(id:string) {
    let trackDbInfo = await trackData.findTrackByParam({id});
    if (!trackDbInfo || !trackDbInfo.name) throw new BizError(ERRORENUM.未找到数据, `库中不存在id=${id}这个企业`);

    await trackData.removeOneData(id);

    return {isSuccess:true};
}


/**
 * 导出服务追踪信息列表
 * @param name 企业名称
 * @param demand 需求类型
 * @param trackType 需求跟进状态
 * @returns 二维数组，第一行为表头，后续行为数据
 */
 export async function outPutTrackData(name: string, demand: number, trackType: number) {
    /** 拼接查询条件 */
    let selectParam: any = {};
    if (name) {
        selectParam.name = { "$regex": `${name}` };
    }
    if (demand) {
        selectParam.demand = demand;
    }
    if (trackType) {
        selectParam.trackType = trackType;
    }

    /** 查询服务追踪数据 */
    let trackDbList = await trackData.findTrackListByParam(selectParam);

    /** 表头定义 */
    const titleList = [
        "企业名称",
        "统一社会信用代码",
        "需求类型",
        "需求内容",
        "联系人",
        "联系方式",
        "需求跟进状态",
        "创建时间",
        "最新跟进人",
        "最新跟进时间",
        "最新跟进详情",
        "最新跟进状态",
        "总跟进次数"
    ];

    let dataList = [titleList]; // 第一行为表头

    trackDbList.forEach(info => {
        /** 提取数据 */
        let changeData: any = extractData(splitResultConfig.trackListConfig, info, true);

        /** 枚举值转文本 */
        changeData.demand = verificationEnumTools.changeEnumValue(configEnum.VISITDEMAND, changeData.demand) || "未选择";
        changeData.trackType = verificationEnumTools.changeEnumValue(configEnum.TRACKTYPE, changeData.trackType) || "未选择";

        /** 时间格式化 */
        changeData.createTime = changeData.createTime ? moment(changeData.createTime).format("YYYY-MM-DD") : "-";

        /** 处理跟进详情 */
        let latestFollowUpName = "-";
        let latestFollowUpTime = "-";
        let latestFollowUpDsc = "-";
        let latestFollowUpStatus = "-";
        let followUpCount = 0;

        if (info.details && info.details.length > 0) {
            // 获取最新跟进记录
            let latestRecord = getLatestFollowUpRecord(info.details);
            if (latestRecord) {
                latestFollowUpName = latestRecord.followUpName || "-";
                latestFollowUpTime = latestRecord.followUpTime ? moment(latestRecord.followUpTime).format("YYYY-MM-DD") : "-";
                latestFollowUpDsc = latestRecord.followUpDsc || "-";
                latestFollowUpStatus = verificationEnumTools.changeEnumValue(configEnum.TRACKTYPE, latestRecord.followUpStatus) || "-";
            }
            
            // 跟进次数
            followUpCount = info.details.length;
        }

        /** 生成一行数据 */
        let row = [
            changeData.name || "-",
            changeData.uscc || "-",
            changeData.demand,
            changeData.content || "-",
            changeData.contact || "-",
            changeData.contactUs || "-",
            changeData.trackType,
            changeData.createTime,
            latestFollowUpName,
            latestFollowUpTime,
            latestFollowUpDsc,
            latestFollowUpStatus,
            followUpCount.toString()
        ];

        dataList.push(row);
    });

    return dataList;
}


/**
 * 导出服务追踪详细信息（空行分隔不同企业）
 * @param name 企业名称
 * @param demand 需求类型
 * @param trackType 需求跟进状态
 * @returns 二维数组，第一行为表头，后续行为数据（空行分隔）
 */
export async function outPutTrackDetailsData(name: string, demand: number, trackType: number, id:string[]) {
    /** 拼接查询条件 */
    let selectParam: any = {};
    if (name) {
        selectParam.name = { "$regex": `${name}` };
    }
    if (demand) {
        selectParam.demand = demand;
    }
    if (trackType) {
        selectParam.trackType = trackType;
    }
    if (id) {
        selectParam.id = {"$in": id};
    }

    /** 查询服务追踪数据 */
    let trackDbList = await trackData.findTrackListByParam(selectParam);

    /** 表头定义 */
    const titleList = [
        "企业名称",
        "统一社会信用代码",
        "需求类型",
        "需求内容",
        "联系人",
        "联系方式",
        "需求创建时间",
        "当前跟进状态",
        "跟进人",
        "跟进时间",
        "跟进详情",
        "跟进状态",
        // "跟进记录ID"
    ];

    let dataList = [titleList]; // 第一行为表头

    /** 处理每个企业的跟踪记录 */
    trackDbList.forEach((info, enterpriseIndex) => {
        /** 提取基础数据 */
        let changeData: any = extractData(splitResultConfig.trackListConfig, info, true);

        /** 枚举值转文本 */
        const demandText = verificationEnumTools.changeEnumValue(configEnum.VISITDEMAND, changeData.demand) || "未选择";
        const currentTrackType = verificationEnumTools.changeEnumValue(configEnum.TRACKTYPE, changeData.trackType) || "未选择";

        /** 时间格式化 */
        const createTime = changeData.createTime ? moment(changeData.createTime).format("YYYY-MM-DD") : "-";

        /** 在企业记录前添加空行（除了第一个企业） */
        if (enterpriseIndex > 0) {
            dataList.push(new Array(titleList.length).fill(""));
        }

        /** 处理跟进详情 */
        if (info.details && info.details.length > 0) {
            // 按跟进时间倒序排列，最新的在前
            const sortedDetails = [...info.details].sort((a, b) => 
                moment(b.followUpTime).valueOf() - moment(a.followUpTime).valueOf()
            );

            sortedDetails.forEach((detail, detailIndex) => {
                const followUpStatus = verificationEnumTools.changeEnumValue(configEnum.TRACKTYPE, detail.followUpStatus) || "未选择";
                const followUpTime = detail.followUpTime ? moment(detail.followUpTime).format("YYYY-MM-DD") : "-";
                
                /** 生成一行数据 */
                let row = [
                    // 只在第一条记录显示企业信息，后续记录留空（Excel可以合并单元格）
                    detailIndex === 0 ? changeData.name || "-" : "",
                    detailIndex === 0 ? changeData.uscc || "-" : "",
                    detailIndex === 0 ? demandText : "",
                    detailIndex === 0 ? changeData.content || "-" : "",
                    detailIndex === 0 ? changeData.contact || "-" : "",
                    detailIndex === 0 ? changeData.contactUs || "-" : "",
                    detailIndex === 0 ? createTime : "",
                    detailIndex === 0 ? currentTrackType : "",
                    detail.followUpName || "-",
                    followUpTime,
                    detail.followUpDsc || "-",
                    followUpStatus,
                    // detail.tdid || "-"
                ];

                dataList.push(row);
            });
        } else {
            /** 如果没有跟进记录，显示一行基础数据 */
            let row = [
                changeData.name || "-",
                changeData.uscc || "-",
                demandText,
                changeData.content || "-",
                changeData.contact || "-",
                changeData.contactUs || "-",
                createTime,
                currentTrackType,
                "-",
                "-",
                "-",
                "-",
                // "-"
            ];

            dataList.push(row);
        }
    });

    return dataList;
}


/**
 * 导出服务追踪信息（简化版，明确分隔）
 * @param name 企业名称
 * @param demand 需求类型
 * @param trackType 需求跟进状态
 * @returns 二维数组，明确分隔不同企业
 */
export async function outPutTrackSimpleData(name: string, demand: number, trackType: number) {
    /** 拼接查询条件 */
    let selectParam: any = {};
    if (name) {
        selectParam.name = { "$regex": `${name}` };
    }
    if (demand) {
        selectParam.demand = demand;
    }
    if (trackType) {
        selectParam.trackType = trackType;
    }

    /** 查询服务追踪数据 */
    let trackDbList = await trackData.findTrackListByParam(selectParam);

    /** 表头定义 */
    const titleList = [
        "企业名称",
        "统一社会信用代码",
        "需求类型",
        "需求内容",
        "联系人",
        "联系方式",
        "需求创建时间",
        "当前跟进状态",
        "跟进人",
        "跟进时间",
        "跟进详情",
        "跟进状态"
    ];

    let dataList = [titleList];

    /** 处理每个企业 */
    trackDbList.forEach((info, enterpriseIndex) => {
        /** 提取基础数据 */
        let changeData: any = extractData(splitResultConfig.trackListConfig, info, true);

        /** 枚举值转文本 */
        const demandText = verificationEnumTools.changeEnumValue(configEnum.VISITDEMAND, changeData.demand) || "未选择";
        const currentTrackType = verificationEnumTools.changeEnumValue(configEnum.TRACKTYPE, changeData.trackType) || "未选择";

        /** 时间格式化 */
        const createTime = changeData.createTime ? moment(changeData.createTime).format("YYYY-MM-DD") : "-";

        /** 添加分隔行（除了第一个企业） */
        if (enterpriseIndex > 0) {
            dataList.push(new Array(titleList.length).fill(""));
        }

        /** 处理跟进详情 */
        if (info.details && info.details.length > 0) {
            // 按跟进时间倒序排列，最新的在前
            const sortedDetails = [...info.details].sort((a, b) => 
                moment(b.followUpTime).valueOf() - moment(a.followUpTime).valueOf()
            );

            sortedDetails.forEach((detail) => {
                const followUpStatus = verificationEnumTools.changeEnumValue(configEnum.TRACKTYPE, detail.followUpStatus) || "未选择";
                const followUpTime = detail.followUpTime ? moment(detail.followUpTime).format("YYYY-MM-DD") : "-";
                
                /** 每行都显示完整信息 */
                let row = [
                    changeData.name || "-",
                    changeData.uscc || "-",
                    demandText,
                    changeData.content || "-",
                    changeData.contact || "-",
                    changeData.contactUs || "-",
                    createTime,
                    currentTrackType,
                    detail.followUpName || "-",
                    followUpTime,
                    detail.followUpDsc || "-",
                    followUpStatus
                ];

                dataList.push(row);
            });
        } else {
            /** 如果没有跟进记录 */
            let row = [
                changeData.name || "-",
                changeData.uscc || "-",
                demandText,
                changeData.content || "-",
                changeData.contact || "-",
                changeData.contactUs || "-",
                createTime,
                currentTrackType,
                "-",
                "-",
                "-",
                "-"
            ];

            dataList.push(row);
        }
    });

    return dataList;
}





