/**
 * 标签相关处理工具方法
 */

import { LABELUPDATEROAD } from "../config/enum/labelEnum";
import * as labelData from "../data/label";


/**
 * 标签去重函数
 * @param {Array} array - 需要去重的数组
 * @param {string|Function} [key] - 对象去重的键名或自定义函数
 * @returns {Array} 去重后的数组
 * 
 * 使用方法：
 * 1、labels = [1, 2, 2, 3, 4, 4, 5];
 * uniqueLabelList(labels) //直接数组去重
 * 
 * 2、labels = [ {id:"1", name:""}, {id:"2", name:""} ];
 * uniqueLabelList(labels, 'id') //根据id去重
 * uniqueLabelList(labels, labels => labels.name) //根据name去重
 */
export function uniqueLabelList(array, key?) {
    if (!Array.isArray(array)) {
        return [];
    }
    
    // 如果是基本类型数组
    if (!key) {
        return [...new Set(array)];
    }
    
    // 如果key是函数
    if (typeof key === 'function') {
        const seen = new Set();
        return array.filter(item => {
            const k = key(item);
            if (seen.has(k)) {
                return false;
            }
            seen.add(k);
            return true;
        });
    }
    
    // 如果key是字符串（对象数组）
    const seen = new Set();
    return array.filter(item => {
        const k = item[key];
        if (seen.has(k)) {
            return false;
        }
        seen.add(k);
        return true;
    });
}


/**
 * 获取年份标签ID
 * @param {Number} parkEntryTime 入驻时间戳
 * @returns {String} 年份标签ID
 */
function getYearLabelId(parkEntryTime) {
    let entryDate = new Date(parkEntryTime);
    let entryYear = entryDate.getFullYear();
    
    if (entryYear === 2023) {
        return "2023";
    } else if (entryYear === 2024) {
        return "2024";
    } else {
        return "2100"; // 其他年份
    }
}

/**
 * 检查标签是否已存在
 * @param {String} labelId 标签ID
 * @param {Array} enterpriseLabels 企业标签数组
 * @returns {Boolean} 是否已存在
 */
function labelExists(labelId, enterpriseLabels) {
    for (let i = 0; i < enterpriseLabels.length; i++) {
        if (enterpriseLabels[i].labelId === labelId) {
            return true;
        }
    }
    return false;
}

/**
 * 标签去重
 * @param {Array} labels 标签数组 [{labelId:""}]
 * @returns {Array} 去重后的标签数组
 */
function deduplicateLabels(labels) {
    let uniqueLabels = [];
    let labelIdSet = {};
    
    if (labels && labels.length > 0) {
        labels.forEach(label => {
            if (label && label.labelId && !labelIdSet[label.labelId]) {
                labelIdSet[label.labelId] = true;
                uniqueLabels.push(label);
            }
        });
    }
    
    return uniqueLabels;
}

/**
 * 根据企业数据生成标签数组
 * @param {Object} enterpriseData 企业数据
 * @param {Array} activeLabels 可用的标签列表（未停用）
 * @returns {Array} 处理后的标签数组
 */
export async function generateEnterpriseLabels(enterpriseData, activeLabels = null) {
    // 如果没有传入可用标签，则获取一次
    let availableLabels = activeLabels;
    if (!availableLabels) {
        availableLabels = await labelData.selectLabelList({
            state: false // 未停用的标签
        });
    }
    
    // 创建标签映射对象
    let activeLabelMap = {};
    
    if (availableLabels && availableLabels.length > 0) {
        availableLabels.forEach(label => {
            let labelIdStr = '';
            if (label.id) labelIdStr = label.id.toString();
            if (labelIdStr) {
                activeLabelMap[labelIdStr] = label;
            }
        });
    }
    
    // 构建企业标签数组
    let enterpriseLabels = [];
    
    // 处理直接传入的标签
    if (enterpriseData.labels && Array.isArray(enterpriseData.labels) && enterpriseData.labels.length > 0) {
        enterpriseData.labels.forEach(labelId => {
            if (labelId !== undefined && labelId !== null) {
                let labelIdStr = labelId.toString();
                let labelInfo = activeLabelMap[labelIdStr];
                
                if (labelInfo && labelInfo.state === false) {
                    enterpriseLabels.push({
                        labelId: labelIdStr,
                        state: true
                    });
                }
            }
        });
    }
    
    // 处理企业性质标签
    if (enterpriseData.enterpriseNature !== undefined && enterpriseData.enterpriseNature !== null) {
        let natureLabelId = enterpriseData.enterpriseNature.toString();
        let labelInfo = activeLabelMap[natureLabelId];
        
        if (labelInfo && labelInfo.state === false && !labelExists(natureLabelId, enterpriseLabels)) {
            enterpriseLabels.push({
                labelId: natureLabelId,
                state: true
            });
        }
    }
    
    // 处理所属物业标签
    if (enterpriseData.property !== undefined && enterpriseData.property !== null) {
        let propertyLabelId = enterpriseData.property.toString();
        let labelInfo = activeLabelMap[propertyLabelId];
        
        if (labelInfo && labelInfo.state === false && !labelExists(propertyLabelId, enterpriseLabels)) {
            enterpriseLabels.push({
                labelId: propertyLabelId,
                state: true
            });
        }
    }
    
    // 处理入驻园区时间标签
    if (enterpriseData.parkEntryTime !== undefined && enterpriseData.parkEntryTime !== null) {
        let yearLabelId = getYearLabelId(enterpriseData.parkEntryTime);
        let labelInfo = activeLabelMap[yearLabelId];
        
        if (labelInfo && labelInfo.state === false && !labelExists(yearLabelId, enterpriseLabels)) {
            enterpriseLabels.push({
                labelId: yearLabelId,
                state: true
            });
        }
    }
    
    // 去重处理
    return deduplicateLabels(enterpriseLabels);
}


/**
 * 获取企业标签（用于回显）
 * @param {Object} enterprise 企业数据
 * @returns {Array} 标签ID数组
 */
// export function getLabelIdsForDisplay(enterprise) {
//     if (!enterprise || !enterprise.labels || !Array.isArray(enterprise.labels)) {
//         return [];
//     }
    
//     let labelIds = [];
//     enterprise.labels.forEach(label => {
//         if (label && label.state === true) {
//             labelIds.push(label.labelId);
//         }
//     });
    
//     return labelIds;
// }

/**
 * 更新企业标签（封装函数，包含获取可用标签）
 * @param {Object} enterpriseData 企业数据
 * @returns {Promise<Array>} 更新后的标签数组
 */
export async function updateEnterpriseLabels(enterpriseData) {
    let activeLabels = await labelData.selectLabelList({
        state: false // 未停用的标签
    });
    return await generateEnterpriseLabels(enterpriseData, activeLabels);
}

