/**
 * 数据初始化
 * 底库数据
 */
import * as enterpriseData from "../data/enterprise";
import * as labelData from "../data/label";
import * as configEnum from "../config/enum";
import * as industryEnum from "../config/enum/industryEnum";
import * as labelEnum from "../config/enum/labelEnum";
import { createDataByParam, findEnterpriseByUscc } from "../data/enterprise";
import { formatAddress, getPwdMd5, randomId } from "./system";
import { TABLEID } from "../config/enum/dbEnum";
import moment from "moment";
let xlsx = require('node-xlsx');
let path = require('path');
let md5 = require("md5");

/**
 * 获取当个excel文件数据
 * @param filePath 
 * @returns 
 */
function getExcel(filePath) {
    let workSheetsFromFile = xlsx.parse(filePath);
    let sheetMap = {};
    let sheetList = [];

    for (let i = 0; i < workSheetsFromFile.length; i++) {
        let sheetInfo = workSheetsFromFile[i];
        sheetMap[sheetInfo.name] = sheetInfo.data;
        sheetList.push(sheetInfo);
    }

    return {sheetMap, sheetList}  
}

const IndustryConfigMap = {
    "农、林、牧、渔业" : "A",
    "采矿业" : "B",
    "制造业" : "C",
    "电力、热力、燃气及水生产和供应业" : "D",
    "建筑业" : "E",
    "批发和零售业" : "F",
    "交通运输、仓储和邮政业" : "G",
    "住宿和餐饮业" : "H",
    "信息传输、软件和信息技术服务业" : "I",
    "金融业" : "J",
    "房地产业" : "K",
    "租赁和商务服务业" : "L",
    "科学研究和技术服务业" : "M",
    "水利、环境和公共设施管理业" : "N",
    "居民服务、修理和其他服务业" : "O",
    "教育" : "P",
    "卫生和社会工作" : "Q",
    "文化、体育和娱乐业" : "R",
    "公共管理、社会保障和社会组织" : "S",
    "国际组织" : "T",
}

/**
 * 行业大类枚举映射
 */
const IndustryMajorEnumMap = {
    'A': industryEnum.A, 
    'B': industryEnum.B, 
    'C': industryEnum.C, 
    'D': industryEnum.D, 
    'E': industryEnum.E, 
    'F': industryEnum.F, 
    'G': industryEnum.G, 
    'H': industryEnum.H, 
    'I': industryEnum.I,
    'J': industryEnum.J, 
    'K': industryEnum.K, 
    'L': industryEnum.L, 
    'M': industryEnum.M, 
    'N': industryEnum.N, 
    'O': industryEnum.O, 
    'P': industryEnum.P, 
    'Q': industryEnum.Q, 
    'R': industryEnum.R,
    'S': industryEnum.S, 
    'T': industryEnum.T
};

/**
 * 企业性质映射
 */
const EnterpriseNatureMap = {
    '实体型': labelEnum.ENTERPRISENATURE.实体型,
    '虚拟型': labelEnum.ENTERPRISENATURE.虚拟型,
    '注册型': labelEnum.ENTERPRISENATURE.注册型
};

/**
 * 企业规模映射
 */
const EnterpriseSizeMap = {
    'S(小型)': configEnum.ENTERPRISESIZE.小型,
    'XS(微型)': configEnum.ENTERPRISESIZE.微型,
    'M(中型)': configEnum.ENTERPRISESIZE.中型,
    'L(大型)': configEnum.ENTERPRISESIZE.大型
};

/**
 * 物业映射
 */
const propertyMap = {
    '于田大厦': labelEnum.PROPERTY.于田大厦,
    '汽车创新港': labelEnum.PROPERTY.汽车创新港,
    '汽车城大厦': labelEnum.PROPERTY.汽车城大厦,
    '嘉亭荟': labelEnum.PROPERTY.嘉亭荟,
    '智驾园': labelEnum.PROPERTY.智驾园,
    '同济科技园': labelEnum.PROPERTY.同济科技园
};


/**
 * 处理行业信息
 * @param {string} industryCategoryGb 国标行业门类
 * @param {string} industryMajorGb 国标行业大类
 * @returns {{industry: Array<number>, industryCategoryGb: string}}
 */
function parseIndustry(industryCategoryGb, industryMajorGb) {
    if (!industryCategoryGb || !industryMajorGb) {
        return { industry: [], industryCategoryGb: '' };
    }
    
    // 获取行业门类代码（字母）
    let industryCategoryCode = IndustryConfigMap[industryCategoryGb] || '';
    
    // 获取行业大类枚举值
    let industryMajorValue = null;
    
    // 获取对应的大类枚举
    let majorEnum = IndustryMajorEnumMap[industryCategoryCode];
    if (majorEnum) {
        // 在大类枚举中查找对应的值
        let entry = Object.entries(majorEnum).find(([key, value]) => 
            key === industryMajorGb || String(value) === industryMajorGb
        );
        
        if (entry) {
            industryMajorValue = entry[1];
        }
    }
    
    // 返回对象，包含两个字段
    return {
        industry: industryMajorValue ? [industryMajorValue] : [],
        industryCategoryGb: industryCategoryCode
    };
}


/**
 * 处理日期字符串
 * @param {string} dateStr 
 * @returns {number|null}
 */
function parseDateString(dateStr) {
    if (!dateStr || dateStr === '-') return null;
    
    try {
        // 处理"2024年"格式的日期
        if (typeof dateStr === 'string' && dateStr.includes('年')) {
            // 提取年份数字，例如："2024年" -> 2024
            const yearMatch = dateStr.match(/(\d{4})年/);
            if (yearMatch && yearMatch[1]) {
                const year = parseInt(yearMatch[1], 10);
                // 返回该年份1月1日的时间戳
                return new Date(`${year}-01-01`).getTime();
            }
        }
        
        // 处理Excel日期格式（如：2020-07-16）
        if (typeof dateStr === 'string') {
            // 尝试多种日期格式
            const dateFormats = [
                'YYYY-MM-DD',
                'YYYY/MM/DD',
                'YYYY年MM月DD日',
                'YYYY-MM',
                'YYYY年'
            ];
            
            for (const format of dateFormats) {
                const parsedDate = moment(dateStr, format, true);
                if (parsedDate.isValid()) {
                    return parsedDate.valueOf();
                }
            }
            
            // 如果上述格式都不匹配，尝试原生Date解析
            let date = new Date(dateStr);
            return isNaN(date.getTime()) ? null : date.getTime();
        }
        
        // 处理Excel数字日期
        if (typeof dateStr === 'number') {
            let utc = new Date();
            return new Date((dateStr - 25569) * 86400 * 1000 + utc.getTimezoneOffset() * 1000 * 60).getTime();
        }
        
        return null;
    } catch (error) {
        console.error('日期解析错误:', dateStr, error);
        return null;
    }
}


/**
 * 处理注册资本和实缴资本
 * @param {string} capitalStr 
 * @returns {string}
 */
function parseCapital(capitalStr) {
    if (!capitalStr || capitalStr === '-') return '-';
    
    // 提取数字部分
    let match = capitalStr.match(/([\d\.]+)/);
    return match ? match[1] + '万元' : capitalStr;
}


/**
 * 处理参保人数
 * @param {string|number} peopleStr 
 * @returns {number}
 */
function parsePeopleCount(peopleStr) {
    if (!peopleStr || peopleStr === '-' || peopleStr === '0') return 0;
    
    if (typeof peopleStr === 'number') {
        return peopleStr;
    }
    
    let num = parseInt(peopleStr);
    return isNaN(num) ? 0 : num;
}


/**
 * 处理租赁面积
 * @param {string|number} areaStr 
 * @returns {number}
 */
function parseLeasedArea(areaStr) {
    if (!areaStr || areaStr === '-' || areaStr === '0') return 0;
    
    if (typeof areaStr === 'number') {
        return areaStr;
    }
    
    let num = parseFloat(areaStr);
    return isNaN(num) ? 0 : num;
}


/**
 * 处理数字字段
 */
function parseNumber(value, defaultValue = 0) {
    if (value === undefined || value === null || value === '' || value === '-') {
        return defaultValue;
    }
    
    if (typeof value === 'number') {
        return value;
    }
    
    const num = parseFloat(value);
    return isNaN(num) ? defaultValue : num;
}


/**
 * 处理企业标签
 * @param {Object} rowData Excel行数据
 * @returns {Array} 标签数组
 */
function parseLabels(rowData) {
    let labels = [];
    let currentTime = new Date().valueOf();
    
    // 1. 基础标签 (BASELABEL)
    if (rowData['area3_73'] === '是') {
        labels.push({
            labelId: labelEnum.BASELABEL["3.73平方公里"],
            labelType: labelEnum.LABELTYPE.基础标签,
            state: true
        });
    }
    
    if (rowData['area100'] === '是') {
        labels.push({
            labelId: labelEnum.BASELABEL["100平方公里"],
            labelType: labelEnum.LABELTYPE.基础标签,
            state: true
        });
    }
    
    if (rowData['isSoftwareRelated'] === '是') {
        labels.push({
            labelId: labelEnum.BASELABEL["软件关联入驻企业"],
            labelType: labelEnum.LABELTYPE.基础标签,
            state: true
        });
    }
    
    if (rowData['isSmallAccount'] === '是') {
        labels.push({
            labelId: labelEnum.BASELABEL["小台账企业"],
            labelType: labelEnum.LABELTYPE.基础标签,
            state: true
        });
    }
    
    if (rowData['isKeyBusiness'] === '是') {
        labels.push({
            labelId: labelEnum.BASELABEL["重点稳商企业"],
            labelType: labelEnum.LABELTYPE.基础标签,
            state: true
        });
    }
    
    // 2. 入驻年份标签 (ENTERPRISESYSTEMLABEL)
    let entryYear = rowData['parkEntryTime'];
    if (entryYear && entryYear != "-") {
        let yearValue;
        switch (entryYear.toString()) {
            case '2023年':
                yearValue = labelEnum.ENTERPRISESYSTEMLABEL["2023年"];
                break;
            case '2024年':
                yearValue = labelEnum.ENTERPRISESYSTEMLABEL["2024年"];
                break;
            case '2025年':
                yearValue = labelEnum.ENTERPRISESYSTEMLABEL["2025年"];
                break;
            case '2026年':
                yearValue = labelEnum.ENTERPRISESYSTEMLABEL["2026年"];
                break;
            case '2027年':
                yearValue = labelEnum.ENTERPRISESYSTEMLABEL["2027年"];
                break;
            case '2028年':
                yearValue = labelEnum.ENTERPRISESYSTEMLABEL["2028年"];
                break;
            // default:
            //     yearValue = labelEnum.ENTERPRISESYSTEMLABEL["其他年份"];
        }
        
        labels.push({
            labelId: yearValue,
            labelType: labelEnum.LABELTYPE.入驻年份,
            state: true
        });
    }

    // 3. 所属物业（PROPERTY）
    let property = rowData['property'];
    if (property && property != "-") {
        let propertyValue;
        switch (property) {
            case "于田大厦":
                propertyValue = labelEnum.PROPERTY.于田大厦;
            break;
            case "同济科技园":
                propertyValue = labelEnum.PROPERTY.同济科技园;
            break;
            case "嘉亭荟":
                propertyValue = labelEnum.PROPERTY.嘉亭荟;
            break;
            case "智驾园":
                propertyValue = labelEnum.PROPERTY.智驾园;
            break;
            case "汽车创新港":
                propertyValue = labelEnum.PROPERTY.汽车创新港;
            break;
            case "汽车城大厦":
                propertyValue = labelEnum.PROPERTY.汽车城大厦;
            break;
        }

        labels.push({
            labelId: propertyValue,
            labelType: labelEnum.LABELTYPE.所属物业,
            state: true
        });
    }

    // 4. 企业性质 enterpriseNature: EnterpriseNatureMap[rowData.enterpriseNature] || labelEnum.ENTERPRISENATURE.实体型
    let enterpriseNature = rowData['enterpriseNature'];
    if (enterpriseNature && enterpriseNature != "-") {
        let enterpriseNatureValue;
        switch (enterpriseNature) {
            case "实体型":
                enterpriseNatureValue = labelEnum.ENTERPRISENATURE.实体型;
            break;
            case "注册型":
                enterpriseNatureValue = labelEnum.ENTERPRISENATURE.注册型;
            break;
            case "虚拟型":
                enterpriseNatureValue = labelEnum.ENTERPRISENATURE.虚拟型;
            break;
        }

        labels.push({
            labelId: enterpriseNatureValue,
            labelType: labelEnum.LABELTYPE.企业性质,
            state: true
        });
    }
    
    return labels;
}


/**
 * 初始化原始数据  
 */
export async function initBasicData() {
    await initLabel();
    await initEnterpriseFromExcel();
}


/**
 * 初始化企业标签库
 */
async function initLabel() {
    try {
        // 检查是否已有数据
        let count = await labelData.selectLabelCount({});
        if (count > 10) {
            console.log("*** 标签数据 无需初始化");
            return;
        }

        let addList = [];
        // 1. 基础标签
        let baseLabel = convertEnumToObject(labelEnum.BASELABEL);
        for (let key in baseLabel) {
            addList.push({
                id: baseLabel[key],
                labelName: key,
                ctMs: new Date().valueOf(),
                goal: labelEnum.LABELGOAL.企业,
                labelType: labelEnum.LABELTYPE.基础标签,
                state: false,
            });
        }
        // 2. 入驻年份
        let yearLabel = convertEnumToObject(labelEnum.YEARLABEL);
        for (let key in yearLabel) {
            addList.push({
                id: yearLabel[key],
                labelName: key,
                ctMs: new Date().valueOf(),
                goal: labelEnum.LABELGOAL.企业,
                labelType: labelEnum.LABELTYPE.入驻年份,
                state: false,
            });
        }
        // 3. 企业性质
        let enterprisenatureLabel = convertEnumToObject(labelEnum.ENTERPRISENATURE);
        for (let key in enterprisenatureLabel) {
            addList.push({
                id: enterprisenatureLabel[key],
                labelName: key,
                ctMs: new Date().valueOf(),
                goal: labelEnum.LABELGOAL.企业,
                labelType: labelEnum.LABELTYPE.企业性质,
                state: false,
            });
        }
        // 4. 所属物业
        let propertyLabel = convertEnumToObject(labelEnum.PROPERTY);
        for (let key in propertyLabel) {
            addList.push({
                id: propertyLabel[key],
                labelName: key,
                ctMs: new Date().valueOf(),
                goal: labelEnum.LABELGOAL.企业,
                labelType: labelEnum.LABELTYPE.所属物业,
                state: false,
            });
        }

        //导入数据库
        console.log(`准备导入 ${addList.length} 条标签数据`);
            
        for (let i = 0; i < addList.length; i ++) {
            let item = addList[i];
            // 检查是否已存在
            let existing = await labelData.selectLabelOne({id:item.id});
            if (existing) {
                console.log(`标签 ${item.labelName} (${item.id}) 已存在，跳过`);
                continue;
            }
            
            // 创建新企业
            await labelData.createOneLabel(item);
            console.log(`创建标签: ${item.labelName}`);
        }
        
        console.log("---> 标签数据 初始化成功");

    } catch (error) {
        console.error("初始化标签数据失败:", error);
        throw error;
    }
}


/**
 * 通用枚举转对象函数
 */
function convertEnumToObject(enumObj) {
    const result = {};
    
    for (const key in enumObj) {
        // 跳过数字键
        if (isNaN(Number(key))) {
            const value = enumObj[key];
            result[key] = value;
        }
    }
    
    return result;
}


/**
 * 初始化企业数据
 */
async function initEnterpriseFromExcel() {
    try {
        // 检查是否已有数据
        let count = await enterpriseData.findEnterpriseCount({});
        if (count > 10) {
            console.log("*** 企业数据 无需初始化");
            return;
        }
        
        // 读取Excel文件
        let {sheetMap} = getExcel(path.join(__dirname.substring(0,__dirname.indexOf("out")), "res", '企业标签数据_完整更新V2.xlsx' ));
        
        // 获取工作表数据（根据Excel中的工作表名称）
        let sheetName = '企业数据';
        let dataList = sheetMap[sheetName];
        
        if (!dataList || dataList.length === 0) {
            console.log(`工作表 ${sheetName} 无数据`);
            return;
        }
        
        console.log(`读取到 ${dataList.length} 行数据`);
        
        // 表头映射（根据Excel列顺序）
        // Excel列顺序：A B C D E F G H I J K L M N O P Q R S T U V W X Y Z AA AB AC AD AE AF AG AH AI AJ AK AL AM AN AO AP AQ AR AS AT AU AV AW
        // 对应表头：企业名称、登记状态、成立日期、注册资本、企业简介、法定代表人、实缴资本、统一社会信用代码、企业地址、所属省份...
        // 注意：需要根据实际的Excel列顺序调整
        let titleMap = {
            "0": "name",                   // A: 企业名称
            "1": "RAS",                    // B: 登记状态
            "2": "logonTime",              // C: 成立日期
            "3": "zhuceziben",             // D: 注册资本
            "4": "jianJie",                // E: 企业简介
            "5": "legalPerson",            // F: 法定代表人
            "6": "shijiaoziben",           // G: 实缴资本
            "7": "uscc",                   // H: 统一社会信用代码
            "8": "logonAdd",               // I: 企业地址
            "9": "province",               // J: 所属省份
            "10": "city",                  // K: 所属城市
            "11": "district",              // L: 所属区县
            "12": "dianHua",               // M: 电话
            "13": "gengDuoDianHua",        // N: 更多电话
            "14": "mail",                  // O: 邮箱
            "15": "moreMail",              // P: 更多邮箱
            "16": "enterpriseType",        // Q: 企业（机构）类型
            "17":"naShuiRenShiBieHao",     // R: 纳税人识别号
            "18": "zhuCeHao",              // S: 注册号
            "19": "zuZhiJiGouDaiMa",       // T: 组织机构代码
            "20": "canBaoRenShu",          // U: 参保人数
            "21": "canBaoRenShuNianBao",   // V: 参保人数所属年报
            "22": "yingYeQiXian",          // W: 营业期限
            "23": "industryCategoryGb",    // X: 国标行业门类
            "24": "industryMajorGb",       // Y: 国标行业大类
            "25": "industryMediumGb",      // Z: 国标行业中类
            "26": "industryMinorGb",       // AA: 国标行业小类
            "27": "industryCategoryQcc",   // AB: 企查查行业门类
            "28": "industryMajorQcc",      // AC: 企查查行业大类
            "29": "industryMediumQcc",     // AD: 企查查行业中类
            "30": "industryMinorQcc",      // AE: 企查查行业小类
            "31": "qiYeGuiMo",             // AF: 企业规模
            "32": "cengYongMing",          // AG: 曾用名
            "33": "yingWenMing",           // AH: 英文名
            "34": "guanWang",              // AI: 官网
            "35": "tongXinDiZhi",          // AJ: 通信地址
            "36": "jingYingFanWei",        // AK: 经营范围
            "37": "dengJiJiGuan",          // AL: 登记机关
            "38": "naShuiRenZiZhi",        // AM: 纳税人资质
            "39": "zuiXinNianBaoNianFen",  // AN: 最新年报年份
            "40": "leasedArea",            // AO: 租赁面积（㎡）
            "41": "enterpriseNature",      // AP: 企业性质
            "42": "property",              // AQ: 所属物业
            "43": "parkEntryTime",         // AR: 入驻年份（用于标签）
            "44": "area3_73",              // AS: 3.73平方公里（标签字段）
            "45": "area100",               // AT: 100平方公里（标签字段）
            "46": "isSoftwareRelated",     // AU: 软件关联入驻企业（标签字段）
            "47": "isSmallAccount",        // AV: 小台账企业（标签字段）
            "48": "isKeyBusiness"          // AW: 重点稳商企业（标签字段）
        };
        
        let titleIndexList = Object.keys(titleMap);
        let currentTime = new Date().valueOf();
        let addList = [];
        
        // 提取表头行（第0行）
        let headerRow = dataList[0] || [];

        // 处理每一行数据（从第1行开始）
        for (let rowIndex = 1; rowIndex < dataList.length; rowIndex++) {
            let subList = dataList[rowIndex];
            
            // 构建行数据对象（用于标签解析）
            let rowData:any = {};
            titleIndexList.forEach((colIndex, idx) => {
                let key = titleMap[colIndex];
                let value = subList[parseInt(colIndex)];
                rowData[key] = value;
            });

            //处理行业领域
            let industryData = {industry:[], industryCategoryGb:""}
            if (rowData.industryCategoryGb != "-") {
                industryData.industryCategoryGb = industryEnum.INDUSTRY[rowData.industryCategoryGb];
            }
            if (rowData.industryCategoryGb != "-" && rowData.industryMajorGb != "-") {
                industryData.industry = [industryEnum.ALLINDUSTRY[rowData.industryMajorGb]];
            }

            let labelList = parseLabels(rowData); //解析标签
            let logonAdd = formatAddress(rowData.logonAdd); //解析注册地址
            
            // 创建企业数据对象
            let addDataInfo:any = {
                isInPut: true,
                createTime: currentTime,
                labels: labelList,
                RAS: rowData.RAS === '存续' ? configEnum.RAS.续存 : configEnum.RAS.注销,
                logOffMS: null, // 注销时间，Excel中暂无此字段
                legalPerson: rowData.legalPerson || '',
                zhuceziben: parseCapital(rowData.zhuceziben),
                shijiaoziben: parseCapital(rowData.shijiaoziben),
                dianHua: rowData.dianHua || '',
                gengDuoDianHua: rowData.gengDuoDianHua || '',
                mail: rowData.mail || '-',
                moreMail: rowData.moreMail || '-',
                enterpriseType: rowData.enterpriseType || '',
                zhuCeHao: rowData.zhuCeHao || '',
                zuZhiJiGouDaiMa: rowData.zuZhiJiGouDaiMa || '',
                canBaoRenShu: parseNumber(rowData.canBaoRenShu, 0),
                canBaoRenShuNianBao: parseNumber(rowData.canBaoRenShuNianBao, 0),
                yingYeQiXian: rowData.yingYeQiXian || '',
                qiYeGuiMo: EnterpriseSizeMap[rowData.qiYeGuiMo] || configEnum.ENTERPRISESIZE.未知,
                guanWang: rowData.guanWang || '',
                tongXinDiZhi: rowData.tongXinDiZhi || '',
                jianJie: rowData.jianJie || '',
                dengJiJiGuan: rowData.dengJiJiGuan || '',
                naShuiRenZiZhi: rowData.naShuiRenZiZhi === '一般纳税人' ? configEnum.NASHUIRENZIZHI.小规模纳税人 : 
                               rowData.naShuiRenZiZhi === '小规模纳税人' ? configEnum.NASHUIRENZIZHI.一般纳税人 : 
                               rowData.naShuiRenZiZhi === '非纳税人' ? configEnum.NASHUIRENZIZHI.增值税一般纳税人 : configEnum.NASHUIRENZIZHI.未知,
                zuiXinNianBaoNianFen: parseNumber(rowData.zuiXinNianBaoNianFen, 0),
                jingYingFanWei: rowData.jingYingFanWei || '',
                industryCategoryGb: industryData.industryCategoryGb || '',
                industryMajorGb: rowData.industryMajorGb || '',
                industryMediumGb: rowData.industryMediumGb || '',
                industryMinorGb: rowData.industryMinorGb || '',
                industryCategoryQcc: rowData.industryCategoryQcc || '',
                industryMajorQcc: rowData.industryMajorQcc || '',
                industryMediumQcc: rowData.industryMediumQcc || '',
                industryMinorQcc: rowData.industryMinorQcc || '',
                
                // 主字段
                name: rowData.name || '',
                uscc: rowData.uscc || '',
                isSettled: rowData.RAS === '存续' ? 1 : 0,
                // industry: industryData.industry,
                logonAddress: logonAdd || '',
                logonTime: parseDateString(rowData.logonTime),
                parkEntryTime: parseDateString(rowData.parkEntryTime),
                leasedArea: parseNumber(rowData.leasedArea, 0),
                // enterpriseNature: EnterpriseNatureMap[rowData.enterpriseNature] || labelEnum.ENTERPRISENATURE.实体型,
                enterpriseNature: EnterpriseNatureMap[rowData.enterpriseNature] || null,
                property: propertyMap[rowData.property] || null,

                //用户相关
                pwd:getPwdMd5(rowData.uscc, md5(rowData.uscc.slice(rowData.uscc.length-6))),
                firstLoginIsChangePwd:false,
            };
            
            if (industryData.industry) addDataInfo.industry = industryData.industry;

            // 验证必填字段
            if (!addDataInfo.name || !addDataInfo.uscc) {
                console.log(`第 ${rowIndex + 1} 行数据缺少企业名称或统一信用代码，跳过`);
                continue;
            }
            
            addList.push(addDataInfo);
        }
        
        console.log(`准备导入 ${addList.length} 条企业数据`);
        
        // 分批导入
        let batchSize = 50;
        for (let i = 0; i < addList.length; i += batchSize) {
            let batch = addList.slice(i, i + batchSize);
            await createEnterpriseToList(batch);
            console.log(`已导入 ${Math.min(i + batchSize, addList.length)}/${addList.length} 条数据`);
            
            // 每批导入后暂停一下
            if (i + batchSize < addList.length) {
                await new Promise(resolve => setTimeout(resolve, 1000));
            }
        }
        
        console.log("---> 企业数据 初始化成功");
        
    } catch (error) {
        console.error("初始化企业数据失败:", error);
        throw error;
    }
}

/**
 * 批量创建企业数据
 * @param {Array} enterpriseList 
 */
async function createEnterpriseToList(enterpriseList) {
    // 这里调用你的数据库操作函数
    for (let enterprise of enterpriseList) {
        try {
            // 检查是否已存在
            let existing = await findEnterpriseByUscc(enterprise.uscc);
            if (existing) {
                console.log(`企业 ${enterprise.name} (${enterprise.uscc}) 已存在，跳过`);
                continue;
            }
            
            // 创建新企业
            await createDataByParam(enterprise);
            console.log(`创建企业: ${enterprise.name}`);
            
        } catch (error) {
            console.error(`创建企业 ${enterprise.name} 失败:`, error);
        }
    }
}






