
const { parseString } = require('xml2js');
import * as iconv from 'iconv-lite';

/**
 * 解析用户信息XML响应
 */
export async function parseUserNameInfoXml(xmlData: string): Promise<any> {
    return new Promise((resolve, reject) => {
        // 先统一转换编码
        const utf8XmlData = convertEncoding(xmlData, 'gbk');
        
        parseString(utf8XmlData, { 
            explicitArray: false, 
            trim: true 
        }, (err, result) => {
            if (err) {
                console.error('XML解析错误:', err);
                return resolve(null);
            }
            
            try {
                // 解析XML结构并提取用户信息
                const response = result.methodResponse;
                if (!response || !response.params || !response.params.param) {
                    console.error('无效的XML响应结构');
                    return resolve(null);
                }
                
                // 提取用户信息数组
                const params = Array.isArray(response.params.param) ? 
                    response.params.param : [response.params.param];
                
                if (params.length > 0 && params[0].value && params[0].value.array) {
                    const data = params[0].value.array.data;
                    if (data && data.value && data.value.length > 0) {
                        // 提取第一个用户信息
                        const userValue = data.value[0];
                        if (userValue.struct) {
                            const member = userValue.struct.member;
                            const userInfo: any = {};
                            
                            // 处理成员字段
                            (Array.isArray(member) ? member : [member]).forEach((m: any) => {
                                if (m.name && m.value) {
                                    const key = m.name;
                                    const value = m.value.string || m.value.int || m.value.double || '';
                                    userInfo[key] = value; // 不再需要单独转换编码
                                }
                            });
                            
                            console.log('成功提取用户信息:', userInfo);
                            return resolve(userInfo);
                        }
                    }
                }
                
                console.error('未找到用户信息在响应中');
                resolve(null);
            } catch (parseError) {
                console.error('响应解析错误:', parseError);
                resolve(null);
            }
        });
    });
}


/**
 * 解析用户权限信息XML响应
 */
export async function parseUserInfoXml(xmlData: string): Promise<any> {
    return new Promise((resolve, reject) => {
        // 先统一转换编码
        const utf8XmlData = convertEncoding(xmlData, 'gbk');
        
        parseString(utf8XmlData, { 
            explicitArray: false, 
            trim: true,
            // 添加值处理器来处理不同类型的数据
            valueProcessors: [
                (value: string, name: string) => {
                    // 处理整型数据
                    if (name === 'int') {
                        return parseInt(value, 10);
                    }
                    // 处理字符串数据（不再需要编码转换）
                    if (name === 'string') {
                        return value;
                    }
                    return value;
                }
            ]
        }, (err, result) => {
            if (err) {
                console.error('XML解析错误:', err);
                return resolve(null);
            }
            
            try {
                // 解析XML结构
                const response = result.methodResponse;
                if (!response || !response.params || !response.params.param) {
                    console.error('无效的XML响应结构');
                    return resolve(null);
                }
                
                // 提取参数值
                const param = response.params.param;
                const value = param.value;
                
                if (value && value.array && value.array.data) {
                    const data = value.array.data;
                    
                    // 检查是否有值
                    if (data.value && data.value.struct) {
                        const struct = data.value.struct;
                        
                        // 提取成员信息
                        if (struct.member) {
                            const members = Array.isArray(struct.member) ? 
                                struct.member : [struct.member];
                            
                            // 构建用户信息对象
                            const userInfo: any = {};
                            
                            members.forEach((member: any) => {
                                if (member.name && member.value) {
                                    const key = member.name;
                                    let value:any = '';
                                    
                                    // 根据值类型提取数据
                                    if (member.value.string) {
                                        value = member.value.string; // 不再需要转换编码
                                    } else if (member.value.int) {
                                        value = parseInt(member.value.int, 10);
                                    } else if (member.value.double) {
                                        value = parseFloat(member.value.double);
                                    } else if (member.value.boolean) {
                                        value = member.value.boolean === '1' || 
                                               member.value.boolean === 'true';
                                    } else {
                                        // 尝试直接获取值
                                        value = member.value;
                                    }
                                    
                                    userInfo[key] = value;
                                }
                            });
                            
                            console.log('成功提取用户信息:', userInfo);
                            return resolve(userInfo);
                        }
                    }
                }
                
                console.error('未找到用户信息在响应中');
                resolve(null);
            } catch (parseError) {
                console.error('响应解析错误:', parseError);
                resolve(null);
            }
        });
    });
}


/**
 * 更简化的解析方法（针对您提供的XML结构）
 */
export async function parseUserInfoXmlSimple(xmlData: string): Promise<any> {
    return new Promise((resolve, reject) => {
        // 先统一转换编码
        const utf8XmlData = convertEncoding(xmlData, 'gbk');
        
        parseString(utf8XmlData, { 
            explicitArray: false, 
            trim: true
        }, (err, result) => {
            if (err) {
                console.error('XML解析错误:', err);
                return resolve(null);
            }
            
            try {
                // 简化解析逻辑
                const userInfo: any = {};
                
                // 提取所有成员
                const members = result?.methodResponse?.params?.param?.value?.array?.data?.value?.struct?.member;
                
                if (!members) {
                    console.error('未找到用户信息在响应中');
                    return resolve(null);
                }
                
                // 处理单个或多个成员
                const memberList = Array.isArray(members) ? members : [members];
                
                memberList.forEach((member: any) => {
                    if (member.name && member.value) {
                        const key = member.name;
                        let value: any;
                        
                        // 根据值类型处理
                        if (member.value.string !== undefined) {
                            value = member.value.string; // 不再需要转换编码
                        } else if (member.value.int !== undefined) {
                            value = parseInt(member.value.int, 10);
                        } else if (member.value.double !== undefined) {
                            value = parseFloat(member.value.double);
                        } else if (member.value.boolean !== undefined) {
                            value = member.value.boolean === '1' || 
                                   member.value.boolean === 'true';
                        } else {
                            value = member.value;
                        }
                        
                        userInfo[key] = value;
                    }
                });
                
                console.log('成功提取用户信息:', userInfo);
                resolve(userInfo);
            } catch (parseError) {
                console.error('响应解析错误:', parseError);
                resolve(null);
            }
        });
    });
}


/**
 * 统一的编码转换函数 (GBK -> UTF-8)
 */
// export function convertEncoding(text: any, from: string = 'gbk'): string {
//     if (!text) return '';
    
//     try {
//         if (Buffer.isBuffer(text)) {
//             return iconv.decode(text, from);
//         }
        
//         if (typeof text === 'string') {
//             // 如果是字符串，先转换为Buffer再解码
//             const buffer = Buffer.from(text, 'binary');
//             return iconv.decode(buffer, from);
//         }
        
//         return String(text);
//     } catch (error) {
//         console.warn('编码转换失败:', error);
//         return typeof text === 'string' ? text : String(text);
//     }
// }


/**
 * 增强的编码转换函数
 */
export function convertEncoding(data: any, from: string = 'gbk'): string {
    if (!data) return '';
    
    try {
        if (Buffer.isBuffer(data)) {
            // 直接解码Buffer
            return iconv.decode(data, from);
        }
        
        if (typeof data === 'string') {
            // 检查是否已经是正确编码
            if (!hasGarbledCharacters(data)) {
                return data;
            }
            
            // 尝试修复可能的双重编码
            const fixed = fixDoubleEncoding(data);
            if (!hasGarbledCharacters(fixed)) {
                return fixed;
            }
            
            // 如果修复失败，尝试将字符串转换为Buffer再解码
            const buffer = Buffer.from(data, 'binary');
            return iconv.decode(buffer, from);
        }
        
        return String(data);
    } catch (error) {
        console.warn('增强编码转换失败:', error);
        return typeof data === 'string' ? data : String(data);
    }
}


/**
 * 检测字符串是否包含乱码字符
 */
export function hasGarbledCharacters(str: string): boolean {
    if (!str || typeof str !== 'string') return false;
    
    // 检查常见的乱码模式
    const garbledPatterns = [
        /\\u[0-9a-fA-F]{4}/g, // Unicode转义序列
        /�/g, // Unicode替换字符
        /[^\x00-\x7F\u4E00-\u9FFF]/g // 非ASCII和非中文字符
    ];
    
    return garbledPatterns.some(pattern => pattern.test(str));
}


/**
 * 修复可能的双重编码问题
 */
export function fixDoubleEncoding(str: string): string {
    if (!str || typeof str !== 'string') return str || '';
    
    try {
        // 尝试将字符串视为UTF-8编码的GBK数据
        // 1. 先将字符串转换为Buffer（假设它是UTF-8）
        const utf8Buffer = Buffer.from(str, 'utf8');
        // 2. 然后将这个Buffer作为GBK解码
        return iconv.decode(utf8Buffer, 'gbk');
    } catch (error) {
        console.warn('双重编码修复失败:', error);
        return str;
    }
}


/**
 * 解析用户身份代码
 * 1：是
 * 0：否
 * @param userTypeString 
 * @returns 
 */
export function parseUserTypeEnhanced(userTypeString) {
    // 配置项：身份类型定义
    const ROLE_CONFIG = [
      { key: 'generalUser', label: '一般注册用户' },
      { key: 'administrator', label: '行政管理人员' },
      { key: 'staff', label: '教职员工' },
      { key: 'student', label: '学生' },
      { key: 'parent', label: '家长' }
    ];
  
    // 验证输入
    if (typeof userTypeString !== 'string') {
      throw new Error('输入必须是一个字符串');
    }
    
    if (userTypeString.length !== 5) {
      throw new Error('字符串长度必须为5位');
    }
    
    if (!/^[01]+$/.test(userTypeString)) {
      throw new Error('字符串只能包含0和1');
    }
  
    // 构建结果对象
    const result = {};
    
    ROLE_CONFIG.forEach((role, index) => {
      result[role.label] = parseInt(userTypeString[index], 10);
    });
  
    return result;
}









