import { Request, Response } from 'express';
import moment from 'moment';
import { BizError } from "../util/bizError";
import { TABLENAME, TABLEID } from "../config/dbEnum";
import { selectDataListByParam, selectDataToTableAssociation, selectDataToTableAssociationToPage, selectDataWithCustomOrder, selectOneDataByParam, selectPaginatedDataWithOrder } from "../data/findData";
import { STATE } from "../config/enum";
import { updateManyData } from "../data/updateData";
import { getMySqlMs, randomId } from "../tools/systemTools";
import { addData } from "../data/addData";
import { ERRORENUM } from "../config/errorEnum";


/**
 * 检查学生是否有已提交的答题记录
 * @param student_id 学生ID
 * @returns 如果有已提交的记录返回false，否则返回true
 */
export async function checkStudentHasAnswered(student_id: string) {
    if (!student_id) {
        // 如果没有学生ID，认为可以答题（返回true）
        return { checkanswered: true };
    }

    // 查找该学生已完成的答题记录
    const existingRecord = await selectOneDataByParam(
        TABLENAME.答题记录表,
        { 
            student_id: student_id, 
            answer_status: STATE.是 // 只检查已完成的记录
        },
        ["record_id"]
    );

    // 根据注释要求：有已提交记录返回false，没有返回true
    const hasAnswered = !!(existingRecord.data && Object.keys(existingRecord.data).length > 0);
    
    // 如果有记录，说明已经答过题，返回false
    // 如果没有记录，说明可以答题，返回true
    return { checkanswered: !hasAnswered };
}


/**
 * 获取第一条服务时间配置
 * @returns 第一条服务时间配置信息
 */
export async function getFirstOpeningTimeConfig() {
    // 获取所有服务时间配置，按更新时间倒序排列（获取最新的配置）
    const configs = await selectDataWithCustomOrder(
        TABLENAME.服务开启时间表,
        {},
        ["otId", "startTime", "endTime", "isOpen", "updated_by", "updated_at"],
        [["updated_at", "DESC"]] // 按更新时间倒序，获取最新的配置
    );
    
    // 如果没有配置，返回默认值
    if (!configs.data || configs.data.length === 0) {
        return {
            startTime: null,
            endTime: null,
            isOpen: 0,
            isEnter: false, // 无配置时不允许进入
            hasConfig: false,
            message: '未配置服务时间'
        };
    }
    
    // 获取第一条配置（最新的配置）
    const firstConfig = configs.data[0];
    
    // 计算 isEnter 字段
    let isEnter = false;
    if (firstConfig.isOpen === 1) {
        const now = new Date();
        const startTime = new Date(firstConfig.startTime);
        const endTime = new Date(firstConfig.endTime);
        
        // 判断当前时间是否在服务时间范围内
        isEnter = now >= startTime && now <= endTime;
    }
    // 如果 isOpen 是 0，isEnter 保持 false
    
    return {
        startTime: firstConfig.startTime,
        endTime: firstConfig.endTime,
        isOpen: firstConfig.isOpen,
        isEnter, // 是否允许进入
        hasConfig: true,
        otId: firstConfig.otId
    };
}


/**
 * 获取当前服务时间配置
 */
export async function getOpeningTimeConfig() {
    const config = await selectDataListByParam(
        TABLENAME.服务开启时间表,
        {},
        ["otId", "startTime", "endTime", "isOpen", "updated_by", "updated_at"]
    );
    
    return config.data || [];
}


/**
 * 创建或更新服务时间配置
 * @param otId 配置ID（可选，为空时创建新配置）
 * @param startTime 开始时间
 * @param endTime 结束时间
 * @param isOpen 是否启用
 * @param updated_by 修改人
 */
export async function setOpeningTimeConfig(startTime: Date, endTime: Date, isOpen: number, updated_by: string, otId?) {
    // 参数验证
    if (!startTime || !endTime) {
        throw new BizError(ERRORENUM.参数错误, '开始时间和结束时间不能为空');
    }
    
    if (startTime >= endTime) {
        throw new BizError(ERRORENUM.参数错误, '开始时间必须早于结束时间');
    }
    
    if (isOpen !== 0 && isOpen !== 1) {
        throw new BizError(ERRORENUM.参数错误, 'isOpen参数必须为0或1');
    }
    
    if (!updated_by) {
        throw new BizError(ERRORENUM.参数错误, '修改人不能为空');
    }
    
    let configId = otId;
    let isUpdate = false;
    
    // 如果没有提供otId，检查是否已存在配置
    if (!configId) {
        const existingConfigs = await getOpeningTimeConfig();
        if (existingConfigs.length > 0) {
            // 使用第一个配置的ID进行更新
            configId = existingConfigs[0].otId;
            isUpdate = true;
        } else {
            // 创建新配置
            configId = randomId(TABLEID.服务开启时间表); // 使用适当的前缀生成ID
        }
    } else {
        isUpdate = true;
    }
    
    const updateInfo = {
        startTime: getMySqlMs(startTime),
        endTime: getMySqlMs(endTime),
        isOpen,
        updated_by,
        updated_at: getMySqlMs()
    };
    
    if (isUpdate) {
        // 更新现有配置
        await updateManyData(
            TABLENAME.服务开启时间表,
            { otId: configId },
            updateInfo
        );
    } else {
        // 创建新配置
        const addInfo = {
            otId: configId,
            ...updateInfo
        };
        await addData(TABLENAME.服务开启时间表, addInfo);
    }
    
    return { 
        isSuccess: true, 
        otId: configId,
        message: isUpdate ? '配置更新成功' : '配置创建成功'
    };
}


/**
 * 启用或禁用服务时间配置
 * @param otId 配置ID
 * @param isOpen 是否启用 (0-否, 1-是)
 * @param updated_by 修改人
 */
export async function toggleOpeningTimeConfig(otId: string, isOpen: number, updated_by: string) {
    // 参数验证
    if (!otId) {
        throw new BizError(ERRORENUM.参数错误, '配置ID不能为空');
    }

    if (isOpen !== 0 && isOpen !== 1) {
        throw new BizError(ERRORENUM.参数错误, 'isOpen参数必须为0或1');
    }

    if (!updated_by) {
        throw new BizError(ERRORENUM.参数错误, '修改人不能为空');
    }

    // 检查配置是否存在
    const existingConfig = await selectOneDataByParam(
        TABLENAME.服务开启时间表,
        { otId },
        ["otId"]
    );

    if (!existingConfig.data) {
        throw new BizError(ERRORENUM.数据不存在, '指定的配置不存在');
    }

    // 更新配置
    await updateManyData(
        TABLENAME.服务开启时间表,
        { otId },
        {
            isOpen,
            updated_by,
            updated_at: getMySqlMs()
        }
    );

    return { 
        isSuccess: true, 
        message: `配置已${isOpen === 1 ? '启用' : '禁用'}`
    };
}


/**
 * 检查当前是否在服务时间内
 */
 export async function checkIsInOpeningTime() {
    const configs = await getOpeningTimeConfig();
    
    // 如果没有配置或没有启用的配置，默认允许访问
    if (!configs || configs.length === 0) {
        return { isInTime: true, message: '未配置服务时间，默认允许访问' };
    }
    
    // 查找启用的配置
    const activeConfig = configs.find(config => config.isOpen === 1);
    
    if (!activeConfig) {
        return { isInTime: false, message: '没有启用的服务时间配置' };
    }
    
    const now = new Date();
    const startTime = new Date(activeConfig.startTime);
    const endTime = new Date(activeConfig.endTime);
    
    const isInTime = now >= startTime && now <= endTime;
    
    return {
        isInTime,
        message: isInTime ? '当前在服务时间内' : '当前不在服务时间内',
        config: activeConfig
    };
}


/**
 * 查询公/智/能三大方向下，各维度的所有题目（题目顺序随机）
 * @returns {Promise<{
 *   公: Array<{dimensionName: string, questionList: Array<any>}>,
 *   智: Array<{dimensionName: string, questionList: Array<any>}>,
 *   能: Array<{dimensionName: string, questionList: Array<any>}>
 * }>} 按方向+维度区分的题目数据（题目随机顺序）
 */
// export async function getQuestions() {
//     // 1. 配置多表关联：题目表关联维度表，获取维度名称、所属方向（公/智/能）
//     const manyTableInfo: any = {};
//     manyTableInfo[TABLENAME.维度表] = {
//       column: ["dimension_id", "dimension_name", "direction"], // 维度表核心字段（文档关联字段）
//       where: {}, // 无筛选，查询文档中所有9个维度
//     };

//     // 2. 配置题目表返回字段（严格对应文档"量表题目"内容及表结构）
//     const questionReturnFields = [
//       "question_id",       // 题目唯一标识（您定义的VARCHAR类型）
//       "question_content",  // 题目具体内容（文档中"量表题目"列的文本）
//       "question_order"     // 题目在维度内的序号（文档中每个维度1-4题的顺序）
//     ];

//     // 3. 执行多表关联查询：获取所有题目及对应维度信息
//     const queryResult = await selectDataToTableAssociation(
//       TABLENAME.题目表,    // 主表：题目表
//       manyTableInfo,       // 关联表配置：维度表
//       {},                  // 主表查询条件：无（查询文档中36道题）
//       questionReturnFields // 题目表返回字段
//     );

//     // 4. 按"维度ID"分组，确保每个维度的题目聚合在一起
//     const dimensionQuestionMap = new Map<string, any>();
//     queryResult.data.forEach((question: any) => {
//       // 提取关联的维度信息（匹配文档中的维度定义）
//       const { 
//         dimension_id, 
//         dimension_name, 
//         direction 
//       } = question.dimension;
//       // 整理单题数据（仅保留文档相关信息）
//       const formattedQuestion = {
//         question_id: question.question_id, //题目ID
//         question_content: question.question_content, //题目内容
//         question_order: question.question_order //维度内序号
//       };

//       // 按维度ID分组
//       if (!dimensionQuestionMap.has(dimension_id)) {
//             dimensionQuestionMap.set(dimension_id, {
//             direction: direction,        // 所属大方向（公/智/能）
//             dimensionName: dimension_name,// 维度名称（如文档中的"家国情怀"）
//                 questions: []                // 该维度下的题目列表
//             });
//         }
//         // 将题目推入对应维度的列表
//         dimensionQuestionMap.get(dimension_id).questions.push(formattedQuestion);
//     });

//     // 5. 对每个维度的题目进行随机排序（打乱原有顺序）
//     Array.from(dimensionQuestionMap.values()).forEach(dimension => {
//         // 使用Fisher-Yates洗牌算法随机排序
//         const questions = dimension.questions;
//         for (let i = questions.length - 1; i > 0; i--) {
//             const j = Math.floor(Math.random() * (i + 1));
//             [questions[i], questions[j]] = [questions[j], questions[i]];
//         }
//         dimension.questions = questions;
//     });

//     // 6. 初始化返回结构：按文档3大方向分类，每个方向下按维度区分
//     const result = {
//         公: [] as Array<{dimensionName: string, questionList: Array<any>}>,
//         智: [] as Array<{dimensionName: string, questionList: Array<any>}>,
//         能: [] as Array<{dimensionName: string, questionList: Array<any>}>
//     };

//     // 7. 按方向+维度整理数据（严格匹配文档"公/智/能"对应维度）
//     Array.from(dimensionQuestionMap.values()).forEach(dimension => {
//       const { direction, dimensionName, questions } = dimension;
//       // 仅保留文档中定义的3个方向，避免异常数据
//       if (result.hasOwnProperty(direction)) {
//         result[direction as keyof typeof result].push({
//             dimensionName: dimensionName, // 文档中的维度名称
//             questionList: questions      // 该维度下随机排序的4道题
//         });
//       }
//     });

//     // 8. 返回按"公/智/能+维度"区分的题目数据（题目顺序随机）
//     return result;
// }


/**
 * 查询公/智/能三大方向下的所有题目（按方向分类，题目随机顺序）
 * @returns {Promise<{
 *   公(evaluation1): Array<any>,
 *   智(evaluation2): Array<any>,
 *   能(evaluation3): Array<any>
 * }>} 按方向分类的题目数据（题目随机顺序）
 */
export async function getQuestionsByDirection() {
    // 1. 配置多表关联：题目表关联维度表，获取维度名称、所属方向（公/智/能）
    const manyTableInfo: any = {};
    manyTableInfo[TABLENAME.维度表] = {
      column: ["dimension_id", "dimension_name", "direction"], // 维度表核心字段
      where: {}, // 无筛选，查询文档中所有9个维度
    };

    // 2. 配置题目表返回字段
    const questionReturnFields = [
      "question_id",       // 题目唯一标识
      "question_content",  // 题目具体内容
      "question_order",    // 题目在维度内的序号
      "dimension_id"       // 维度ID，用于关联
    ];

    // 3. 执行多表关联查询：获取所有题目及对应维度信息
    const queryResult = await selectDataToTableAssociation(
      TABLENAME.题目表,    // 主表：题目表
      manyTableInfo,       // 关联表配置：维度表
      {},                  // 主表查询条件：无
      questionReturnFields // 题目表返回字段
    );

    // 4. 按方向分类题目，不按维度分组
    const result = {
      公: [] as Array<any>,
      智: [] as Array<any>,
      能: [] as Array<any>
    };

    // 5. 将题目按方向分类
    queryResult.data.forEach((question: any) => {
      const direction = question.dimension?.direction;
      if (direction && result.hasOwnProperty(direction)) {
        const formattedQuestion = {
          question_id: question.question_id,
          question_content: question.question_content,
          question_order: question.question_order,
          dimension_id: question.dimension_id,
          dimension_name: question.dimension?.dimension_name
        };
        result[direction as keyof typeof result].push(formattedQuestion);
      }
    });

    // 6. 对每个方向的题目进行随机排序
    Object.keys(result).forEach((direction) => {
      const questions = result[direction as keyof typeof result];
      // 使用Fisher-Yates洗牌算法随机排序
      for (let i = questions.length - 1; i > 0; i--) {
        const j = Math.floor(Math.random() * (i + 1));
        [questions[i], questions[j]] = [questions[j], questions[i]];
      }
    });

    // 7. 返回按方向分类的题目数据（题目随机顺序）
    let dataList = {
      evaluation1:result.公,
      evaluation2:result.智,
      evaluation3:result.能
    };

    return dataList;
}


/**
 * 完成答题接口（单题）
 * @param student_id 学生ID
 * @param question_id 题目ID
 * @param score 得分（1-7分）
 */
export async function completeAnswer(student_id: string, student_name: string, question_id: string, score: number) {
    // 新增检查：判断学生是否已经答过题
    const hasAnswered = await checkStudentHasAnswered(student_id);
    if (!hasAnswered.checkanswered) {
        throw new BizError(ERRORENUM.重复答题, '您已经完成过答题，不能重复答题');
    }

    // 1. 参数校验
    if (!student_id || !question_id || score === undefined || score === null) {
        throw new BizError(ERRORENUM.参数错误);
    }

    if (score < 1 || score > 7) {
        throw new BizError(ERRORENUM.参数错误, '分数必须在1-7之间');
    }

    // 2. 检查/创建答题记录
    const record = await checkOrCreateRecord(student_id, student_name);

    // 3. 检查是否已答过本题
    const existingDetail:any = await selectOneDataByParam(
        TABLENAME.答题记录明细表, 
        { question_id, record_id: record.record_id }, 
        []
    );

    // 4. 更新或创建明细记录
    if (existingDetail && Object.keys(existingDetail).length > 0) {
        await updateManyData(
            TABLENAME.答题记录明细表, 
            { detail_id: existingDetail.detail_id }, 
            { score }
        );
    } else {
        const addInfo = {
            detail_id: randomId(TABLEID.答题记录明细表),
            record_id: record.record_id,
            question_id,
            score
        };
        await addData(TABLENAME.答题记录明细表, addInfo);
    }

    // 5. 更新总分
    await updateTotalScore(record.record_id);

    return { isSuccess: true, record_id: record.record_id };
}


/**
 * 批量完成答题接口
 * @param student_id 学生ID
 * @param answers 答题数组
 */
export async function completeAnswerBatch(student_id: string, student_name: string, answers: Array<{question_id: string, score: number}>) {
    // 新增检查：判断学生是否已经答过题
    // const hasAnswered = await checkStudentHasAnswered(student_id);
    // if (hasAnswered) {
    //     throw new BizError(ERRORENUM.重复答题, '您已经完成过答题，不能重复答题');
    // }
    const hasAnswered = await checkStudentHasAnswered(student_id);
    if (!hasAnswered.checkanswered) {
        throw new BizError(ERRORENUM.重复答题, '您已经完成过答题，不能重复答题');
    }

    // 1. 参数校验
    if (!student_id || !student_name || !answers || !Array.isArray(answers) || answers.length === 0) {
        throw new BizError(ERRORENUM.参数错误);
    }

    // 2. 检查/创建答题记录
    const record = await checkOrCreateRecord(student_id, student_name);

    if (record) {

    }

    // 3. 批量处理答题明细
    for (const answer of answers) {
        const { question_id, score } = answer;

        if (!question_id || score === undefined || score === null) {
            throw new BizError(ERRORENUM.参数错误, '每个答题项必须包含question_id和score');
        }

        if (score < 1 || score > 7) {
            throw new BizError(ERRORENUM.参数错误, `题目${question_id}的分数必须在1-7之间`);
        }

        // 检查是否已答过本题
        const existingDetail:any = await selectOneDataByParam(
            TABLENAME.答题记录明细表, 
            { question_id, record_id: record.record_id }, 
            []
        );

        if (existingDetail.data && Object.keys(existingDetail.data).length > 0) {
            await updateManyData(
                TABLENAME.答题记录明细表, 
                { detail_id: existingDetail.data.detail_id }, 
                { score }
            );
        } else {
            const addInfo = {
                detail_id: randomId(TABLEID.答题记录明细表),
                record_id: record.record_id,
                question_id,
                score
            };
            await addData(TABLENAME.答题记录明细表, addInfo);
        }
    }

    // 4. 更新总分
    await updateTotalScore(record.record_id);

    return { record_id: record.record_id };
}


/**
 * 完成全部答题（标记答题状态为已完成）
 * @param record_id 答题记录ID
 */
export async function finishAnswer(record_id: string) {
    if (!record_id) {
        throw new BizError(ERRORENUM.参数错误);
    }

    // 检查答题记录是否存在
    const recordInfo = await selectOneDataByParam(
        TABLENAME.答题记录表, 
        { record_id }, 
        ["record_id"]
    );
    
    if (!recordInfo.data) {
        throw new BizError(ERRORENUM.答题记录不存在);
    }

    // 更新答题状态为已完成
    const updateInfo = {
        answer_status: STATE.是,
        answer_time: getMySqlMs()
    };

    await updateManyData(
        TABLENAME.答题记录表, 
        { record_id }, 
        updateInfo
    );

    return { isSuccess: true };
}


/**
 * 获取答题结果
 * @param record_id 答题记录ID
 */
export async function getAnswerResult(record_id: string) {
    if (!record_id) {
        throw new BizError(ERRORENUM.参数错误);
    }

    // 获取答题记录和明细
    const manyTableInfo: any = {};
    manyTableInfo[TABLENAME.答题记录表] = {
        column: ["total_score", "answer_time", "answer_status"],
        where: {}
    };
    manyTableInfo[TABLENAME.题目表] = {
        column: ["question_id", "dimension_id", "question_content"],
        where: {}
    };
    manyTableInfo[TABLENAME.维度表] = {
        column: ["dimension_name", "direction"],
        where: {}
    };

    const answerInfo: any = await selectDataToTableAssociation(
        TABLENAME.答题记录明细表,
        manyTableInfo,
        { record_id },
        ["detail_id", "record_id", "question_id", "score"]
    );

    if (!answerInfo) {
        throw new BizError(ERRORENUM.答题记录不存在);
    }

    // 计算各方向得分
    const scores = {
        公: 0,
        智: 0,
        能: 0
    };

    answerInfo.forEach((detail: any) => {
        const direction = detail.gaoxin_维度表?.direction;
        if (direction && scores.hasOwnProperty(direction)) {
            scores[direction as keyof typeof scores] += parseInt(detail.score || 0);
        }
    });

    const totalScore = parseInt(answerInfo[0]?.gaoxin_答题记录表?.total_score || 0);

    return {
        record_id,
        total_score: totalScore,
        scores,
        answer_time: answerInfo[0]?.gaoxin_答题记录表?.answer_time,
        answer_status: answerInfo[0]?.gaoxin_答题记录表?.answer_status
    };
}


/**
 * 查找未完成的记录或创建新记录
 * @param studentId 学生ID
 */
async function checkOrCreateRecord(studentId: string, studentName: string): Promise<any> {
    let addInfo = {};
    
    // 查找未完成的答题记录
    let existingRecord = await selectOneDataByParam(
        TABLENAME.答题记录表, 
        { student_id: studentId, student_name: studentName, answer_status: STATE.否 }, 
        ["record_id", "student_id", "total_score", "answer_time", "answer_status"]
    );

    if (existingRecord.data && Object.keys(existingRecord.data).length > 0) {
        addInfo = existingRecord.data;
    } else {
        // 创建新的答题记录
        addInfo = {
            record_id: randomId(TABLEID.答题记录表),
            student_id: studentId,
            student_name: studentName,
            answer_time: getMySqlMs(),
            total_score: 0,
            answer_status: STATE.否
        };
        await addData(TABLENAME.答题记录表, addInfo);
    }

    return addInfo;
}


/**
 * 更新总分
 * @param recordId 答题记录ID
 */
async function updateTotalScore(recordId: string) {
    // 获取该记录的所有答题明细
    const existingDetails:any = await selectDataListByParam(
        TABLENAME.答题记录明细表, 
        { record_id: recordId }, 
        ["score"]
    );

    // 计算总分
    let sumScore = 0;
    existingDetails.data.forEach((info: any) => {
        sumScore += parseInt(info.score || 0);
    });

    // 更新答题记录的总分
    let updateInfo = {
        total_score: sumScore,
        answer_time: getMySqlMs()
    };
    
    await updateManyData(
        TABLENAME.答题记录表, 
        { record_id: recordId }, 
        updateInfo
    );
}


/**
 * 获取答题结果及对应的狮子形象
 * @param record_id 答题记录ID
 */
export async function getAnswerResultWithLionImage(record_id: string) {
    if (!record_id) {
        throw new BizError(ERRORENUM.参数错误);
    }

    // 1. 获取答题记录基本信息
    const recordInfo = await selectOneDataByParam(
        TABLENAME.答题记录表,
        { record_id },
        ["record_id", "student_id", "total_score", "answer_time", "answer_status"]
    );

    if (!recordInfo.data) {
        throw new BizError(ERRORENUM.答题记录不存在);
    }

    // 修改答题记录为已完成：
    await finishAnswer(record_id);

    // 2. 获取各方向得分
    const scores = await calculateDirectionScores(record_id);

    // 3. 根据得分判断对应的狮子形象
    const lionImage = await getLionImageByScores(scores);

    return {
        record_id,
        total_score: recordInfo.data.total_score,
        scores, // 各方向得分
        lion_image: lionImage, // 对应的狮子形象数据
        answer_time: recordInfo.data.answer_time,
        answer_status: recordInfo.data.answer_status
    };
}


/**
 * 计算各方向得分
 * @param record_id 答题记录ID
 */
async function calculateDirectionScores(record_id: string) {
    // 获取答题明细及对应的题目信息
    let manyTableInfo: any = {};
    manyTableInfo[TABLENAME.题目表] = {
        column: ["question_id", "dimension_id"],
        where: {}
    };

    let answerDetails = await selectDataToTableAssociation(
        TABLENAME.答题记录明细表,
        manyTableInfo,
        { record_id },
        ["detail_id", "record_id", "question_id", "score"]
    );

    // 初始化各方向得分
    let directionScores = {
        公: 0,
        智: 0,
        能: 0
    };

    // 收集所有维度ID，用于批量查询维度信息
    let dimensionIds = new Set<string>(); // 创建空集合
    answerDetails.data.forEach((detail: any) => {
        let dimensionId;
        if (detail.dataValues && detail.dataValues !== null) {
            let question = detail.dataValues.question;
            dimensionId = question.dimension_id;
        }
        if (dimensionId) {
            dimensionIds.add(dimensionId);
        }
    });

    // 批量查询维度信息
    let dimensions = await selectDataListByParam(
        TABLENAME.维度表,
        { dimension_id: Array.from(dimensionIds) },
        ["dimension_id", "direction"]
    );

    // 创建维度ID到方向的映射
    let dimensionDirectionMap = {};
    dimensions.data.forEach((info: any) => {
        let dimension = info.dataValues;
        dimensionDirectionMap[dimension.dimension_id] = dimension.direction;
    });

    // 计算各方向总分
    answerDetails.data.forEach((detail: any) => {
        let dimensionId;
        if (detail.question && detail.question !== null) {
            let question = detail.question;
            dimensionId = question.dimension_id;

            let score = parseInt(detail.score || 0);
        
            if (dimensionId) {
                let direction = dimensionDirectionMap[dimensionId];
                if (direction && directionScores.hasOwnProperty(direction)) {
                    directionScores[direction as keyof typeof directionScores] += score;
                }
            }
        }
    });

    return directionScores;
}


/**
 * 根据得分获取对应的狮子形象
 * @param scores 各方向得分 {公: number, 智: number, 能: number}
 */
 async function getLionImageByScores(scores: {公: number; 智: number; 能: number}) {
    const { 公: publicScore, 智: intelligenceScore, 能: abilityScore } = scores;

    // 获取所有狮子形象的判断条件
    const allLionImages = await selectDataListByParam(
        TABLENAME.狮子形象表,
        {},
        [
            'lion_id',
            'standard_name',
            'alias_name',
            'characteristic',
            'judgment_condition',
            'magic_artifact',
            'magic_artifact_text',
            'suggestion',
            'suggestion_text',
            'lion_image'
        ]
    );

    // 遍历所有狮子形象，找到匹配的判断条件
    for (const lionImage of allLionImages.data) {
        const condition = lionImage.judgment_condition;
        
        if (condition && evaluateCondition(condition, publicScore, intelligenceScore, abilityScore)) {
            return lionImage;
        }
    }

    // 如果没有找到完全匹配的条件，返回默认的潜力成长狮
    const defaultLion = allLionImages.data.find((lion: any) => 
        lion.judgment_condition === '公<50&智<50&能<50'
    );

    return defaultLion || allLionImages.data[0]; // 返回默认狮子或第一个狮子
}


/**
 * 评估判断条件是否满足
 * @param condition 判断条件字符串，如 "公≥50 & 智≥50 & 能≥50"
 * @param publicScore 公方向得分
 * @param intelligenceScore 智方向得分
 * @param abilityScore 能方向得分
 */
function evaluateCondition(condition: string, publicScore: number, intelligenceScore: number, abilityScore: number): boolean {
    // 将条件字符串拆分为单个条件
    const conditions = condition.split('&').map(c => c.trim());
    
    for (const cond of conditions) {
        if (!evaluateSingleCondition(cond, publicScore, intelligenceScore, abilityScore)) {
            return false;
        }
    }
    
    return true;
}


/**
 * 评估单个条件
 * @param condition 单个条件字符串，如 "公≥50"
 */
function evaluateSingleCondition(condition: string, publicScore: number, intelligenceScore: number, abilityScore: number): boolean {
    // 匹配条件格式：方向 + 操作符 + 数值
    const match = condition.match(/(公|智|能)([<>≥≤≈=]+)(\d+)/);
    
    if (!match) {
        return false; // 条件格式不正确
    }
    
    const direction = match[1];
    const operator = match[2];
    const value = parseInt(match[3]);
    
    // 根据方向获取对应的分数
    let score: number;
    switch (direction) {
        case '公':
            score = publicScore;
            break;
        case '智':
            score = intelligenceScore;
            break;
        case '能':
            score = abilityScore;
            break;
        default:
            return false;
    }
    
    // 根据操作符进行比较
    switch (operator) {
        case '>':
            return score > value;
        case '<':
            return score < value;
        case '≥':
        case '>=':
            return score >= value;
        case '≤':
        case '<=':
            return score <= value;
        case '≈':
            // 近似判断，允许±5的误差
            return Math.abs(score - value) <= 5;
        case '=':
            return score === value;
        default:
            return false;
    }
}


/**
 * 获取所有狮子形象数据（用于前端展示）
 */
export async function getAllLionImages() {
    const lionImages = await selectDataListByParam(
        TABLENAME.狮子形象表,
        {},
        [
            'lion_id',
            'standard_name',
            'alias_name',
            'characteristic',
            'judgment_condition',
            'magic_artifact',
            'magic_artifact_text',
            'suggestion',
            'suggestion_text',
            'lion_image'
        ]
    );

    return lionImages;
}


/**
 * 根据学生ID获取最新的答题结果和狮子形象
 * @param student_id 学生ID
 */
export async function getLatestAnswerResultByStudent(student_id: string) {
    if (!student_id) {
        throw new BizError(ERRORENUM.参数错误);
    }

    // 获取学生最新的答题记录
    const latestRecord = await selectDataWithCustomOrder(
        TABLENAME.答题记录表,
        { student_id },
        ["record_id"],
        [["answer_time", "DESC" ]] // 按时间倒序，获取最新的记录
    );

    if (!latestRecord.data) {
        throw new BizError(ERRORENUM.答题记录不存在);
    }

    // 获取详细的答题结果和狮子形象
    return await getAnswerResultWithLionImage(latestRecord.data.record_id);
}


// 管理员页面 =========================================================================================================================

/**
 * 获取所有学生答题记录及详细得分
 * @returns 包含学生姓名和各维度得分的列表
 */
export async function getAllStudentAnswerRecords() {
    // 1. 先同步uac_user表中的活跃用户到answer_record表
    await syncActiveUsersToAnswerRecord();

    // 1. 获取所有已完成答题的记录
    const completedRecords = await selectDataListByParam(
        TABLENAME.答题记录表,
        { answer_status: STATE.是 }, // 只获取已完成的记录
        ["record_id", "student_id", "student_name", "total_score", "answer_time"]
    );

    if (!completedRecords.data || completedRecords.data.length === 0) {
        return [];
    }

    const result = [];

    // 2. 遍历每个学生的答题记录
    for (const record of completedRecords.data) {
        const studentRecord = await getStudentDetailedScores(record.record_id);
        if (studentRecord) {
            result.push(studentRecord);
        }
    }

    return result;
}


/**
 * 获取单个学生的详细得分情况
 */
async function getStudentDetailedScores(record_id: string) {
    // 获取答题明细及对应的题目和维度信息
    const answerDetails = await selectDataToTableAssociation(
        TABLENAME.答题记录明细表,
        {
            [TABLENAME.题目表]: {
                column: ["question_id", "dimension_id"],
                where: {}
            }
        },
        { record_id },
        ["detail_id", "score", "question_id"]
    );

    if (!answerDetails.data || answerDetails.data.length === 0) {
        return null;
    }

    // 收集所有维度ID，用于批量查询维度信息
    const dimensionIds = new Set<string>();
    answerDetails.data.forEach((detail: any) => {
        const dimensionId = detail.question?.dimension_id;
        if (dimensionId) {
            dimensionIds.add(dimensionId);
        }
    });

    // 批量查询维度信息
    const dimensions = await selectDataListByParam(
        TABLENAME.维度表,
        { dimension_id: Array.from(dimensionIds) },
        ["dimension_id", "dimension_name"]
    );

    // 创建维度ID到维度名称的映射
    const dimensionNameMap = new Map();
    dimensions.data.forEach((dimension: any) => {
        dimensionNameMap.set(dimension.dimension_id, dimension.dimension_name);
    });

    // 获取答题记录基本信息
    const recordInfo = await selectOneDataByParam(
        TABLENAME.答题记录表,
        { record_id },
        ["student_name", "total_score", "answer_time"]
    );

    // 初始化各维度得分
    const dimensionScores: { [key: string]: number } = {
        '家国情怀': 0,
        '国际视野': 0,
        '责任担当': 0,
        '学业扎实': 0,
        '勇于创新': 0,
        '善于学习': 0,
        '健康生活': 0,
        '审美情趣': 0,
        '劳动意识': 0
    };

    // 计算各维度得分
    answerDetails.data.forEach((detail: any) => {
        const dimensionId = detail.question?.dimension_id;
        const dimensionName = dimensionNameMap.get(dimensionId);
        const score = parseInt(detail.score || 0);
        
        if (dimensionName && dimensionScores.hasOwnProperty(dimensionName)) {
            dimensionScores[dimensionName] += score;
        }
    });

    // 计算三大方向得分
    const publicScore = dimensionScores['家国情怀'] + dimensionScores['国际视野'] + dimensionScores['责任担当'];
    const intelligenceScore = dimensionScores['学业扎实'] + dimensionScores['勇于创新'] + dimensionScores['善于学习'];
    const abilityScore = dimensionScores['健康生活'] + dimensionScores['审美情趣'] + dimensionScores['劳动意识'];

    return {
        studentName: recordInfo.data?.student_name,
        // 公方向
        jgqh: dimensionScores['家国情怀'],
        gjsy: dimensionScores['国际视野'],
        zrdd: dimensionScores['责任担当'],
        gong: publicScore,
        // 智方向
        xyzs: dimensionScores['学业扎实'],
        yycx: dimensionScores['勇于创新'],
        syxx: dimensionScores['善于学习'],
        zhi: intelligenceScore,
        // 能方向
        jksh: dimensionScores['健康生活'],
        smqq: dimensionScores['审美情趣'],
        ldys: dimensionScores['劳动意识'],
        neng: abilityScore,
        // 总分
        totalScore: recordInfo.data?.total_score,
        answerTime: recordInfo.data?.answer_time || "暂未提交答题"
    };
}


/**
 * 获取所有学生答题记录（简化版 - 批量查询优化性能）- 分页版本
 * @param page 页码，从1开始
 * @param pageSize 每页大小
 */
// export async function getAllStudentAnswerRecordsOptimized(page: number = 1, pageSize: number = 10) {
//     try {
//         // 1. 先同步活跃用户
//         await syncActiveUsersToAnswerRecord();

//         // 2. 获取活跃用户ID
//         const activeUsers = await selectDataListByParam(
//             TABLENAME.统一用户表,
//             { active_flag: 1 },
//             ["user_id"]
//         );

//         const activeUserIds = activeUsers.data?.map((user: any) => user.user_id) || [];

//         if (activeUserIds.length === 0) {
//             return {
//                 list: [],
//                 total: 0,
//                 page,
//                 pageSize,
//                 totalPages: 0
//             };
//         }

//         // 3. 获取每个用户的最新答题记录（使用替代方案）
//         const { records: latestRecords, total } = await getLatestStudentRecords(activeUserIds, page, pageSize);

//         if (!latestRecords || latestRecords.length === 0) {
//             return {
//                 list: [],
//                 total,
//                 page,
//                 pageSize,
//                 totalPages: Math.ceil(total / pageSize)
//             };
//         }

//         // 4. 分离已完成和未完成的记录
//         const completedRecords = latestRecords.filter(record => record.answer_status === 1);
//         const recordIds = completedRecords.map((record: any) => record.record_id);

//         // 5. 只为已完成记录获取答题明细和维度信息
//         let recordScoreMap = new Map();
//         const dimensionKeys = [
//             '家国情怀', '国际视野', '责任担当',
//             '学业扎实', '勇于创新', '善于学习',
//             '健康生活', '审美情趣', '劳动意识'
//         ];

//         if (recordIds.length > 0) {
//             // 获取答题明细
//             const allAnswerDetails = await selectDataListByParam(
//                 TABLENAME.答题记录明细表,
//                 { record_id: recordIds },
//                 ["record_id", "score", "question_id"]
//             );

//             if (allAnswerDetails.data && allAnswerDetails.data.length > 0) {
//                 // 获取题目和维度信息
//                 const questionIds = [...new Set(allAnswerDetails.data.map((detail: any) => detail.question_id))];
                
//                 const includeQuestionDimension: any = {};
//                 includeQuestionDimension[TABLENAME.维度表] = {
//                     column: ["dimension_id", "dimension_name"],
//                     where: {}
//                 };

//                 const questionsWithDimensions = await selectDataToTableAssociation(
//                     TABLENAME.题目表,
//                     includeQuestionDimension,
//                     { question_id: questionIds },
//                     ["question_id", "dimension_id"]
//                 );

//                 // 创建题目维度映射
//                 const questionDimensionMap = new Map();
//                 questionsWithDimensions.data?.forEach((item: any) => {
//                     const dimensionName = item.dimension?.dimension_name;
//                     if (dimensionName) {
//                         questionDimensionMap.set(item.question_id, dimensionName);
//                     }
//                 });

//                 // 初始化已完成记录的维度分数
//                 completedRecords.forEach((record: any) => {
//                     const initialScores: { [key: string]: number } = {};
//                     dimensionKeys.forEach(key => {
//                         initialScores[key] = 0;
//                     });
//                     recordScoreMap.set(record.record_id, initialScores);
//                 });

//                 // 计算各维度分数
//                 allAnswerDetails.data.forEach((detail: any) => {
//                     const dimensionName = questionDimensionMap.get(detail.question_id);
//                     const score = parseInt(detail.score || 0);
                    
//                     if (dimensionName && recordScoreMap.has(detail.record_id)) {
//                         const scores = recordScoreMap.get(detail.record_id);
//                         if (scores && scores.hasOwnProperty(dimensionName)) {
//                             scores[dimensionName] += score;
//                         }
//                     }
//                 });
//             }
//         }

//         // 6. 构建最终结果
//         const list = latestRecords.map((record: any) => {
//             if (record.answer_status === 0) {
//                 // 未完成记录
//                 return {
//                     record_id: record.record_id,
//                     studentName: record.student_name,
//                     jgqh: 0,
//                     gjsy: 0,
//                     zrdd: 0,
//                     gong: 0,
//                     xyzs: 0,
//                     yycx: 0,
//                     syxx: 0,
//                     zhi: 0,
//                     jksh: 0,
//                     smqq: 0,
//                     ldys: 0,
//                     neng: 0,
//                     totalScore: 0,
//                     answerTime: "暂未提交答题",
//                     answerStatus: record.answer_status
//                 };
//             } else {
//                 // 已完成记录
//                 const dimensionScores = recordScoreMap.get(record.record_id) || {};
                
//                 const publicScore = (dimensionScores['家国情怀'] || 0) + 
//                                   (dimensionScores['国际视野'] || 0) + 
//                                   (dimensionScores['责任担当'] || 0);
                                  
//                 const intelligenceScore = (dimensionScores['学业扎实'] || 0) + 
//                                         (dimensionScores['勇于创新'] || 0) + 
//                                         (dimensionScores['善于学习'] || 0);
                                        
//                 const abilityScore = (dimensionScores['健康生活'] || 0) + 
//                                    (dimensionScores['审美情趣'] || 0) + 
//                                    (dimensionScores['劳动意识'] || 0);

//                 return {
//                     record_id: record.record_id,
//                     studentName: record.student_name,
//                     jgqh: dimensionScores['家国情怀'] || 0,
//                     gjsy: dimensionScores['国际视野'] || 0,
//                     zrdd: dimensionScores['责任担当'] || 0,
//                     gong: publicScore,
//                     xyzs: dimensionScores['学业扎实'] || 0,
//                     yycx: dimensionScores['勇于创新'] || 0,
//                     syxx: dimensionScores['善于学习'] || 0,
//                     zhi: intelligenceScore,
//                     jksh: dimensionScores['健康生活'] || 0,
//                     smqq: dimensionScores['审美情趣'] || 0,
//                     ldys: dimensionScores['劳动意识'] || 0,
//                     neng: abilityScore,
//                     totalScore: record.total_score || 0,
//                     answerTime: record.answer_time || "已提交",
//                     answerStatus: record.answer_status
//                 };
//             }
//         });

//         const totalPages = Math.ceil(total / pageSize);

//         return {
//             list,
//             total,
//             page,
//             pageSize,
//             totalPages
//         };

//     } catch (error) {
//         console.error('获取答题记录失败:', error);
//         return {
//             list: [],
//             total: 0,
//             page,
//             pageSize,
//             totalPages: 0
//         };
//     }
// }





// 辅助函数：获取每个用户的最新答题记录


/**
 * 获取所有学生答题记录（简化版 - 批量查询优化性能）- 分页版本
 * @param page 页码，从1开始
 * @param pageSize 每页大小
 */
export async function getAllStudentAnswerRecordsOptimized(page: number = 1, pageSize: number = 10) {
    try {
        // 1. 先同步活跃用户
        await syncActiveUsersToAnswerRecord();

        // 2. 获取活跃用户ID
        const activeUsers = await selectDataListByParam(
            TABLENAME.统一用户表,
            { active_flag: 1 },
            ["user_id"]
        );

        const activeUserIds = activeUsers.data?.map((user: any) => user.user_id) || [];

        if (activeUserIds.length === 0) {
            return {
                list: [],
                total: 0,
                page,
                pageSize,
                totalPages: 0
            };
        }

        // 3. 获取每个学生的答题记录（优先取已答题记录，相同学生多条已答题记录时取最早的一条）
        const { records: studentRecords, total } = await getStudentRecordsWithPriority(activeUserIds, page, pageSize);

        if (!studentRecords || studentRecords.length === 0) {
            return {
                list: [],
                total,
                page,
                pageSize,
                totalPages: Math.ceil(total / pageSize)
            };
        }

        // 4. 获取所有记录的答题明细和维度信息（包括未完成记录）
        const allRecordIds = studentRecords.map((record: any) => record.record_id);
        let recordScoreMap = new Map();
        const dimensionKeys = [
            '家国情怀', '国际视野', '责任担当',
            '学业扎实', '勇于创新', '善于学习',
            '健康生活', '审美情趣', '劳动意识'
        ];

        if (allRecordIds.length > 0) {
            // 获取所有记录的答题明细
            const allAnswerDetails = await selectDataListByParam(
                TABLENAME.答题记录明细表,
                { record_id: allRecordIds },
                ["record_id", "score", "question_id"]
            );

            if (allAnswerDetails.data && allAnswerDetails.data.length > 0) {
                // 获取题目和维度信息
                const questionIds = [...new Set(allAnswerDetails.data.map((detail: any) => detail.question_id))];
                
                const includeQuestionDimension: any = {};
                includeQuestionDimension[TABLENAME.维度表] = {
                    column: ["dimension_id", "dimension_name"],
                    where: {}
                };

                const questionsWithDimensions = await selectDataToTableAssociation(
                    TABLENAME.题目表,
                    includeQuestionDimension,
                    { question_id: questionIds },
                    ["question_id", "dimension_id"]
                );

                // 创建题目维度映射
                const questionDimensionMap = new Map();
                questionsWithDimensions.data?.forEach((item: any) => {
                    const dimensionName = item.dimension?.dimension_name;
                    if (dimensionName) {
                        questionDimensionMap.set(item.question_id, dimensionName);
                    }
                });

                // 初始化所有记录的维度分数
                studentRecords.forEach((record: any) => {
                    const initialScores: { [key: string]: number } = {};
                    dimensionKeys.forEach(key => {
                        initialScores[key] = 0;
                    });
                    recordScoreMap.set(record.record_id, initialScores);
                });

                // 计算各维度分数（包括未完成记录）
                allAnswerDetails.data.forEach((detail: any) => {
                    const dimensionName = questionDimensionMap.get(detail.question_id);
                    const score = parseInt(detail.score || 0);
                    
                    if (dimensionName && recordScoreMap.has(detail.record_id)) {
                        const scores = recordScoreMap.get(detail.record_id);
                        if (scores && scores.hasOwnProperty(dimensionName)) {
                            scores[dimensionName] += score;
                        }
                    }
                });
            }
        }

        // 5. 构建最终结果 - 正确显示所有记录的实际数据
        const list = studentRecords.map((record: any) => {
            const dimensionScores = recordScoreMap.get(record.record_id) || {};
            
            const publicScore = (dimensionScores['家国情怀'] || 0) + 
                              (dimensionScores['国际视野'] || 0) + 
                              (dimensionScores['责任担当'] || 0);
                              
            const intelligenceScore = (dimensionScores['学业扎实'] || 0) + 
                                    (dimensionScores['勇于创新'] || 0) + 
                                    (dimensionScores['善于学习'] || 0);
                                    
            const abilityScore = (dimensionScores['健康生活'] || 0) + 
                               (dimensionScores['审美情趣'] || 0) + 
                               (dimensionScores['劳动意识'] || 0);

            return {
                record_id: record.record_id,
                student_id: record.student_id,
                studentName: record.student_name,
                // 维度得分 - 使用实际数据
                jgqh: dimensionScores['家国情怀'] || 0,
                gjsy: dimensionScores['国际视野'] || 0,
                zrdd: dimensionScores['责任担当'] || 0,
                gong: publicScore,
                xyzs: dimensionScores['学业扎实'] || 0,
                yycx: dimensionScores['勇于创新'] || 0,
                syxx: dimensionScores['善于学习'] || 0,
                zhi: intelligenceScore,
                jksh: dimensionScores['健康生活'] || 0,
                smqq: dimensionScores['审美情趣'] || 0,
                ldys: dimensionScores['劳动意识'] || 0,
                neng: abilityScore,
                // 其他字段 - 使用实际数据
                totalScore: record.total_score || 0,
                answerTime: record.answer_status === 1 ? (record.answer_time || "已提交") : (record.answer_time || "暂未提交答题"),
                answerStatus: record.answer_status
            };
        });

        const totalPages = Math.ceil(total / pageSize);

        return {
            list,
            total,
            page,
            pageSize,
            totalPages
        };

    } catch (error) {
        console.error('获取答题记录失败:', error);
        return {
            list: [],
            total: 0,
            page,
            pageSize,
            totalPages: 0
        };
    }
}


// export async function getAllStudentAnswerRecordsOptimized(page: number = 1, pageSize: number = 10) {
//     try {
//         // 1. 先同步活跃用户
//         await syncActiveUsersToAnswerRecord();

//         // 2. 获取活跃用户ID
//         const activeUsers = await selectDataListByParam(
//             TABLENAME.统一用户表,
//             { active_flag: 1 },
//             ["user_id"]
//         );

//         const activeUserIds = activeUsers.data?.map((user: any) => user.user_id) || [];

//         if (activeUserIds.length === 0) {
//             return {
//                 list: [],
//                 total: 0,
//                 page,
//                 pageSize,
//                 totalPages: 0
//             };
//         }

//         // 3. 获取每个学生的答题记录（优先取已答题记录，相同学生多条已答题记录时取最早的一条）
//         const { records: studentRecords, total } = await getStudentRecordsWithPriority(activeUserIds, page, pageSize);

//         if (!studentRecords || studentRecords.length === 0) {
//             return {
//                 list: [],
//                 total,
//                 page,
//                 pageSize,
//                 totalPages: Math.ceil(total / pageSize)
//             };
//         }

//         // 4. 分离已完成和未完成的记录
//         const completedRecords = studentRecords.filter(record => record.answer_status === 1);
//         const recordIds = completedRecords.map((record: any) => record.record_id);

//         // 5. 只为已完成记录获取答题明细和维度信息
//         let recordScoreMap = new Map();
//         const dimensionKeys = [
//             '家国情怀', '国际视野', '责任担当',
//             '学业扎实', '勇于创新', '善于学习',
//             '健康生活', '审美情趣', '劳动意识'
//         ];

//         if (recordIds.length > 0) {
//             // 获取答题明细
//             const allAnswerDetails = await selectDataListByParam(
//                 TABLENAME.答题记录明细表,
//                 { record_id: recordIds },
//                 ["record_id", "score", "question_id"]
//             );

//             if (allAnswerDetails.data && allAnswerDetails.data.length > 0) {
//                 // 获取题目和维度信息
//                 const questionIds = [...new Set(allAnswerDetails.data.map((detail: any) => detail.question_id))];
                
//                 const includeQuestionDimension: any = {};
//                 includeQuestionDimension[TABLENAME.维度表] = {
//                     column: ["dimension_id", "dimension_name"],
//                     where: {}
//                 };

//                 const questionsWithDimensions = await selectDataToTableAssociation(
//                     TABLENAME.题目表,
//                     includeQuestionDimension,
//                     { question_id: questionIds },
//                     ["question_id", "dimension_id"]
//                 );

//                 // 创建题目维度映射
//                 const questionDimensionMap = new Map();
//                 questionsWithDimensions.data?.forEach((item: any) => {
//                     const dimensionName = item.dimension?.dimension_name;
//                     if (dimensionName) {
//                         questionDimensionMap.set(item.question_id, dimensionName);
//                     }
//                 });

//                 // 初始化已完成记录的维度分数
//                 completedRecords.forEach((record: any) => {
//                     const initialScores: { [key: string]: number } = {};
//                     dimensionKeys.forEach(key => {
//                         initialScores[key] = 0;
//                     });
//                     recordScoreMap.set(record.record_id, initialScores);
//                 });

//                 // 计算各维度分数
//                 allAnswerDetails.data.forEach((detail: any) => {
//                     const dimensionName = questionDimensionMap.get(detail.question_id);
//                     const score = parseInt(detail.score || 0);
                    
//                     if (dimensionName && recordScoreMap.has(detail.record_id)) {
//                         const scores = recordScoreMap.get(detail.record_id);
//                         if (scores && scores.hasOwnProperty(dimensionName)) {
//                             scores[dimensionName] += score;
//                         }
//                     }
//                 });
//             }
//         }

//         // 6. 构建最终结果
//         const list = studentRecords.map((record: any) => {
//             if (record.answer_status === 0) {
//                 // 未完成记录
//                 return {
//                     record_id: record.record_id,
//                     studentName: record.student_name,
//                     jgqh: 0,
//                     gjsy: 0,
//                     zrdd: 0,
//                     gong: 0,
//                     xyzs: 0,
//                     yycx: 0,
//                     syxx: 0,
//                     zhi: 0,
//                     jksh: 0,
//                     smqq: 0,
//                     ldys: 0,
//                     neng: 0,
//                     totalScore: 0,
//                     answerTime: "暂未提交答题",
//                     answerStatus: record.answer_status
//                 };
//             } else {
//                 // 已完成记录
//                 const dimensionScores = recordScoreMap.get(record.record_id) || {};
                
//                 const publicScore = (dimensionScores['家国情怀'] || 0) + 
//                                   (dimensionScores['国际视野'] || 0) + 
//                                   (dimensionScores['责任担当'] || 0);
                                  
//                 const intelligenceScore = (dimensionScores['学业扎实'] || 0) + 
//                                         (dimensionScores['勇于创新'] || 0) + 
//                                         (dimensionScores['善于学习'] || 0);
                                        
//                 const abilityScore = (dimensionScores['健康生活'] || 0) + 
//                                    (dimensionScores['审美情趣'] || 0) + 
//                                    (dimensionScores['劳动意识'] || 0);

//                 return {
//                     record_id: record.record_id,
//                     studentName: record.student_name,
//                     jgqh: dimensionScores['家国情怀'] || 0,
//                     gjsy: dimensionScores['国际视野'] || 0,
//                     zrdd: dimensionScores['责任担当'] || 0,
//                     gong: publicScore,
//                     xyzs: dimensionScores['学业扎实'] || 0,
//                     yycx: dimensionScores['勇于创新'] || 0,
//                     syxx: dimensionScores['善于学习'] || 0,
//                     zhi: intelligenceScore,
//                     jksh: dimensionScores['健康生活'] || 0,
//                     smqq: dimensionScores['审美情趣'] || 0,
//                     ldys: dimensionScores['劳动意识'] || 0,
//                     neng: abilityScore,
//                     totalScore: record.total_score || 0,
//                     answerTime: record.answer_time || "已提交",
//                     answerStatus: record.answer_status
//                 };
//             }
//         });

//         const totalPages = Math.ceil(total / pageSize);

//         return {
//             list,
//             total,
//             page,
//             pageSize,
//             totalPages
//         };

//     } catch (error) {
//         console.error('获取答题记录失败:', error);
//         return {
//             list: [],
//             total: 0,
//             page,
//             pageSize,
//             totalPages: 0
//         };
//     }
// }


// 辅助函数：获取每个学生的答题记录（优先已答题，相同学生多条已答题记录时取有数据的记录）
async function getStudentRecordsWithPriority(activeUserIds: string[], page: number, pageSize: number) {
    try {
        // 获取所有活跃用户的答题记录
        const allRecords = await selectDataListByParam(
            TABLENAME.答题记录表,
            { student_id: activeUserIds },
            ["record_id", "student_id", "student_name", "total_score", "answer_time", "answer_status"]
        );

        if (!allRecords.data || allRecords.data.length === 0) {
            return { records: [], total: 0 };
        }

        // 按用户分组，处理每个学生的记录
        const userRecordMap = new Map();
        
        allRecords.data.forEach((record: any) => {
            const studentId = record.student_id;
            const existingRecord = userRecordMap.get(studentId);
            
            if (!existingRecord) {
                // 如果还没有记录，直接设置
                userRecordMap.set(studentId, record);
            } else {
                const existingStatus = existingRecord.answer_status;
                const currentStatus = record.answer_status;
                const existingTime = new Date(existingRecord.answer_time);
                const currentTime = new Date(record.answer_time);
                
                // 优先逻辑：已答题记录优先于未答题记录
                if (existingStatus === 0 && currentStatus === 1) {
                    // 现有记录是未答题，新记录是已答题，使用新记录
                    userRecordMap.set(studentId, record);
                } else if (existingStatus === 1 && currentStatus === 1) {
                    // 两个都是已答题记录，取时间更早的记录
                    if (currentTime < existingTime) {
                        userRecordMap.set(studentId, record);
                    }
                } else if (existingStatus === 1 && currentStatus === 0) {
                    // 现有记录是已答题，新记录是未答题，保持现有记录
                    // 不做任何操作
                } else if (existingStatus === 0 && currentStatus === 0) {
                    // 两个都是未答题记录，检查是否有数据
                    const existingHasData = existingRecord.total_score > 0 || existingRecord.answer_time;
                    const currentHasData = record.total_score > 0 || record.answer_time;
                    
                    if (!existingHasData && currentHasData) {
                        // 现有记录无数据，新记录有数据，使用新记录
                        userRecordMap.set(studentId, record);
                    } else if (existingHasData && !currentHasData) {
                        // 现有记录有数据，新记录无数据，保持现有记录
                        // 不做任何操作
                    } else if (existingHasData && currentHasData) {
                        // 两个都有数据，取时间更早的记录
                        if (currentTime < existingTime) {
                            userRecordMap.set(studentId, record);
                        }
                    } else {
                        // 两个都无数据，取时间更早的记录
                        if (currentTime < existingTime) {
                            userRecordMap.set(studentId, record);
                        }
                    }
                }
            }
        });

        // 转换为数组
        const uniqueRecords = Array.from(userRecordMap.values());
        
        // 总体排序逻辑：按答题时间倒序（最新的在前）
        uniqueRecords.sort((a, b) => {
            const timeA = new Date(a.answer_time).getTime();
            const timeB = new Date(b.answer_time).getTime();
            return timeB - timeA; // 倒序：时间大的在前
        });

        // 手动分页
        const startIndex = (page - 1) * pageSize;
        const endIndex = startIndex + pageSize;
        const paginatedRecords = uniqueRecords.slice(startIndex, endIndex);

        return {
            records: paginatedRecords,
            total: uniqueRecords.length
        };
        
    } catch (error) {
        console.error('获取学生记录失败:', error);
        return { records: [], total: 0 };
    }
}


// 辅助函数：获取每个学生的最早答题记录
async function getEarliestStudentRecords(activeUserIds: string[], page: number, pageSize: number) {
    try {
        // 获取所有活跃用户的答题记录
        const allRecords = await selectDataListByParam(
            TABLENAME.答题记录表,
            { student_id: activeUserIds },
            ["record_id", "student_id", "student_name", "total_score", "answer_time", "answer_status"]
        );

        if (!allRecords.data || allRecords.data.length === 0) {
            return { records: [], total: 0 };
        }

        // 按用户分组，取每个学生的最早记录
        const userRecordMap = new Map();
        
        allRecords.data.forEach((record: any) => {
            const existingRecord = userRecordMap.get(record.student_id);
            
            // 如果还没有记录，或者当前记录时间更早，则更新
            if (!existingRecord) {
                userRecordMap.set(record.student_id, record);
            } else {
                const existingTime = new Date(existingRecord.answer_time);
                const currentTime = new Date(record.answer_time);
                
                // 修改：取时间更早的记录
                if (currentTime < existingTime) {
                    userRecordMap.set(record.student_id, record);
                }
            }
        });

        // 转换为数组
        const uniqueRecords = Array.from(userRecordMap.values());
        
        // 修改排序逻辑：总体按答题时间倒序（最新的在前）
        uniqueRecords.sort((a, b) => {
            const timeA = new Date(a.answer_time).getTime();
            const timeB = new Date(b.answer_time).getTime();
            return timeB - timeA; // 倒序：时间大的在前
        });

        // 手动分页
        const startIndex = (page - 1) * pageSize;
        const endIndex = startIndex + pageSize;
        const paginatedRecords = uniqueRecords.slice(startIndex, endIndex);

        return {
            records: paginatedRecords,
            total: uniqueRecords.length
        };
        
    } catch (error) {
        console.error('获取最早学生记录失败:', error);
        return { records: [], total: 0 };
    }
}


async function getLatestStudentRecords(activeUserIds: string[], page: number, pageSize: number) {
    try {
        // 获取所有活跃用户的答题记录
        const allRecords = await selectDataListByParam(
            TABLENAME.答题记录表,
            { student_id: activeUserIds },
            ["record_id", "student_id", "student_name", "total_score", "answer_time", "answer_status"]
        );

        if (!allRecords.data || allRecords.data.length === 0) {
            return { records: [], total: 0 };
        }

        // 按用户分组，取每个用户的最新记录
        const userRecordMap = new Map();
        
        allRecords.data.forEach((record: any) => {
            const existingRecord = userRecordMap.get(record.student_id);
            
            // 如果还没有记录，或者当前记录时间更新，则更新
            if (!existingRecord) {
                userRecordMap.set(record.student_id, record);
            } else {
                const existingTime = new Date(existingRecord.answer_time);
                const currentTime = new Date(record.answer_time);
                
                if (currentTime > existingTime) {
                    userRecordMap.set(record.student_id, record);
                }
            }
        });

        // 转换为数组
        const uniqueRecords = Array.from(userRecordMap.values());
        
        // 排序逻辑：已完成在前，然后按时间倒序
        uniqueRecords.sort((a, b) => {
            // 先按答题状态排序：已完成(1)在前，未完成(0)在后
            if (a.answer_status !== b.answer_status) {
                return b.answer_status - a.answer_status;
            }
            
            // 相同状态下按时间倒序（最新的在前）
            const timeA = new Date(a.answer_time).getTime();
            const timeB = new Date(b.answer_time).getTime();
            return timeB - timeA;
        });

        // 手动分页
        const startIndex = (page - 1) * pageSize;
        const endIndex = startIndex + pageSize;
        const paginatedRecords = uniqueRecords.slice(startIndex, endIndex);

        return {
            records: paginatedRecords,
            total: uniqueRecords.length
        };
        
    } catch (error) {
        console.error('获取最新学生记录失败:', error);
        return { records: [], total: 0 };
    }
}


/**
 * 同步uac_user表中的活跃用户到answer_record表
 * 为每个active_flag=1且user_type第4位为'1'（学生身份）的用户创建一条未完成的答题记录（如果不存在）
 */
 async function syncActiveUsersToAnswerRecord() {
    try {
        // 1. 获取uac_user表中所有active_flag=1的用户
        const activeUsers = await selectDataListByParam(
            TABLENAME.统一用户表,
            { active_flag: 1 },
            ["user_id", "user_name", "user_type"]
        );

        if (!activeUsers.data || activeUsers.data.length === 0) {
            return;
        }

        // 2. 筛选出学生身份的用户（user_type第4位为'1'）
        const studentUsers = activeUsers.data.filter((user: any) => {
            const userType = user.user_type || '';
            // 确保user_type至少有4位，且第4位为'1'
            return userType.length >= 4 && userType.charAt(3) === '1';
        });

        if (studentUsers.length === 0) {
            return;
        }

        // 3. 批量检查这些学生用户是否已有答题记录
        const studentIds = studentUsers.map((user: any) => user.user_id);
        
        const existingRecords = await selectDataListByParam(
            TABLENAME.答题记录表,
            { student_id: studentIds },
            ["student_id"]
        );

        // 创建已存在记录的学生ID集合
        const existingStudentIds = new Set();
        if (existingRecords.data && existingRecords.data.length > 0) {
            existingRecords.data.forEach((record: any) => {
                existingStudentIds.add(record.student_id);
            });
        }

        // 4. 为没有答题记录的学生用户创建未完成记录
        const recordsToCreate = [];
        
        for (const user of studentUsers) {
            if (!existingStudentIds.has(user.user_id)) {
                recordsToCreate.push({
                    record_id: randomId(TABLEID.答题记录表),
                    student_id: user.user_id,
                    student_name: user.user_name || user.user_id,
                    total_score: 0,
                    answer_time: null,
                    answer_status: STATE.否
                });
            }
        }

        // 5. 批量创建记录
        if (recordsToCreate.length > 0) {
            for (const record of recordsToCreate) {
                await addData(TABLENAME.答题记录表, record);
            }
            console.log(`成功为 ${recordsToCreate.length} 个活跃学生用户创建未完成答题记录`);
        }
        
    } catch (error) {
        console.error('同步活跃学生用户到答题记录表时出错:', error);
    }
}


/**
 * 单个答题记录数据下载
 * @param record_id 
 * @returns 
 */
export async function getStudentDetailed(record_id: string) {
    // 获取答题明细及对应的题目和维度信息
    const answerDetails = await selectDataToTableAssociation(
        TABLENAME.答题记录明细表,
        {
            [TABLENAME.题目表]: {
                column: ["question_id", "dimension_id"],
                where: {}
            }
        },
        { record_id },
        ["detail_id", "score", "question_id"]
    );
    
    let output = [["学号", "学生姓名", "家国情怀", "国际视野", "责任担当", "公得分", "学业扎实", "勇于创新", "善于学习", "智得分", "健康生活", "审美情趣", "劳动意识", "能得分", "总分", "答题时间"]];
    if (answerDetails.data || answerDetails.data.length != 0) {
        // 收集所有维度ID，用于批量查询维度信息
        const dimensionIds = new Set<string>();
        answerDetails.data.forEach((detail: any) => {
            const dimensionId = detail.question?.dimension_id;
            if (dimensionId) {
                dimensionIds.add(dimensionId);
            }
        });

        // 批量查询维度信息
        const dimensions = await selectDataListByParam(
            TABLENAME.维度表,
            { dimension_id: Array.from(dimensionIds) },
            ["dimension_id", "dimension_name"]
        );

        // 创建维度ID到维度名称的映射
        const dimensionNameMap = new Map();
        dimensions.data.forEach((dimension: any) => {
            dimensionNameMap.set(dimension.dimension_id, dimension.dimension_name);
        });

        // 获取答题记录基本信息
        const recordInfo = await selectOneDataByParam(
            TABLENAME.答题记录表,
            { record_id },
            ["student_id", "student_name", "total_score", "answer_time"]
        );

        // 初始化各维度得分
        const dimensionScores: { [key: string]: number } = {
            '家国情怀': 0,
            '国际视野': 0,
            '责任担当': 0,
            '学业扎实': 0,
            '勇于创新': 0,
            '善于学习': 0,
            '健康生活': 0,
            '审美情趣': 0,
            '劳动意识': 0
        };

        // 计算各维度得分
        answerDetails.data.forEach((detail: any) => {
            const dimensionId = detail.question?.dimension_id;
            const dimensionName = dimensionNameMap.get(dimensionId);
            const score = parseInt(detail.score || 0);
            
            if (dimensionName && dimensionScores.hasOwnProperty(dimensionName)) {
                dimensionScores[dimensionName] += score;
            }
        });

        // 计算三大方向得分
        const publicScore = dimensionScores['家国情怀'] + dimensionScores['国际视野'] + dimensionScores['责任担当'];
        const intelligenceScore = dimensionScores['学业扎实'] + dimensionScores['勇于创新'] + dimensionScores['善于学习'];
        const abilityScore = dimensionScores['健康生活'] + dimensionScores['审美情趣'] + dimensionScores['劳动意识'];

        
        output.push([
            recordInfo.data?.student_id, 
            recordInfo.data?.student_name, 
            dimensionScores['家国情怀'],
            dimensionScores['国际视野'],
            dimensionScores['责任担当'],
            publicScore,
            dimensionScores['学业扎实'],
            dimensionScores['勇于创新'],
            dimensionScores['善于学习'],
            intelligenceScore,
            dimensionScores['健康生活'],
            dimensionScores['审美情趣'],
            dimensionScores['劳动意识'],
            abilityScore,
            recordInfo.data?.total_score,
            recordInfo.data?.answer_time || "暂未提交答题"
        ]);
    }

    return output;
}


/**
 * 批量答题记录数据下载
 * @param record_ids 答题记录ID数组
 * @returns 二维数组，包含所有记录的详细数据
 */
export async function getBatchStudentDetailed(record_ids?: string[] | null | undefined) {
    let normalizedRecordIds: string[] = [];
    let allRecords: any[] = [];

    // 1. 处理参数：如果没有传入record_ids，则使用与getAllStudentAnswerRecordsOptimized相同的逻辑
    if (!record_ids) {
        // 使用与getAllStudentAnswerRecordsOptimized相同的逻辑获取所有记录（无分页）
        const result = await getAllStudentAnswerRecordsOptimizedWithoutPagination();
        allRecords = result.list || [];
        
        if (allRecords.length === 0) {
            return [[
                "学号", "学生姓名", 
                "家国情怀", "国际视野", "责任担当", "公得分",
                "学业扎实", "勇于创新", "善于学习", "智得分",
                "健康生活", "审美情趣", "劳动意识", "能得分",
                "总分", "答题时间", "答题状态"
            ]];
        }
        
        normalizedRecordIds = allRecords.map((record: any) => record.record_id);
        
    } else {
        // 有特定record_ids时，按传入的顺序获取
        if (Array.isArray(record_ids)) {
            normalizedRecordIds = record_ids.filter(id => id && typeof id === 'string');
        } else if (typeof record_ids === 'string') {
            normalizedRecordIds = [record_ids];
        } else {
            normalizedRecordIds = [String(record_ids)];
        }
        
        if (normalizedRecordIds.length > 0) {
            // 获取指定记录的信息
            const recordsInfo = await selectDataWithCustomOrder(
                TABLENAME.答题记录表,
                { record_id: normalizedRecordIds },
                ["record_id", "student_id", "student_name", "total_score", "answer_time", "answer_status"],
                [
                    ["answer_time", "DESC"]
                ]
            );
            
            allRecords = recordsInfo.data || [];
        }
    }

    // 2. 构建输出数据
    const output = [[
        "学号", "学生姓名", 
        "家国情怀", "国际视野", "责任担当", "公得分",
        "学业扎实", "勇于创新", "善于学习", "智得分",
        "健康生活", "审美情趣", "劳动意识", "能得分",
        "总分", "答题时间", "答题状态"
    ]];

    if (allRecords.length === 0) {
        return output;
    }

    // 3. 批量获取所有答题明细
    const allAnswerDetails = await selectDataListByParam(
        TABLENAME.答题记录明细表,
        { record_id: normalizedRecordIds },
        ["record_id", "score", "question_id"]
    );

    // 4. 初始化各维度默认得分（0分）
    const defaultDimensionScores = {
        '家国情怀': 0,
        '国际视野': 0,
        '责任担当': 0,
        '学业扎实': 0,
        '勇于创新': 0,
        '善于学习': 0,
        '健康生活': 0,
        '审美情趣': 0,
        '劳动意识': 0
    };

    // 5. 如果有答题明细，创建题目ID到维度名称的映射
    const questionDimensionMap = new Map();
    if (allAnswerDetails.data && allAnswerDetails.data.length > 0) {
        const questionIds = [...new Set(allAnswerDetails.data.map((detail: any) => detail.question_id))];
        
        const includeQuestionDimension: any = {};
        includeQuestionDimension[TABLENAME.维度表] = {
            column: ["dimension_id", "dimension_name"],
            where: {}
        };

        const questionsWithDimensions = await selectDataToTableAssociation(
            TABLENAME.题目表,
            includeQuestionDimension,
            { question_id: questionIds },
            ["question_id", "dimension_id"]
        );

        questionsWithDimensions.data.forEach((item: any) => {
            const dimensionName = item.dimension?.dimension_name;
            if (dimensionName) {
                questionDimensionMap.set(item.question_id, dimensionName);
            }
        });
    }

    // 6. 按记录ID分组计算各维度得分
    const recordScoresMap = new Map();

    // 初始化所有记录的得分都为默认值（0分）
    normalizedRecordIds.forEach(recordId => {
        recordScoresMap.set(recordId, {...defaultDimensionScores});
    });

    // 如果有答题明细，更新实际得分
    if (allAnswerDetails.data && allAnswerDetails.data.length > 0) {
        allAnswerDetails.data.forEach((detail: any) => {
            const recordId = detail.record_id;
            const dimensionName = questionDimensionMap.get(detail.question_id);
            const score = parseInt(detail.score || 0);

            if (!recordScoresMap.has(recordId)) {
                recordScoresMap.set(recordId, {...defaultDimensionScores});
            }

            const scores = recordScoresMap.get(recordId);
            if (dimensionName && scores && scores.hasOwnProperty(dimensionName)) {
                scores[dimensionName] += score;
            }
        });
    }

    // 7. 按排序后的顺序输出数据
    for (const record of allRecords) {
        const dimensionScores = recordScoresMap.get(record.record_id) || {...defaultDimensionScores};

        // 计算三大方向得分
        const publicScore = (dimensionScores['家国情怀'] || 0) + 
                        (dimensionScores['国际视野'] || 0) + 
                        (dimensionScores['责任担当'] || 0);
        const intelligenceScore = (dimensionScores['学业扎实'] || 0) + 
                                (dimensionScores['勇于创新'] || 0) + 
                                (dimensionScores['善于学习'] || 0);
        const abilityScore = (dimensionScores['健康生活'] || 0) + 
                        (dimensionScores['审美情趣'] || 0) + 
                        (dimensionScores['劳动意识'] || 0);

        output.push([
            record.student_id || '', 
            record.student_name || '',
            dimensionScores['家国情怀'] || 0,
            dimensionScores['国际视野'] || 0,
            dimensionScores['责任担当'] || 0,
            publicScore,
            dimensionScores['学业扎实'] || 0,
            dimensionScores['勇于创新'] || 0,
            dimensionScores['善于学习'] || 0,
            intelligenceScore,
            dimensionScores['健康生活'] || 0,
            dimensionScores['审美情趣'] || 0,
            dimensionScores['劳动意识'] || 0,
            abilityScore,
            record.total_score || 0,
            record.answer_time || '暂未答题',
            record.answer_status === 1 ? '已完成' : '未完成'
        ]);
    }

    return output;
}


// 辅助函数：getAllStudentAnswerRecordsOptimized的无分页版本
async function getAllStudentAnswerRecordsOptimizedWithoutPagination() {
    try {
        // 1. 先同步活跃用户
        await syncActiveUsersToAnswerRecord();

        // 2. 获取活跃用户ID
        const activeUsers = await selectDataListByParam(
            TABLENAME.统一用户表,
            { active_flag: 1 },
            ["user_id"]
        );

        const activeUserIds = activeUsers.data?.map((user: any) => user.user_id) || [];

        if (activeUserIds.length === 0) {
            return {
                list: [],
                total: 0
            };
        }

        // 3. 获取每个学生的答题记录（优先取已答题记录，相同学生多条已答题记录时取最早的一条）
        const { records: studentRecords, total } = await getStudentRecordsWithPriority(activeUserIds, 1, 100000); // 使用大pageSize获取所有记录

        if (!studentRecords || studentRecords.length === 0) {
            return {
                list: [],
                total: 0
            };
        }

        // 4. 分离已完成和未完成的记录
        const completedRecords = studentRecords.filter(record => record.answer_status === 1);
        const recordIds = completedRecords.map((record: any) => record.record_id);

        // 5. 只为已完成记录获取答题明细和维度信息
        let recordScoreMap = new Map();
        const dimensionKeys = [
            '家国情怀', '国际视野', '责任担当',
            '学业扎实', '勇于创新', '善于学习',
            '健康生活', '审美情趣', '劳动意识'
        ];

        if (recordIds.length > 0) {
            // 获取答题明细
            const allAnswerDetails = await selectDataListByParam(
                TABLENAME.答题记录明细表,
                { record_id: recordIds },
                ["record_id", "score", "question_id"]
            );

            if (allAnswerDetails.data && allAnswerDetails.data.length > 0) {
                // 获取题目和维度信息
                const questionIds = [...new Set(allAnswerDetails.data.map((detail: any) => detail.question_id))];
                
                const includeQuestionDimension: any = {};
                includeQuestionDimension[TABLENAME.维度表] = {
                    column: ["dimension_id", "dimension_name"],
                    where: {}
                };

                const questionsWithDimensions = await selectDataToTableAssociation(
                    TABLENAME.题目表,
                    includeQuestionDimension,
                    { question_id: questionIds },
                    ["question_id", "dimension_id"]
                );

                // 创建题目维度映射
                const questionDimensionMap = new Map();
                questionsWithDimensions.data?.forEach((item: any) => {
                    const dimensionName = item.dimension?.dimension_name;
                    if (dimensionName) {
                        questionDimensionMap.set(item.question_id, dimensionName);
                    }
                });

                // 初始化已完成记录的维度分数
                completedRecords.forEach((record: any) => {
                    const initialScores: { [key: string]: number } = {};
                    dimensionKeys.forEach(key => {
                        initialScores[key] = 0;
                    });
                    recordScoreMap.set(record.record_id, initialScores);
                });

                // 计算各维度分数
                allAnswerDetails.data.forEach((detail: any) => {
                    const dimensionName = questionDimensionMap.get(detail.question_id);
                    const score = parseInt(detail.score || 0);
                    
                    if (dimensionName && recordScoreMap.has(detail.record_id)) {
                        const scores = recordScoreMap.get(detail.record_id);
                        if (scores && scores.hasOwnProperty(dimensionName)) {
                            scores[dimensionName] += score;
                        }
                    }
                });
            }
        }

        // 6. 构建最终结果（与getAllStudentAnswerRecordsOptimized格式一致）
        const list = studentRecords.map((record: any) => {
            if (record.answer_status === 0) {
                // 未完成记录
                return {
                    record_id: record.record_id,
                    student_id: record.student_id,
                    student_name: record.student_name,
                    total_score: 0,
                    answer_time: record.answer_time,
                    answer_status: record.answer_status,
                    // 维度得分字段（为了保持格式一致）
                    jgqh: 0,
                    gjsy: 0,
                    zrdd: 0,
                    gong: 0,
                    xyzs: 0,
                    yycx: 0,
                    syxx: 0,
                    zhi: 0,
                    jksh: 0,
                    smqq: 0,
                    ldys: 0,
                    neng: 0
                };
            } else {
                // 已完成记录
                const dimensionScores = recordScoreMap.get(record.record_id) || {};
                
                const publicScore = (dimensionScores['家国情怀'] || 0) + 
                                  (dimensionScores['国际视野'] || 0) + 
                                  (dimensionScores['责任担当'] || 0);
                                  
                const intelligenceScore = (dimensionScores['学业扎实'] || 0) + 
                                        (dimensionScores['勇于创新'] || 0) + 
                                        (dimensionScores['善于学习'] || 0);
                                        
                const abilityScore = (dimensionScores['健康生活'] || 0) + 
                                   (dimensionScores['审美情趣'] || 0) + 
                                   (dimensionScores['劳动意识'] || 0);

                return {
                    record_id: record.record_id,
                    student_id: record.student_id,
                    student_name: record.student_name,
                    total_score: record.total_score || 0,
                    answer_time: record.answer_time || "已提交",
                    answer_status: record.answer_status,
                    // 维度得分字段
                    jgqh: dimensionScores['家国情怀'] || 0,
                    gjsy: dimensionScores['国际视野'] || 0,
                    zrdd: dimensionScores['责任担当'] || 0,
                    gong: publicScore,
                    xyzs: dimensionScores['学业扎实'] || 0,
                    yycx: dimensionScores['勇于创新'] || 0,
                    syxx: dimensionScores['善于学习'] || 0,
                    zhi: intelligenceScore,
                    jksh: dimensionScores['健康生活'] || 0,
                    smqq: dimensionScores['审美情趣'] || 0,
                    ldys: dimensionScores['劳动意识'] || 0,
                    neng: abilityScore
                };
            }
        });

        return {
            list,
            total
        };

    } catch (error) {
        console.error('获取无分页答题记录失败:', error);
        return {
            list: [],
            total: 0
        };
    }
}


/**
 * 获取所有答题记录ID（用于批量下载）
 * @param filters 可选筛选条件
 * @returns 所有符合条件的答题记录ID数组
 */
export async function getAllRecordIds(filters: any = {}) {
    // 合并筛选条件，默认只获取已完成的记录
    const whereCondition = {
        answer_status: STATE.是,
        ...filters
    };

    const records = await selectDataListByParam(
        TABLENAME.答题记录表,
        whereCondition,
        ["record_id"]
    );

    if (!records.data || records.data.length === 0) {
        return [];
    }

    return records.data.map((record: any) => record.record_id);
}


/**
 * 分页批量下载答题记录数据
 * @param page 页码
 * @param pageSize 每页大小
 * @returns 分页的批量数据
 */
export async function getBatchStudentDetailedByPage(page: number = 1, pageSize: number = 100) {
    // 1. 获取分页的记录ID
    const includeConf: any = {};
    const pagedRecords = await selectDataToTableAssociationToPage(
        TABLENAME.答题记录表,
        includeConf,
        { answer_status: STATE.是 },
        ["record_id"],
        page,
        pageSize
    );

    if (!pagedRecords.data || pagedRecords.data.length === 0) {
        return {
            data: [],
            total: 0,
            page,
            pageSize,
            totalPages: 0
        };
    }

    const recordIds = pagedRecords.data.map((record: any) => record.record_id);

    // 2. 获取总记录数
    const totalRecords = await selectDataListByParam(
        TABLENAME.答题记录表,
        { answer_status: STATE.是 },
        ["record_id"]
    );
    const total = totalRecords.data?.length || 0;
    const totalPages = Math.ceil(total / pageSize);

    // 3. 批量获取详细数据
    const batchData = await getBatchStudentDetailed(recordIds);

    return {
        data: batchData,
        total,
        page,
        pageSize,
        totalPages
    };
}






