/**
 * 备份数据库
 */
import { Client } from 'ssh2';
import { exec } from 'child_process';
import { promisify } from 'util';
import fs from 'fs';
import path from 'path';
import moment from 'moment';

const execAsync = promisify(exec);

// 备份配置
interface BackupConfig {
  host: string;
  port: number;
  user: string;
  password: string;
  database: string;
  localBackupPath: string; // 本地临时备份路径
  remoteBackupPath: string; // 远程备份路径
  remoteHost: string; // 远程服务器IP
  remoteUser: string; // 远程服务器用户名
  remotePassword: string; // 远程服务器密码
  sshPort?: number; // SSH端口
  retentionDays: number;
  keepLocalBackup: boolean; // 新增：是否保留本地备份文件
}

const defaultConfig: BackupConfig = {
  host: process.env.DB_HOST || 'localhost',
  port: parseInt(process.env.DB_PORT || '3306'),
  user: process.env.DB_USER || 'root',
  password: process.env.DB_PASSWORD || '123456',
  database: process.env.DB_NAME || 'publicIntelligence',
  localBackupPath: process.env.LOCAL_BACKUP_PATH || './nm_backups',
  remoteBackupPath: process.env.REMOTE_BACKUP_PATH || '/mnt/nm_gzn/backups',
  remoteHost: process.env.REMOTE_HOST || '123.207.147.179',
  remoteUser: process.env.REMOTE_USER || 'root',
  remotePassword: process.env.REMOTE_PASSWORD || 'GNIWT20110919!@@@',
  sshPort: parseInt(process.env.SSH_PORT || '22'),
  retentionDays: parseInt(process.env.BACKUP_RETENTION_DAYS || '30'),
  keepLocalBackup: process.env.KEEP_LOCAL_BACKUP === 'true' || false, // 默认不保留; // 新增：是否保留本地备份文件
};

export class BackupService {
  private config: BackupConfig;
  private isBackingUp: boolean = false;

  constructor(config?: Partial<BackupConfig>) {
    this.config = { ...defaultConfig, ...config };
    this.ensureLocalDirectory();
  }

  private ensureLocalDirectory(): void {
    if (!fs.existsSync(this.config.localBackupPath)) {
      fs.mkdirSync(this.config.localBackupPath, { recursive: true });
    }
  }

  private generateBackupFileName(): string {
    const timestamp = moment().format('YYYYMMDD_HHmmss');
    return `${this.config.database}_backup_${timestamp}.sql`;
  }

  /**
   * 使用SSH2执行远程命令
   */
  private async executeRemoteCommand(command: string): Promise<string> {
    return new Promise((resolve, reject) => {
      const conn = new Client();

      conn.on('ready', () => {
        console.log('SSH连接已建立');
        conn.exec(command, (err, stream) => {
          if (err) {
            conn.end();
            return reject(err);
          }

          let stdout = '';
          let stderr = '';

          stream.on('close', (code: number) => {
            conn.end();
            if (code === 0) {
              resolve(stdout);
            } else {
              reject(new Error(`命令执行失败，退出码: ${code}, 错误: ${stderr}`));
            }
          }).on('data', (data: string) => {
            stdout += data;
          }).stderr.on('data', (data: string) => {
            stderr += data;
          });
        });
      }).on('error', (err) => {
        reject(err);
      }).connect({
        host: this.config.remoteHost,
        port: this.config.sshPort!,
        username: this.config.remoteUser,
        password: this.config.remotePassword
      });
    });
  }

  /**
   * 使用SCP上传文件到远程服务器
   */
  private async uploadToRemote(localFilePath: string): Promise<string> {
    const fileName = path.basename(localFilePath);
    const remoteFilePath = `${this.config.remoteBackupPath}/${fileName}`;

    return new Promise((resolve, reject) => {
      const conn = new Client();

      conn.on('ready', () => {
        conn.sftp((err, sftp) => {
          if (err) {
            conn.end();
            return reject(err);
          }

          // 确保远程目录存在
          this.executeRemoteCommand(`mkdir -p ${this.config.remoteBackupPath} && chmod 755 ${this.config.remoteBackupPath}`)
            .then(() => {
              const readStream = fs.createReadStream(localFilePath);
              const writeStream = sftp.createWriteStream(remoteFilePath);

              writeStream.on('close', () => {
                conn.end();
                console.log(`上传完成: ${remoteFilePath}`);
                resolve(remoteFilePath);
              });

              writeStream.on('error', (err) => {
                conn.end();
                reject(err);
              });

              readStream.pipe(writeStream);
            })
            .catch(reject);
        });
      }).on('error', (err) => {
        reject(err);
      }).connect({
        host: this.config.remoteHost,
        port: this.config.sshPort!,
        username: this.config.remoteUser,
        password: this.config.remotePassword
      });
    });
  }

  /**
   * 执行本地数据库备份
   */
  private async createLocalBackup(): Promise<string> {
    const fileName = this.generateBackupFileName();
    const localFilePath = path.join(this.config.localBackupPath, fileName);
  
    console.log(`备份文件路径: ${localFilePath}`);
    console.log(`本地备份目录是否存在: ${fs.existsSync(this.config.localBackupPath)}`);
  
    const mysqldumpPath = 'D:\\mysql\\mysql-8.0.29-winx64\\bin\\mysqldump.exe';
  
    // 添加 --column-statistics=0 参数
    const command = `"${mysqldumpPath}" --host=${this.config.host} --port=${this.config.port} --user=${this.config.user} --password=${this.config.password} --column-statistics=0 ${this.config.database} > "${localFilePath}"`;
  
    console.log(`执行命令: ${command}`);
    console.log(`开始本地备份: ${this.config.database}`);
  
    try {
      const { stdout, stderr } = await execAsync(command);
  
      console.log(`命令执行完成，stdout: ${stdout}`);
      console.log(`命令执行完成，stderr: ${stderr}`);
  
      // 检查文件是否真的创建了
      if (fs.existsSync(localFilePath)) {
        const stats = fs.statSync(localFilePath);
        console.log(`备份文件已创建，大小: ${stats.size} 字节`);
      } else {
        console.error(`备份文件未创建: ${localFilePath}`);
        // 检查目录内容
        const files = fs.readdirSync(this.config.localBackupPath);
        console.log(`目录内容: ${files.join(', ')}`);
      }
  
      if (stderr && !stderr.includes('Using a password on the command line interface can be insecure')) {
        console.warn('备份警告:', stderr);
      }
  
      console.log(`本地备份完成: ${localFilePath}`);
      return localFilePath;
    } catch (error) {
      console.error('备份命令执行失败:', error);
      throw error;
    }
  }

  /**
   * 清理远程服务器的旧备份文件
   */
  private async cleanRemoteOldBackups(): Promise<void> {
    const cleanCommand = `find ${this.config.remoteBackupPath} -name "${this.config.database}_backup_*.sql" -mtime +${this.config.retentionDays} -delete`;

    try {
      await this.executeRemoteCommand(cleanCommand);
      console.log('远程旧备份清理完成');
    } catch (error) {
      console.error('远程清理失败:', error);
    }
  }

  /**
   * 清理本地临时文件
   */
  private cleanLocalTempFiles(): void {
    try {
      const files = fs.readdirSync(this.config.localBackupPath);
      files.forEach(file => {
        if (file.endsWith('.sql')) {
          const filePath = path.join(this.config.localBackupPath, file);
          fs.unlinkSync(filePath);
        }
      });
    } catch (error) {
      console.error('清理本地临时文件失败:', error);
    }
  }

  /**
   * 完整的备份流程
   */
  public async backupDatabase(): Promise<{ localPath: string; remotePath: string }> {
    if (this.isBackingUp) {
      throw new Error('已有备份任务正在进行中');
    }
  
    this.isBackingUp = true;
  
    try {
      console.log('=== 开始数据库备份流程 ===');
  
      // 创建本地备份
      const localFilePath = await this.createLocalBackup();
  
      // 验证文件确实存在
      if (!fs.existsSync(localFilePath)) {
        throw new Error(`备份文件未创建: ${localFilePath}`);
      }
  
      const fileStats = fs.statSync(localFilePath);
      if (fileStats.size === 0) {
        console.warn('警告: 备份文件大小为0字节');
      }
  
      console.log(`备份文件验证成功，大小: ${fileStats.size} 字节`);
  
      // 上传到远程服务器
      console.log('开始上传到远程服务器...');
      const remoteFilePath = await this.uploadToRemote(localFilePath);
  
      // 清理远程旧备份
      await this.cleanRemoteOldBackups();
  
      // 只在配置不保留本地备份时才清理
      if (!this.config.keepLocalBackup) {
        this.cleanLocalTempFiles();
      }
  
      console.log('=== 数据库备份完成 ===');
      return {
        localPath: localFilePath,
        remotePath: remoteFilePath
      };
  
    } catch (error) {
      console.error('数据库备份失败:', error);
      throw error;
    } finally {
      this.isBackingUp = false;
    }
  }

  /**
   * 测试SSH连接
   */
  public async testSSHConnection(): Promise<void> {
    try {
      console.log('测试SSH连接...');
      const result = await this.executeRemoteCommand('echo "SSH连接测试成功"');
      console.log('SSH连接测试成功:', result.trim());
    } catch (error) {
      console.error('SSH连接测试失败:', error);
      throw error;
    }
  }

  /**
   * 测试数据库连接
   */
  public async testDatabaseConnection(): Promise<void> {
    try {
      console.log('测试数据库连接...');
      const mysqlPath = 'D:\\mysql\\mysql-8.0.29-winx64\\bin\\mysql.exe'; // 使用绝对路径
      const testCommand = `"${mysqlPath}" --host=${this.config.host} --port=${this.config.port} --user=${this.config.user} --password=${this.config.password} -e "SELECT 1" ${this.config.database}`;
      await execAsync(testCommand);
      console.log('数据库连接测试成功');
    } catch (error) {
      console.error('数据库连接测试失败:', error);
      throw error;
    }
  }

  /**
   * 完整的备份测试
   */
  public async testBackup(): Promise<void> {
    console.log('开始备份功能测试...');
    
    try {
      await this.testSSHConnection();
      await this.testDatabaseConnection();
      
      const result = await this.backupDatabase();
      console.log('备份测试成功！');
      console.log('本地文件:', result.localPath);
      console.log('远程文件:', result.remotePath);
      
    } catch (error) {
      console.error('备份测试失败:', error);
      throw error;
    }
  }

  /**
   * 启动定时备份任务
   */
  public startScheduledBackup(hour: number = 2, minute: number = 0): void {
    console.log(`启动定时备份任务，每天 ${hour}:${minute.toString().padStart(2, '0')} 执行，备份到服务器: ${this.config.remoteHost}`);

    const scheduleBackup = async () => {
      const now = new Date();
      const targetTime = new Date();
      targetTime.setHours(hour, minute, 0, 0);

      // 如果今天的目标时间已经过去，就设置到明天
      if (now.getTime() > targetTime.getTime()) {
        targetTime.setDate(targetTime.getDate() + 1);
        console.log(`今天 ${hour}:${minute.toString().padStart(2, '0')} 已过，将在明天同一时间执行`);
      }

      const timeUntilBackup = targetTime.getTime() - now.getTime();
      
      console.log(`下次备份将在 ${Math.round(timeUntilBackup / 1000 / 60)} 分钟后执行`);

      setTimeout(async () => {
        try {
          console.log('开始执行定时备份...');
          await this.backupDatabase();
          console.log('定时备份完成');
        } catch (error) {
          console.error('定时备份失败:', error);
        }

        // 设置下一次备份（24小时后）
        setTimeout(scheduleBackup, 24 * 60 * 60 * 1000);
      }, timeUntilBackup);
    };

    // 立即检查并启动第一次调度
    scheduleBackup();
  }

  /**
   * 立即执行一次备份
   */
  public async backupNow(): Promise<{ localPath: string; remotePath: string }> {
    console.log('立即执行备份...');
    return await this.backupDatabase();
  }
}

