import { Buffer } from "buffer";
import { AUTHENTICATIONTYPEENNUM, TYPEENUM } from "../config/enum";
import { ERRORENUM } from "../config/errorEnum";
import { checkStrLeng, checkType } from "../tools/eccParam";
import { convertEncoding, fixDoubleEncoding, hasGarbledCharacters, parseUserInfoXml, parseUserTypeEnhanced } from "../tools/parseXml";
import { BizError } from "../util/bizError";
import * as iconv from 'iconv-lite';
import { getMySqlMs, getUserToken, randomId } from "../tools/systemTools";
import { selectOneDataByParam } from "../data/findData";
import { TABLEID, TABLENAME } from "../config/dbEnum";
import { addData } from "../data/addData";
import { updateManyData } from "../data/updateData";

var xmlrpc = require('xmlrpc');
const https = require('https');
const dns = require('dns');
const { parseString } = require('xml2js');

// XML-RPC认证信息
const XML_RPC_USER = 'K12RPC';
const XML_RPC_PASS = 'K12RPC!Pwd1901';
// 生成Base64认证头
const AUTH_BASE64 = Buffer.from(`${XML_RPC_USER}:${XML_RPC_PASS}`).toString('base64');


/**
 * 网络诊断函数
 */
async function networkDiagnosis(): Promise<void> {
    console.log('=== 开始网络诊断 ===');
    
    const targetHost = 'platform.nmzx.xhedu.sh.cn';
    const targetPort = 443;
    
    try {
        // 1. DNS解析测试
        console.log('1. DNS解析测试...');
        const addresses = await dns.promises.resolve4(targetHost);
        console.log('DNS解析结果:', addresses);
        
        // 2. 端口连通性测试
        console.log('2. 端口连通性测试...');
        await new Promise((resolve, reject) => {
            const socket = require('net').createConnection(targetPort, targetHost, () => {
                console.log('端口连通性: 成功');
                socket.end();
                resolve(true);
            });
            
            socket.setTimeout(5000);
            socket.on('timeout', () => {
                console.log('端口连通性: 超时');
                socket.destroy();
                reject(new Error('连接超时'));
            });
            
            socket.on('error', (error) => {
                console.log('端口连通性: 失败', error.message);
                reject(error);
            });
        });
        
        // 3. HTTPS请求测试
        console.log('3. HTTPS请求测试...');
        const testResult = await testHTTPSRequest();
        console.log('HTTPS测试结果:', testResult);
        
    } catch (error) {
        console.error('网络诊断失败:', error.message);
        throw error;
    }
}

/**
 * 简单的HTTPS请求测试
 */
async function testHTTPSRequest(): Promise<boolean> {
    return new Promise((resolve, reject) => {
        const options = {
            hostname: 'platform.nmzx.xhedu.sh.cn',
            port: 443,
            path: '/',
            method: 'GET',
            timeout: 5000,
            rejectUnauthorized: false,
            headers: {
                'Authorization': `Basic ${AUTH_BASE64}`
            }
        };

        const req = https.request(options, (res) => {
            console.log('HTTPS测试状态码:', res.statusCode);
            resolve(res.statusCode === 200 || res.statusCode === 404);
        });

        req.on('error', (error) => {
            console.error('HTTPS测试错误:', error.message);
            reject(error);
        });

        req.on('timeout', () => {
            console.error('HTTPS测试超时');
            req.destroy();
            reject(new Error('请求超时'));
        });

        req.end();
    });
}

/**
 * 创建全局客户端实例（单例模式）
 */
const createUACClient = () => {
    const clientConfig = {
        host: process.env.UAC_HOST || 'platform.nmzx.xhedu.sh.cn',
        port: parseInt(process.env.UAC_PORT || '443'),
        path: '/platform/rpc/index.php',
        basic_auth: {
            user: XML_RPC_USER,
            pass: XML_RPC_PASS
        },
        // encoding: 'GBK',
        headers: {
            'Content-Type': 'text/xml',
            'User-Agent': 'Node.js XML-RPC Client',
            'Authorization': `Basic ${AUTH_BASE64}`
        }
    };
    
    console.log('创建UAC客户端，使用认证用户:', XML_RPC_USER);
    
    return xmlrpc.createSecureClient(clientConfig);
};

let uacClient: any = null;

/**
 * 校验session
 * @param req 
 * @param res 
 */
export async function checkSession(req, res) {
    try {
        // 先进行网络诊断
        await networkDiagnosis();
        
        const sessid = req.query.sessid || 
                      req.headers['x-session-id'] || 
                      req.body.sessid;
        console.log(sessid);
        
        if (!checkType(sessid, TYPEENUM.string)) {
            throw new BizError(ERRORENUM.参数错误);
        }
        
        if (!checkStrLeng(sessid)) {
            throw new BizError(ERRORENUM.请求参数错误);
        }

        console.log('开始验证会话:', sessid.substring(0, 10) + '...');
        console.log('使用XML-RPC认证用户:', XML_RPC_USER);
        
        // 使用手动请求方法
        const userId = await k12CallWithManualRequest(sessid);
        
        if (!userId) {
            throw new BizError(ERRORENUM.您的登录已失效);
        }

        // 使用新的getUserInfoManual函数
        let userInfo = await getUserInfoManual(userId);

        // 添加空值检查
        if (!userInfo) {
            throw new BizError(ERRORENUM.获取用户信息失败);
        }

        let permission = 0;
        let userType:any = parseUserTypeEnhanced(userInfo.user_type);
        let type = AUTHENTICATIONTYPEENNUM.一般注册用户;
        if (userType.学生 == 1) {
            type = AUTHENTICATIONTYPEENNUM.学生;
            permission = 0;
        } else if (userType.教职员工 == 1) {
            permission = 1;
        }

        //单点登录成功，将用户写入系统用户表
        let timestamp = getMySqlMs();
        let token = getUserToken(userId + timestamp);
        let userDbData = await selectOneDataByParam(TABLENAME.管理后台用户, {loginId:userId}, ["loginId", "permission"]);
        if (!Object.keys(userDbData.data).length) {
            //系统用户表没有该账号，新建一个
            let addUInfo = {
                aId:randomId(TABLEID.管理后台用户),
                loginId: userId,
                permission,
                token,
                tokenMs:timestamp
            };
            await addData(TABLENAME.管理后台用户, addUInfo);
        } else {
            let updateUInfo = {
                token,
                tokenMs:timestamp
            }
            await updateManyData(TABLENAME.管理后台用户, {loginId: userId}, updateUInfo);
        }

        let permissionDbData:any = await selectOneDataByParam(TABLENAME.管理后台用户, {loginId:userId}, ["loginId", "permission"]);

        const resultInfo = {
            type,
            userId: userId,
            userInfo, // 添加用户信息
            timestamp,
            userName: userInfo.user_name,
            // adminFlag: userInfo.admin_flag.int, //账号标识：-1：公共帐号，0：一般成员，1：所属组组长，2：超级管理员
            userType, //“一般注册用户”、“行政管理人员”、“教职员工”、“学生”、“家长”
            token,
            tokenMs: timestamp,
            permission: permissionDbData.data.permission, //本系统权限-是否管理员（0-否，1-是）
        };

        res.success(resultInfo);
    } catch (error) {
        console.error('Session检查异常:', error);
        
        if (error.code === 'ENOTFOUND') {
            throw new BizError(ERRORENUM.网络连接失败);
        } else if (error.code === 'ECONNREFUSED') {
            throw new BizError(ERRORENUM.服务不可用);
        } else if (error.code === 'ETIMEDOUT') {
            throw new BizError(ERRORENUM.请求超时);
        } else {
            throw new BizError(ERRORENUM.系统繁忙请稍后重试);
        }
    }
}


/**
 * 获取登录账号的权限
 * 手动获取用户信息 调用user.getUserInfo方法，完全控制编码处理
 */
async function getUserInfoManual(userId: string): Promise<any> {
    return new Promise((resolve, reject) => {
        try {
            const xmlBody = `<?xml version="1.0" encoding="GBK"?>
    <methodCall>
        <methodName>user.getUserInfo</methodName>
        <params>
            <param>
                <value><string>user_id='${userId}'</string></value>
            </param>
            <param>
                <value><string></string></value>
            </param>
            <param>
                <value><int>1</int></value>
            </param>
            <param>
                <value><int>0</int></value>
            </param>
        </params>
    </methodCall>`;
    
            const options = {
                hostname: 'platform.nmzx.xhedu.sh.cn',
                port: 443,
                path: '/platform/rpc/index.php',
                method: 'POST',
                timeout: 10000,
                headers: {
                    'Content-Type': 'text/xml',
                    'Authorization': `Basic ${AUTH_BASE64}`,
                    'Content-Length': Buffer.byteLength(xmlBody, 'gbk'),
                    'Accept': 'text/xml',
                    'Connection': 'close'
                },
                rejectUnauthorized: false
            };
    
            const req = https.request(options, async (res) => {
                const chunks: Buffer[] = [];
                
                res.on('data', (chunk) => {
                    // 收集所有 chunk 到数组中
                    chunks.push(Buffer.isBuffer(chunk) ? chunk : Buffer.from(chunk));
                });

                res.on('end', async () => {
                    const uint8Arrays = chunks.map(chunk => {
                        if (chunk instanceof Uint8Array) {
                            return chunk;
                        } else {
                            // 使用类型断言确保 TypeScript 知道这是一个 Buffer
                            const bufferChunk = chunk as Buffer;
                            return new Uint8Array(
                                bufferChunk.buffer, 
                                bufferChunk.byteOffset, 
                                bufferChunk.byteLength
                            );
                        }
                    });

                    console.log('原始响应十六进制:', uint8Arrays.toString().substring(0, 100));
                    
                    // 计算总长度
                    const totalLength = uint8Arrays.reduce((total, chunk) => total + chunk.length, 0);
                    
                    // 创建合并后的 Uint8Array
                    const mergedUint8Array = new Uint8Array(totalLength);
                    let offset = 0;
                    
                    // 复制所有 chunk 到 mergedUint8Array
                    for (const chunk of uint8Arrays) {
                        mergedUint8Array.set(chunk, offset);
                        offset += chunk.length;
                    }
                    
                    // 将 Uint8Array 转换回 Buffer
                    const mergedBuffer = Buffer.from(mergedUint8Array);
                    
                    if (res.statusCode !== 200) {
                        console.error('获取用户信息失败，状态码:', res.statusCode);
                        return resolve(null);
                    }
                    
                    try {
                        // 直接尝试多种编码方式解析
                        let decodedResponse;
                        
                        // 尝试1: 直接作为GBK解码
                        try {
                            decodedResponse = iconv.decode(mergedBuffer, 'gbk');
                            console.log('使用GBK解码成功');
                        } catch (e) {
                            console.log('GBK解码失败，尝试UTF-8:', e.message);
                            // 尝试2: 作为UTF-8解码
                            decodedResponse = mergedBuffer.toString('utf8');
                        }
                        
                        // 检查是否有乱码字符
                        if (hasGarbledCharacters(decodedResponse)) {
                            console.log('检测到乱码，尝试修复编码');
                            // 尝试3: 修复可能的双重编码
                            decodedResponse = fixDoubleEncoding(decodedResponse);
                        }
                        
                        // 解析XML
                        const result = await parseUserInfoXml(decodedResponse);
                        resolve(result);
                    } catch (error) {
                        console.error('解析用户信息失败:', error);
                        // 尝试原始响应作为最后手段
                        try {
                            const result = await parseUserInfoXml(mergedBuffer.toString('binary'));
                            resolve(result);
                        } catch (e) {
                            console.error('所有解析尝试都失败');
                            resolve(null);
                        }
                    }
                });
            });

            req.on('error', (error) => {
                console.error('请求错误:', error);
                resolve(null);
            });

            req.on('timeout', () => {
                console.error('请求超时');
                req.destroy();
                resolve(null);
            });

            req.write(xmlBody, 'binary');
            req.end();
        } catch (error) {
            console.error('获取用户信息异常:', error);
            return null;
        }
    });
}


/**
 * 处理用户信息响应，修复乱码
 */
function processUserInfoResponse(response: any): any {
    if (Array.isArray(response) && response.length > 0) {
        const userInfo = response[0];
        
        // 递归修复对象中的所有字符串字段
        const fixEncodingRecursively = (obj: any): any => {
            if (typeof obj === 'string') {
                return fixGBKEncoding(obj);
            } else if (Array.isArray(obj)) {
                return obj.map(item => fixEncodingRecursively(item));
            } else if (typeof obj === 'object' && obj !== null) {
                const result: any = {};
                for (const key in obj) {
                    result[key] = fixEncodingRecursively(obj[key]);
                }
                return result;
            }
            return obj;
        };
        
        return fixEncodingRecursively(userInfo);
    }
    return response;
}


/**
 * 修复GBK编码的字符串
 */
function fixGBKEncoding(str: string): string {
    if (!str || typeof str !== 'string') return str || '';
    
    // 检查是否包含Unicode替换字符
    if (str.includes('�') || /[\uFFFD]/.test(str)) {
        try {
            // 假设原始字符串是GBK编码被错误解析为UTF-8
            // 先转换回Buffer，再重新解码
            const buffer = iconv.encode(str, 'utf8');
            return iconv.decode(buffer, 'gbk');
        } catch (error) {
            console.warn('GBK修复失败:', error, '原始字符串:', str);
            return str;
        }
    }
    
    return str;
}


/**
 * 改进的手动XML-RPC请求（正确解析响应）
 */
export async function k12CallWithManualRequest(sessid: string): Promise<string | false> {
    return new Promise((resolve, reject) => {
        try {
            const xmlBody = `<?xml version="1.0" encoding="GBK"?>
<methodCall>
    <methodName>user.isUserLogin</methodName>
    <params>
        <param>
            <value><string>${sessid}</string></value>
        </param>
    </params>
</methodCall>`;

            const options = {
                hostname: 'platform.nmzx.xhedu.sh.cn',
                port: 443,
                path: '/platform/rpc/index.php',
                method: 'POST',
                timeout: 10000,
                headers: {
                    'Content-Type': 'text/xml',
                    'Authorization': `Basic ${AUTH_BASE64}`,
                    'Content-Length': Buffer.byteLength(xmlBody, 'gbk'),
                    'Accept': 'text/xml',
                    'Connection': 'close'
                },
                rejectUnauthorized: false
            };

            console.log('发送请求到:', options.hostname + options.path);
            
            const req = https.request(options, (res) => {
                let responseData = '';
                
                console.log('响应状态码:', res.statusCode);
                console.log('响应头:', JSON.stringify(res.headers));

                res.setEncoding('binary');

                res.on('data', (chunk) => {
                    responseData += chunk;
                });

                res.on('end', async () => {
                    console.log('完整响应长度:', responseData.length);
                    console.log('响应预览:', responseData.substring(0, 200));
                    
                    if (res.statusCode === 404) {
                        console.error('服务端返回404 Not Found');
                        resolve(false);
                        return;
                    }
                    
                    if (res.statusCode === 401) {
                        console.error('服务端返回401 Unauthorized - 认证失败');
                        resolve(false);
                        return;
                    }
                
                    try {
                        // 先将响应数据从GBK转换为UTF-8
                        const utf8Response = convertGBKToString(Buffer.from(responseData, 'binary'));
                        
                        // 解析XML响应
                        const result = await parseXmlResponse(utf8Response);
                        if (result && typeof result === 'string') {
                            resolve(result); // 返回解析到的nmzx2401910
                        } else {
                            resolve(false);
                        }
                    } catch (parseError) {
                        console.error('XML解析失败:', parseError);
                        resolve(false);
                    }
                });
            });

            req.on('error', (error) => {
                console.error('请求错误:', error.message, error.code);
                resolve(false);
            });

            req.on('timeout', () => {
                console.error('请求超时');
                req.destroy();
                resolve(false);
            });

            req.write(xmlBody, 'binary');
            req.end();

        } catch (error) {
            console.error('请求构建错误:', error);
            resolve(false);
        }
    });
}


/**
 * 解析XML响应并提取值
 */
 async function parseXmlResponse(xmlData: string | Buffer): Promise<string | false> {
    return new Promise((resolve, reject) => {
        // 确保输入是字符串
        const xmlString = typeof xmlData === 'string' ? xmlData : convertEncoding(xmlData, 'gbk');
        
        parseString(xmlString, { 
            explicitArray: false, 
            trim: true,
            tagNameProcessors: [(name: string) => name]
        }, (err, result) => {
            if (err) {
                console.error('XML解析错误:', err);
                return resolve(false);
            }
            
            try {
                const response = result.methodResponse;
                if (!response || !response.params || !response.params.param) {
                    console.error('无效的XML响应结构');
                    return resolve(false);
                }
                
                const param = response.params.param;
                const value = param.value;
                
                if (value && value.string) {
                    const resultValue = value.string;
                    console.log('成功提取值:', resultValue);
                    return resolve(resultValue);
                } else {
                    console.error('未找到string值在响应中');
                    return resolve(false);
                }
            } catch (parseError) {
                console.error('响应解析错误:', parseError);
                resolve(false);
            }
        });
    });
}


/**
 * 将GBK编码的Buffer或字符串转换为UTF-8字符串
 */
function convertGBKToString(input: any): string {
    if (!input) return '';
    
    try {
        // 如果输入是Buffer，直接解码
        if (Buffer.isBuffer(input)) {
            return iconv.decode(input, 'gbk');
        }
        
        // 如果输入是字符串，检查是否需要解码
        if (typeof input === 'string') {
            // 检查是否包含乱码字符（通常是GBK编码被当作UTF-8读取的结果）
            if (/[^\u0000-\u007F\u4E00-\u9FFF]/.test(input) && /[\uFFFD]/.test(input)) {
                // 先将字符串转换为Buffer，然后从GBK解码
                const buffer = Buffer.from(input, 'binary');
                return iconv.decode(buffer, 'gbk');
            }
            return input;
        }
        
        // 其他类型转换为字符串
        return String(input);
    } catch (error) {
        console.warn('GBK解码失败:', error, '原始输入:', input);
        return typeof input === 'string' ? input : String(input);
    }
}


/**
 * 使用xmlrpc库的调用方法（也需要解析返回值）
 */
export function k12Call(sessid: string, timeout: number = 8000): Promise<string | false> {
    return new Promise((resolve, reject) => {
        if (!uacClient) {
            uacClient = createUACClient();
        }

        const timer = setTimeout(() => {
            reject(new Error('UAC服务响应超时'));
        }, timeout);

        console.log('调用UAC服务，会话ID:', sessid.substring(0, 10) + '...');

        uacClient.methodCall('user.isUserLogin', [sessid], (error: any, value: any) => {
            clearTimeout(timer);
            
            if (error) {
                console.error('UAC RPC调用失败:', error.message);
                return resolve(false);
            }
            
            console.log('UAC RPC调用成功，返回值:', value, '类型:', typeof value);
            
            // 如果xmlrpc库能正确解析，直接返回值
            if (typeof value === 'string' && value) {
                resolve(value);
            } else {
                console.log('返回值格式不符合预期，尝试手动解析');
                resolve(false);
            }
        });
    });
}

/**
 * 测试解析函数
 */
export async function testXmlParsing(): Promise<void> {
    const testXml = `<?xml version="1.0" encoding="GBK"?>
<methodResponse>
<params>
<param>
<value><string>nmzx2401910</string></value>
</param>
</params>
</methodResponse>`;

    console.log('=== 测试XML解析 ===');
    const result = await parseXmlResponse(testXml);
    console.log('解析结果:', result);
}

// 在文件末尾添加测试
(async () => {
    console.log('=== 测试XML解析功能 ===');
    await testXmlParsing();
})();



