import { TABLENAME } from '../config/dbEnum';
import { systemConfig } from '../config/serverConfig';
import { addData } from '../data/addData';
import { delData } from '../data/delData';
import { selectDataListByParam, selectOneDataByParam } from '../data/findData';
import { updateManyData } from '../data/updateData';

/**
 * UAC用户同步接口实现
 */

// 用户信息结构体类型定义
interface UACUserStruct {
    user_id: string;
    group_id: number;
    user_name: string;
    nickname?: string;
    gender?: string;
    email?: string;
    user_type: string;
    active_flag: number;
    admin_flag: number;
    create_date: string;
    extend_info?: string;
    person_id?: string;
    id_type?: number;
    id_card?: string;
    name?: string;
    verified_status?: number;
    phone?: string;
    birthday?: string;
    password?: string;
    forbidden_reason?: number;
    xhjw_user_id?: number;
    is_sx?: number;
    order_id?: number;
    student_user_id?: string;
    is_train?: number;
    zw_mark?: number;
    eduid?: string;
    uoid?: number;
    user_mark?: string;
}

// 组信息结构体类型定义
interface UACGroupStruct {
    group_id: number;
    group_name: string;
    parent_id: number;
    thread_id: number;
    group_flag: number;
    sx_org_code?: string;
    dec_grade_id?: number;
    dec_classroom_id?: number;
    group_code?: string;
    group_mark?: string;
    group_mflag?: number;
    order_id?: number;
}

/**
 * 新增用户接口 - 由UAC自动调用
 */
export async function userAddUser(userData: UACUserStruct): Promise<boolean> {
    try {
        console.log('🔵 UAC调用新增用户接口:', userData.user_id);
        console.log('📦 用户数据:', JSON.stringify(userData, null, 2));
        
        // 1. 检查用户是否已存在
        const existingUser = await getUserFromDatabase(userData.user_id);
        if (existingUser && Object.keys(existingUser).length > 0) { // 检查对象是否为空
            console.log('🟡 用户已存在，执行更新操作:', userData.user_id);
            return await userUpdateUserCommon(userData);
        }
        
        // 2. 插入新用户到数据库
        const success = await insertUserToDatabase(userData);
        
        if (success) {
            console.log('🟢 新增用户成功:', userData.user_id);
            return true;
        } else {
            console.error('🔴 新增用户失败:', userData.user_id);
            return false;
        }
    } catch (error) {
        console.error('🔴 新增用户接口异常:', error);
        return false;
    }
}

/**
 * 更新用户基本信息接口 - 由UAC自动调用
 */
export async function userUpdateUserCommon(userData: UACUserStruct): Promise<boolean> {
    try {
        console.log('🔵 UAC调用更新用户接口:', userData.user_id);
        console.log('📦 更新数据:', JSON.stringify(userData, null, 2));
        
        // 更新用户信息到数据库
        const success = await updateUserInDatabase(userData);
        
        if (success) {
            console.log('🟢 更新用户成功:', userData.user_id);
            return true;
        } else {
            console.error('🔴 更新用户失败:', userData.user_id);
            return false;
        }
    } catch (error) {
        console.error('🔴 更新用户接口异常:', error);
        return false;
    }
}


/**
 * 删除用户接口 - 由UAC自动调用
 */
export async function userDeleteUser(userIds: string): Promise<boolean> {
    try {
        console.log('🔵 UAC调用删除用户接口:', userIds);
        
        // 参数已经是格式：'user1','user2','user3'，可以直接用于SQL IN子句
        // 但我们的delData函数可能需要数组，所以还是解析为数组
        // 注意：文档说明参数已经用单引号括起来，所以这里需要去除单引号并分割
        const userIdList = userIds.split(',').map(id => id.replace(/'/g, '').trim());
        console.log('📋 要删除的用户ID列表:', userIdList);
        
        // 批量删除用户
        const success = await deleteUsersFromDatabase(userIdList);
        
        if (success) {
            console.log('🟢 删除用户成功:', userIds);
            return true;
        } else {
            console.error('🔴 删除用户失败:', userIds);
            return false;
        }
    } catch (error) {
        console.error('🔴 删除用户接口异常:', error);
        return false;
    }
}


/**
 * 同步用户基本信息接口 - 一次性复制UAC当前用户
 */
export async function userSyncUser(userData: UACUserStruct): Promise<boolean> {
    try {
        console.log('🔵 UAC调用同步用户接口:', userData.user_id);
        console.log('📦 同步数据:', JSON.stringify(userData, null, 2));
        
        // 检查用户是否已存在
        const existingUser = await getUserFromDatabase(userData.user_id);
        if (existingUser && Object.keys(existingUser).length > 0) {
            console.log('🟡 用户存在，执行更新操作');
            return await userUpdateUserCommon(userData);
        } else {
            console.log('🟡 用户不存在，执行新增操作');
            return await userAddUser(userData);
        }
    } catch (error) {
        console.error('🔴 同步用户接口异常:', error);
        return false;
    }
}

/**
 * 新增组接口 - 由UAC自动调用
 */
// export async function groupAddGroup(groupData: UACGroupStruct): Promise<boolean> {
//     try {
//         console.log('🔵 UAC调用新增组接口:', groupData.group_id);
//         console.log('📦 组数据:', JSON.stringify(groupData, null, 2));
        
//         // 插入新组到数据库
//         const success = await insertGroupToDatabase(groupData);
        
//         if (success) {
//             console.log('🟢 新增组成功:', groupData.group_id);
//             return true;
//         } else {
//             console.error('🔴 新增组失败:', groupData.group_id);
//             return false;
//         }
//     } catch (error) {
//         console.error('🔴 新增组接口异常:', error);
//         return false;
//     }
// }

/**
 * 更新组信息接口 - 由UAC自动调用
 */
// export async function groupUpdateGroup(groupData: UACGroupStruct): Promise<boolean> {
//     try {
//         console.log('🔵 UAC调用更新组接口:', groupData.group_id);
//         console.log('📦 更新数据:', JSON.stringify(groupData, null, 2));
        
//         // 更新组信息到数据库
//         const success = await updateGroupInDatabase(groupData);
        
//         if (success) {
//             console.log('🟢 更新组成功:', groupData.group_id);
//             return true;
//         } else {
//             console.error('🔴 更新组失败:', groupData.group_id);
//             return false;
//         }
//     } catch (error) {
//         console.error('🔴 更新组接口异常:', error);
//         return false;
//     }
// }

/**
 * 删除组接口 - 由UAC自动调用
 */
// export async function groupDeleteGroup(groupId: number): Promise<boolean> {
//     try {
//         console.log('🔵 UAC调用删除组接口:', groupId);
        
//         // 删除组
//         const success = await deleteGroupFromDatabase(groupId);
        
//         if (success) {
//             console.log('🟢 删除组成功:', groupId);
//             return true;
//         } else {
//             console.error('🔴 删除组失败:', groupId);
//             return false;
//         }
//     } catch (error) {
//         console.error('🔴 删除组接口异常:', error);
//         return false;
//     }
// }

/**
 * 同步组数据接口 - 一次性复制UAC当前组
 */
// export async function groupSyncGroup(groupData: UACGroupStruct): Promise<boolean> {
//     try {
//         console.log('🔵 UAC调用同步组接口:', groupData.group_id);
//         console.log('📦 同步数据:', JSON.stringify(groupData, null, 2));
        
//         // 检查组是否已存在
//         const existingGroup = await getGroupFromDatabase(groupData.group_id);
//             if (existingGroup && existingGroup.length > 0) { // 检查数组是否为空
//                 console.log('🟡 组存在，执行更新操作');
//                 return await groupUpdateGroup(groupData);
//             } else {
//                 console.log('🟡 组不存在，执行新增操作');
//                 return await groupAddGroup(groupData);
//             }
//     } catch (error) {
//         console.error('🔴 同步组接口异常:', error);
//         return false;
//     }
// }

// 数据库操作函数
async function getUserFromDatabase(userId: string): Promise<any> {
    try {
        console.log('🔍 查询用户是否存在:', userId);
        let result = await selectOneDataByParam(TABLENAME.统一用户表, {user_id: userId});
        console.log('📊 查询结果:', result.data && Object.keys(result.data).length > 0 ? '用户存在' : '用户不存在');
        return result.data; // 返回 data 字段而不是整个 result 对象
    } catch (error) {
        console.error('🔴 查询用户失败:', error);
        return null;
    }
}

async function insertUserToDatabase(userData: UACUserStruct): Promise<boolean> {
    try {
        console.log('💾 开始插入用户到数据库:', userData.user_id);
        
        let addInfo = {
            user_id: userData.user_id,
            group_id: userData.group_id,
            user_name: userData.user_name,
            nickname: userData.nickname,
            gender: userData.gender,
            email: userData.email,
            user_type: userData.user_type,
            active_flag: userData.active_flag,
            admin_flag: userData.admin_flag,
            create_date: userData.create_date,
            extend_info: userData.extend_info,
            person_id: userData.person_id,
            id_type: userData.id_type,
            id_card: userData.id_card,
            name: userData.name,
            verified_status: userData.verified_status,
            phone: userData.phone,
            birthday: userData.birthday,
            password: userData.password,
            forbidden_reason: userData.forbidden_reason,
            xhjw_user_id: userData.xhjw_user_id,
            is_sx: userData.is_sx,
            order_id: userData.order_id,
            student_user_id: userData.student_user_id,
            is_train: userData.is_train,
            zw_mark: userData.zw_mark,
            eduid: userData.eduid,
            uoid: userData.uoid,
            user_mark: userData.user_mark,
            // permission: 0
        }
        
        await addData(TABLENAME.统一用户表, addInfo);
        console.log('💾 插入用户成功:', userData.user_id);
        return true;
    } catch (error) {
        console.error('🔴 插入用户到数据库失败:', error);
        return false;
    }
}

async function updateUserInDatabase(userData: UACUserStruct): Promise<boolean> {
    try {
        console.log('💾 开始更新用户数据:', userData.user_id);
        
        let updateInfo = {
            group_id: userData.group_id,
            user_name: userData.user_name,
            nickname: userData.nickname,
            gender: userData.gender,
            email: userData.email,
            user_type: userData.user_type,
            active_flag: userData.active_flag,
            admin_flag: userData.admin_flag,
            extend_info: userData.extend_info,
            person_id: userData.person_id,
            id_type: userData.id_type,
            id_card: userData.id_card,
            name: userData.name,
            verified_status: userData.verified_status,
            phone: userData.phone,
            birthday: userData.birthday,
            password: userData.password,
            forbidden_reason: userData.forbidden_reason,
            xhjw_user_id: userData.xhjw_user_id,
            is_sx: userData.is_sx,
            order_id: userData.order_id,
            student_user_id: userData.student_user_id,
            is_train: userData.is_train,
            zw_mark: userData.zw_mark,
            eduid: userData.eduid,
            uoid: userData.uoid,
            user_mark: userData.user_mark
        };

        console.log('📋 更新字段:', Object.keys(updateInfo).filter(key => updateInfo[key] !== undefined));
        
        await updateManyData(TABLENAME.统一用户表, { user_id: userData.user_id }, updateInfo);
        console.log('💾 更新用户成功:', userData.user_id);
        return true;
    } catch (error) {
        console.error('🔴 更新用户到数据库失败:', error);
        return false;
    }
}


async function deleteUsersFromDatabase(userIds: string[]): Promise<boolean> {
    try {
        console.log('🗑️ 开始删除用户:', userIds);
        // 注意：这里假设delData函数可以接受数组形式的user_id
        await delData(TABLENAME.统一用户表, { user_id: userIds });
        console.log('🗑️ 删除用户成功:', userIds);
        return true;
    } catch (error) {
        console.error('🔴 从数据库删除用户失败:', error);
        return false;
    }
}


// async function getGroupFromDatabase(groupId: number): Promise<any> {
//     try {
//         console.log('🔍 查询组是否存在:', groupId);
//         let result: any = await selectDataListByParam(TABLENAME.统一用户表, { group_id: groupId });
//         console.log('📊 查询结果:', result && result.length > 0 ? `找到${result.length}个组` : '组不存在');
//         return result; // 这里返回的是数组，逻辑保持不变
//     } catch (error) {
//         console.error('🔴 查询组信息失败:', error);
//         return null;
//     }
// }

// async function insertGroupToDatabase(groupData: UACGroupStruct): Promise<boolean> {
//     try {
//         console.log('💾 开始插入组到数据库:', groupData.group_id);
//         await addData(TABLENAME.统一用户表, groupData);
//         console.log('💾 插入组成功:', groupData.group_id);
//         return true;
//     } catch (error) {
//         console.error('🔴 插入组到数据库失败:', error);
//         return false;
//     }
// }

// async function updateGroupInDatabase(groupData: UACGroupStruct): Promise<boolean> {
//     try {
//         console.log('💾 开始更新组数据:', groupData.group_id);
//         await updateManyData(TABLENAME.统一用户表, { group_id: groupData.group_id }, groupData);
//         console.log('💾 更新组成功:', groupData.group_id);
//         return true;
//     } catch (error) {
//         console.error('🔴 更新组到数据库失败:', error);
//         return false;
//     }
// }

// async function deleteGroupFromDatabase(groupId: number): Promise<boolean> {
//     try {
//         console.log('🗑️ 开始删除组:', groupId);
//         await delData(TABLENAME.统一用户表, { group_id: groupId });
//         console.log('🗑️ 删除组成功:', groupId);
//         return true;
//     } catch (error) {
//         console.error('🔴 从数据库删除组失败:', error);
//         return false;
//     }
// }


/**
 * XML-RPC服务器设置
 */
export async function setupXmlRpcServer() {
    if (!systemConfig.xmlRpcServer.enabled) {
        console.log('⚪ XML-RPC服务器未启用');
        return null;
    }

    const xmlrpc = require('xmlrpc');
    const config = systemConfig.xmlRpcServer;
    
    console.log('🚀 正在创建XML-RPC服务器，端口:', config.port);
    
    try {
        // 创建XML-RPC服务器 - 使用正确的配置
        const server = xmlrpc.createServer({
            host: config.host || '0.0.0.0',
            port: config.port,
            encoding: 'GBK'
        }, () => {
            console.log('🟢 XML-RPC服务器启动成功:', {
                host: config.host,
                port: config.port,
                encoding: 'GBK'
            });
        });

        // 添加方法处理
        server.on('NotFound', (method, params) => {
            console.log('❌ 方法未找到:', method);
        });

        // 修改方法注册 - 正确处理XML-RPC的参数顺序
        const methods = {
            'user.addUser': safeHandleUserAddUser,
            'user.updateUserCommon': safeHandleUserUpdateUserCommon,
            'user.deleteUser': safeHandleUserDeleteUser,
            'user.syncUser': safeHandleUserSyncUser,
            // 'group.addGroup': safeHandleGroupAddGroup,
            // 'group.updateGroup': safeHandleGroupUpdateGroup,
            // 'group.deleteGroup': safeHandleGroupDeleteGroup,
            // 'group.syncGroup': safeHandleGroupSyncGroup,
            'system.listMethods': safeHandleSystemListMethods
        };

        Object.entries(methods).forEach(([methodName, handler]) => {
            server.on(methodName, (error, params, callback) => {
                console.log('📨 收到XML-RPC调用:', methodName);
                
                // XML-RPC库传递的参数顺序是 (error, params, callback)
                // 我们需要调整为 (params, callback)
                if (typeof callback === 'function') {
                    // 如果有错误，直接返回错误
                    if (error) {
                        console.error('❌ XML-RPC调用错误:', error);
                        callback(error, 0);
                        return;
                    }
                    
                    // 调用处理函数，传递正确的参数顺序
                    handler(params, callback);
                } else {
                    // 如果参数顺序不对，尝试重新解析
                    console.log('⚠️  检测到参数顺序问题，尝试重新解析...');
                    
                    // 可能是 (params, callback) 顺序，尝试调整
                    const actualParams = error; // 第一个参数实际上是 params
                    const actualCallback = params; // 第二个参数实际上是 callback
                    
                    if (typeof actualCallback === 'function') {
                        handler(actualParams, actualCallback);
                    } else {
                        console.error('❌ 无法解析参数顺序');
                        if (typeof actualCallback === 'function') {
                            actualCallback(new Error('Invalid parameter order'), 0);
                        }
                    }
                }
            });
        });

        return server;
    } catch (error) {
        console.error('🔴 创建XML-RPC服务器失败:', error);
        return null;
    }
}


// 添加安全包装的处理函数
async function safeHandleUserSyncUser(params: any[], callback: Function) {
    try {
        console.log('🔵 安全处理 user.syncUser');
        
        // 添加参数验证
        if (!params || !Array.isArray(params) || params.length === 0) {
            console.error('❌ 参数错误: params 为空或不是数组');
            if (typeof callback === 'function') {
                callback(new Error('Invalid parameters: params is empty or not an array'), false);
            }
            return;
        }
        
        const userData = params[0] as UACUserStruct;
        console.log('📦 用户数据:', JSON.stringify(userData, null, 2));
        
        const result = await userSyncUser(userData);
        // console.log('📤 响应结果:', result);
        console.log('📤 响应结果:', result ? 1 : 0);
        
        if (typeof callback === 'function') {
            callback(null, result); // 改为直接返回布尔值
        }
    } catch (error) {
        console.error('🔴 user.syncUser处理错误:', error);
        if (typeof callback === 'function') {
            callback(error, false);
        }
    }
}


async function safeHandleUserAddUser(params: any[], callback: Function) {
    try {
        console.log('🔵 安全处理 user.addUser');
        
        // 添加参数验证
        if (!params || !Array.isArray(params) || params.length === 0) {
            console.error('❌ 参数错误: params 为空或不是数组');
            if (typeof callback === 'function') {
                callback(new Error('Invalid parameters: params is empty or not an array'), 0);
            }
            return;
        }
        
        const userData = params[0] as UACUserStruct;
        console.log('📦 用户数据:', JSON.stringify(userData, null, 2));
        
        const result = await userAddUser(userData);
        console.log('📤 响应结果:', result ? 1 : 0);
        
        if (typeof callback === 'function') {
            callback(null, result);
        }
    } catch (error) {
        console.error('🔴 user.addUser处理错误:', error);
        if (typeof callback === 'function') {
            callback(error, false);
        }
    }
}

async function safeHandleUserUpdateUserCommon(params: any[], callback: Function) {
    try {
        console.log('🔵 安全处理 user.updateUserCommon');
        
        // 添加参数验证
        if (!params || !Array.isArray(params) || params.length === 0) {
            console.error('❌ 参数错误: params 为空或不是数组');
            if (typeof callback === 'function') {
                callback(new Error('Invalid parameters: params is empty or not an array'), 0);
            }
            return;
        }
        
        const userData = params[0] as UACUserStruct;
        console.log('📦 更新数据:', JSON.stringify(userData, null, 2));
        
        const result = await userUpdateUserCommon(userData);
        console.log('📤 响应结果:', result ? 1 : 0);
        
        if (typeof callback === 'function') {
            callback(null, result);
        }
    } catch (error) {
        console.error('🔴 user.updateUserCommon处理错误:', error);
        if (typeof callback === 'function') {
            callback(error, false);
        }
    }
}

async function safeHandleUserDeleteUser(params: any[], callback: Function) {
    try {
        console.log('🔵 安全处理 user.deleteUser');
        
        // 添加参数验证
        if (!params || !Array.isArray(params) || params.length === 0) {
            console.error('❌ 参数错误: params 为空或不是数组');
            if (typeof callback === 'function') {
                callback(new Error('Invalid parameters: params is empty or not an array'), 0);
            }
            return;
        }
        
        const userIds = params[0] as string;
        console.log('📋 要删除的用户ID:', userIds);
        
        const result = await userDeleteUser(userIds);
        console.log('📤 响应结果:', result ? 1 : 0);
        
        if (typeof callback === 'function') {
            callback(null, result);
        }
    } catch (error) {
        console.error('🔴 user.deleteUser处理错误:', error);
        if (typeof callback === 'function') {
            callback(error, false);
        }
    }
}

// async function safeHandleGroupAddGroup(params: any[], callback: Function) {
//     try {
//         console.log('🔵 安全处理 group.addGroup');
        
//         // 添加参数验证
//         if (!params || !Array.isArray(params) || params.length === 0) {
//             console.error('❌ 参数错误: params 为空或不是数组');
//             if (typeof callback === 'function') {
//                 callback(new Error('Invalid parameters: params is empty or not an array'), 0);
//             }
//             return;
//         }
        
//         const groupData = params[0] as UACGroupStruct;
//         console.log('📦 组数据:', JSON.stringify(groupData, null, 2));
        
//         const result = await groupAddGroup(groupData);
//         console.log('📤 响应结果:', result ? 1 : 0);
        
//         if (typeof callback === 'function') {
//             callback(null, result);
//         }
//     } catch (error) {
//         console.error('🔴 group.addGroup处理错误:', error);
//         if (typeof callback === 'function') {
//             callback(error, 0);
//         }
//     }
// }

// async function safeHandleGroupUpdateGroup(params: any[], callback: Function) {
//     try {
//         console.log('🔵 安全处理 group.updateGroup');
        
//         // 添加参数验证
//         if (!params || !Array.isArray(params) || params.length === 0) {
//             console.error('❌ 参数错误: params 为空或不是数组');
//             if (typeof callback === 'function') {
//                 callback(new Error('Invalid parameters: params is empty or not an array'), 0);
//             }
//             return;
//         }
        
//         const groupData = params[0] as UACGroupStruct;
//         console.log('📦 更新数据:', JSON.stringify(groupData, null, 2));
        
//         const result = await groupUpdateGroup(groupData);
//         console.log('📤 响应结果:', result ? 1 : 0);
        
//         if (typeof callback === 'function') {
//             callback(null, result);
//         }
//     } catch (error) {
//         console.error('🔴 group.updateGroup处理错误:', error);
//         if (typeof callback === 'function') {
//             callback(error, 0);
//         }
//     }
// }

// async function safeHandleGroupDeleteGroup(params: any[], callback: Function) {
//     try {
//         console.log('🔵 安全处理 group.deleteGroup');
        
//         // 添加参数验证
//         if (!params || !Array.isArray(params) || params.length === 0) {
//             console.error('❌ 参数错误: params 为空或不是数组');
//             if (typeof callback === 'function') {
//                 callback(new Error('Invalid parameters: params is empty or not an array'), 0);
//             }
//             return;
//         }
        
//         const groupId = params[0] as number;
//         console.log('📋 要删除的组ID:', groupId);
        
//         const result = await groupDeleteGroup(groupId);
//         console.log('📤 响应结果:', result ? 1 : 0);
        
//         if (typeof callback === 'function') {
//             callback(null, result);
//         }
//     } catch (error) {
//         console.error('🔴 group.deleteGroup处理错误:', error);
//         if (typeof callback === 'function') {
//             callback(error, 0);
//         }
//     }
// }

// async function safeHandleGroupSyncGroup(params: any[], callback: Function) {
//     try {
//         console.log('🔵 安全处理 group.syncGroup');
        
//         // 添加参数验证
//         if (!params || !Array.isArray(params) || params.length === 0) {
//             console.error('❌ 参数错误: params 为空或不是数组');
//             if (typeof callback === 'function') {
//                 callback(new Error('Invalid parameters: params is empty or not an array'), 0);
//             }
//             return;
//         }
        
//         const groupData = params[0] as UACGroupStruct;
//         console.log('📦 同步数据:', JSON.stringify(groupData, null, 2));
        
//         const result = await groupSyncGroup(groupData);
//         console.log('📤 响应结果:', result ? 1 : 0);
        
//         if (typeof callback === 'function') {
//             callback(null, result);
//         }
//     } catch (error) {
//         console.error('🔴 group.syncGroup处理错误:', error);
//         if (typeof callback === 'function') {
//             callback(error, 0);
//         }
//     }
// }

async function safeHandleSystemListMethods(params: any[], callback: Function) {
    try {
        console.log('🔵 安全处理 system.listMethods');
        
        // 添加参数验证（system.listMethods 可能没有参数，所以这里只检查 callback）
        if (typeof callback !== 'function') {
            console.error('❌ 参数错误: callback 不是函数');
            return;
        }
        
        const methods = [
            'user.addUser', 'user.updateUserCommon', 'user.deleteUser', 'user.syncUser',
            'group.addGroup', 'group.updateGroup', 'group.deleteGroup', 'group.syncGroup'
        ];
        
        callback(null, methods);
    } catch (error) {
        console.error('🔴 system.listMethods处理错误:', error);
        if (typeof callback === 'function') {
            callback(error);
        }
    }
}


/**
 * 初始化UAC集成
 */
export async function initUACIntegration() {
    if (!systemConfig.xmlRpcServer.enabled) {
        console.log('⚪ UAC用户自动同步功能未启用');
        return null;
    }
    
    console.log('🚀 正在启动XML-RPC服务器，端口:', systemConfig.xmlRpcServer.port);
    console.log('📋 配置详情:', JSON.stringify(systemConfig.xmlRpcServer, null, 2));
    
    try {
        // 启动XML-RPC服务器
        const xmlRpcServer = await setupXmlRpcServer();
        
        if (xmlRpcServer) {
            console.log('🟢 XML-RPC服务器启动成功，端口:', systemConfig.xmlRpcServer.port);
            
            // 添加服务器事件监听
            xmlRpcServer.on('error', (err) => {
                console.error('🔴 XML-RPC服务器错误:', err);
            });
            
            return xmlRpcServer;
        } else {
            console.error('🔴 XML-RPC服务器启动失败');
            return null;
        }
    } catch (error) {
        console.error('🔴 XML-RPC服务器启动异常:', error);
        return null;
    }
}


